# Copyright 2026 Observational Health Data Sciences and Informatics
#
# This file is part of CohortGenerator
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

.loadJson <- function(definition, simplifyVector = FALSE, simplifyDataFrame = FALSE, ...) {
  if (is.character(definition)) {
    definition <- jsonlite::fromJSON(definition,
      simplifyVector = simplifyVector,
      simplifyDataFrame = simplifyDataFrame,
      ...
    )
  }

  if (!is.list(definition)) {
    stop("Cannot instanitate object invalid type ", class(definition))
  }
  definition
}

.toJSON <- function(obj) {
  jsonlite::toJSON(obj, pretty = TRUE)
}


# SubsetCohortWindow -------------
#' @title Time Window For Cohort Subset Operator
#' @export
#' @description
#' Representation of a time window to use when subsetting a target cohort with a subset cohort
SubsetCohortWindow <- R6::R6Class(
  classname = "SubsetCohortWindow",
  private = list(
    .startDay = as.integer(0),
    .endDay = as.integer(0),
    .targetAnchor = "cohortStart",
    .subsetAnchor = "cohortStart",
    .negate = FALSE
  ),
  public = list(
    #' @description List representation of object
    toList = function() {
      objRepr <- list()
      if (length(private$.startDay)) {
        objRepr$startDay <- safeUnbox(private$.startDay)
      }
      if (length(private$.endDay)) {
        objRepr$endDay <- safeUnbox(private$.endDay)
      }
      if (length(private$.targetAnchor)) {
        objRepr$targetAnchor <- safeUnbox(private$.targetAnchor)
      }

      if (length(private$.subsetAnchor)) {
        objRepr$subsetAnchor <- safeUnbox(private$.subsetAnchor)
      }
      if (length(private$.negate)) {
        objRepr$negate <- safeUnbox(private$.negate)
      }
      objRepr
    },
    #' To JSON
    #' @description json serialized representation of object
    toJSON = function() {
      .toJSON(self$toList())
    },

    #' Is Equal to
    #' @description Compare SubsetCohortWindow to another
    #' @param criteria SubsetCohortWindow instance
    isEqualTo = function(criteria) {
      checkmate::assertR6(criteria, "SubsetCohortWindow")
      return(all(
        self$startDay == criteria$startDay,
        self$endDay == criteria$endDay,
        self$targetAnchor == criteria$targetAnchor,
        self$subsetAnchor == criteria$subsetAnchor,
        self$negate == criteria$negate
      ))
    }
  ),
  active = list(
    #' @field startDay Integer
    startDay = function(startDay) {
      if (missing(startDay)) {
        return(private$.startDay)
      }
      checkmate::assertIntegerish(x = startDay)
      private$.startDay <- as.integer(startDay)
      return(self)
    },
    #' @field endDay Integer
    endDay = function(endDay) {
      if (missing(endDay)) {
        return(private$.endDay)
      }
      checkmate::assertIntegerish(x = endDay)
      private$.endDay <- as.integer(endDay)
      return(self)
    },
    #' @field targetAnchor Boolean
    targetAnchor = function(targetAnchor) {
      if (missing(targetAnchor)) {
        return(private$.targetAnchor)
      }
      checkmate::assertChoice(x = targetAnchor, choices = c("cohortStart", "cohortEnd"))
      private$.targetAnchor <- targetAnchor
      return(self)
    },
    #' @field subsetAnchor Boolean
    subsetAnchor = function(subsetAnchor) {
      if (missing(subsetAnchor)) {
        return(private$.subsetAnchor)
      }
      checkmate::assertChoice(x = subsetAnchor, choices = c("cohortStart", "cohortEnd"))
      private$.subsetAnchor <- subsetAnchor
      return(self)
    },
    #' @field negate Boolean
    negate = function(negate) {
      if (missing(negate)) {
        return(private$.negate)
      }
      checkmate::assertChoice(x = negate, choices = c(TRUE, FALSE))
      private$.negate <- negate
      return(self)
    }
  )
)

# createSubsetCohortWindow ------------------------------
#' @title Create a relative time window for cohort subset operations
#' @description
#' This function is used to create a relative time window for
#' cohort subset operations. The cohort window allows you to define an interval
#' of time relative to the target cohort's start/end date and the
#' subset cohort's start/end date.
#' @export
#' @param startDay  The start day for the time window
#' @param endDay The end day for the time window
#' @param targetAnchor To anchor using the target cohort's start date or end date.
#'                     The parameter is specified as 'cohortStart' or 'cohortEnd'.
#' @param subsetAnchor To anchor using the subset cohort's start date or end date.
#'                     The parameter is specified as 'cohortStart' or 'cohortEnd'.
#' @param negate  Allows for negating a window, a way to detect that a subset does not occur relative to a target
#' @returns a SubsetCohortWindow instance
createSubsetCohortWindow <- function(startDay, endDay, targetAnchor, subsetAnchor = NULL, negate = FALSE) {
  if (is.null(subsetAnchor)) {
    subsetAnchor <- "cohortStart"
  }

  window <- SubsetCohortWindow$new()
  window$startDay <- startDay
  window$endDay <- endDay
  window$targetAnchor <- targetAnchor
  window$subsetAnchor <- subsetAnchor
  window$negate <- negate
  window
}

# SubsetOperator ------------------------------
#' @title Abstract base class for subsets.
#' @export
#' @description
#' Abstract Base Class for subsets. Subsets should inherit from this and implement their own requirements.
#' @seealso CohortSubsetOperator
#' @seealso DemographicSubsetOperator
#' @seealso LimitSubsetOperator
#'
#' @field name  name of subset operation - should describe what the operation does e.g. "Males under the age of 18", "Exposed to Celecoxib"
#'
SubsetOperator <- R6::R6Class(
  classname = "SubsetOperator",
  private = list(
    queryBuilder = QueryBuilder,
    suffixStr = "S",
    .name = NULL,
    baseFields = c("name")
  ),
  public = list(
    #' @param definition json character or list - definition of subset operator
    #'
    #' @return instance of object
    initialize = function(definition = NULL) {
      if (!is.null(definition)) {
        definition <- .loadJson(definition)
        for (field in names(definition)) {
          if (field %in% self$publicFields()) {
            self[[field]] <- definition[[field]]
          }
        }
      }
      self
    },
    #' Class Name
    #' @description Class name of object
    classname = function() {
      class(self)[1]
    },

    #' Get auto generated name
    #' @description
    #' Not intended to be used - should be implemented in subclasses
    getAutoGeneratedName = function() {
      return(private$suffixStr)
    },

    #' Return query builder instance
    #' @param id - integer that should be unique in the sql (e.g. increment it by one for each subset operation in set)
    #' @description Return query builder instance
    getQueryBuilder = function(id) {
      private$queryBuilder$new(self, id)
    },
    #' Public Fields
    #' @description Publicly settable fields of object
    publicFields = function() {
      # Note that this will probably break if you subclass a subclass
      return(c(private$baseFields, names(get(self$classname())$active)))
    },

    #' Is Equal to
    #' @description Compare Subsets - are they identical or not?
    #' Checks all fields and settings
    #'
    #' @param  subsetOperatorB A subset to test equivalence to
    isEqualTo = function(subsetOperatorB) {
      checkmate::assertR6(subsetOperatorB, "SubsetOperator")
      if (!all(class(self) == class(subsetOperatorB))) {
        return(FALSE)
      }

      for (field in self$publicFields()) {
        # DemographicCriteriaSubsetOpertior has additional equality test
        if (!is.atomic(self[[field]])) {
          next
        }

        if (is.null(self[[field]]) & is.null(subsetOperatorB[[field]])) {
          next
        }

        if (is.null(self[[field]]) & !is.null(subsetOperatorB[[field]])) {
          return(FALSE)
        }

        if (!is.null(self[[field]]) & is.null(subsetOperatorB[[field]])) {
          return(FALSE)
        }

        if (self[[field]] != subsetOperatorB[[field]]) {
          return(FALSE)
        }
      }

      return(TRUE)
    },

    #' To list
    #' @description convert to List representation
    toList = function() {
      repr <- list(
        name = safeUnbox(self$name),
        subsetType = safeUnbox(self$classname())
      )
      return(repr)
    },

    #' To Json
    #' @description convert to json serialized representation
    #' @return list representation of object as json character
    toJSON = function() {
      .toJSON(self$toList())
    },

    #' Pretty print
    #'
    #' @param ...   further arguments passed to or from other methods.
    print = function(...) {
      cat(glue::glue("*** {self$classname()} ***"))
      cat("\n")
      if (!is.null(private$.name)) {
        cat(self$name)
        cat("\n")
      }
      cat(self$getAutoGeneratedName())
      cat("\n")
      cat("SQL: ")
      qb <- self$getQueryBuilder(1)
      print(qb)
    }
  ),
  active = list(
    name = function(name) {
      if (missing(name)) {
        if (!is.null(private$.name)) {
          return(private$.name)
        }

        return(self$getAutoGeneratedName())
      }
      checkmate::assertCharacter(name, null.ok = TRUE)
      private$.name <- name
      self
    }
  )
)


# CohortSubsetOperator ------------------------------
#' @title Cohort Subset Operator
#' @export
#' @description
#' A subset of type cohort - subset a population to only those contained within defined cohort
CohortSubsetOperator <- R6::R6Class(
  classname = "CohortSubsetOperator",
  inherit = SubsetOperator,
  private = list(
    suffixStr = "Coh",
    queryBuilder = CohortSubsetQb,
    .cohortIds = integer(0),
    .cohortCombinationOperator = "all",
    .negate = FALSE,
    .windows = list()
  ),
  public = list(
    #' @param definition json character or list - definition of subset operator
    #'
    #' @return instance of object
    initialize = function(definition = NULL) {
      # support backwards compatibility with old style of storing definitions
      if (!is.null(definition)) {
        oldFormat <- c("startWindow", "endWindow") %in% names(definition)
        if (any(oldFormat)) {
          definition$startWindow$subsetAnchor <- "cohortStart"
          definition$startWindow$subsetAnchor <- "cohortEnd"
          definition["windows"] <- list(definition$startWindow, definition$endWindow)
          definition$startWindow <- NULL
          definition$endWindow <- NULL
        }
      }
      super$initialize(definition)
    },
    #' to List
    #' @description List representation of object
    toList = function() {
      objRepr <- super$toList()
      objRepr$cohortIds <- private$.cohortIds
      objRepr$cohortCombinationOperator <- safeUnbox(private$.cohortCombinationOperator)
      objRepr$negate <- safeUnbox(private$.negate)
      objRepr$windows <- lapply(private$.windows, function(x) {
        x$toList()
      })

      objRepr
    },
    #' Get auto generated name
    #' @description name generated from subset operation properties
    #'
    #' @return character
    getAutoGeneratedName = function() {
      nameString <- ""
      if (self$negate) {
        nameString <- paste0(nameString, "not in ")
      } else {
        nameString <- paste0(nameString, "in ")
      }

      if (length(self$cohortIds) > 1) {
        nameString <- paste0(nameString, self$cohortCombinationOperator, " of ")
      }

      cohortIds <- sprintf("cohorts: (%s)", paste(self$cohortIds, collapse = ", "))
      nameString <- paste0(nameString, cohortIds)

      windowString <- lapply(self$windows, function(window) {
        paste(
          "subset",
          tolower(SqlRender::camelCaseToTitleCase(window$subsetAnchor)),
          "is within D:",
          window$startDay,
          "- D:",
          window$endDay,
          "of target",
          tolower(SqlRender::camelCaseToTitleCase(window$targetAnchor))
        )
      })

      nameString <- paste(
        nameString,
        "where",
        paste(windowString, collapse = " and ")
      )

      return(paste0(nameString))
    }
  ),
  active = list(
    #' @field cohortIds Integer ids of cohorts to subset to
    cohortIds = function(cohortIds) {
      if (missing(cohortIds)) {
        return(private$.cohortIds)
      }

      cohortIds <- as.integer(cohortIds)
      checkmate::assertIntegerish(cohortIds, min.len = 1)
      checkmate::assertFALSE(any(is.na(cohortIds)))
      private$.cohortIds <- cohortIds
      self
    },
    #' @field cohortCombinationOperator How to combine the cohorts
    cohortCombinationOperator = function(cohortCombinationOperator) {
      if (missing(cohortCombinationOperator)) {
        return(private$.cohortCombinationOperator)
      }

      checkmate::assertChoice(x = cohortCombinationOperator, choices = c("any", "all"))
      private$.cohortCombinationOperator <- cohortCombinationOperator
      self
    },
    #' @field negate Inverse the subset rule? TRUE will take the patients NOT in the subset
    negate = function(negate) {
      if (missing(negate)) {
        return(private$.negate)
      }

      checkmate::assertLogical(x = negate)
      private$.negate <- negate
      self
    },
    #' @field windows list of time windows to use when evaluating the subset
    #' cohort relative to the target cohort
    windows = function(windows) {
      if (missing(windows)) {
        return(private$.windows)
      }
      realWindows <- list()
      for (window in windows) {
        if (is.list(window)) {
          window <- do.call(createSubsetCohortWindow, window)
        }
        realWindows[[length(realWindows) + 1]] <- window
      }

      checkmate::assertList(x = realWindows, types = "SubsetCohortWindow")
      private$.windows <- realWindows
      self
    }
  )
)

# createCohortSubset ------------------------------
#' A definition of subset functions to be applied to a set of cohorts
#' @export
#' @family subsets
#' @param name  optional name of operator
#' @param cohortIds integer - set of cohort ids to subset to
#' @param cohortCombinationOperator "any" or "all" if using more than one cohort id allow a subject to be in any cohort
#'                                  or require that they are in all cohorts in specified windows
#'
#' @param startWindow               DEPRECATED: Use `windows` instead.
#' @param endWindow                 DEPRECATED: Use `windows` instead.
#' @param windows                   A list of time windows to use to evaluate subset cohorts in relation to the
#'                                  target  cohorts. The logic is to always apply these windows with logical AND conditions.
#'                                  See [@seealso [createSubsetCohortWindow()]] for more details on how to create
#'                                  these windows.
#' @param negate                    The opposite of this definition - include patients who do NOT meet the specified criteria
#' @returns a CohortSubsetOperator instance
createCohortSubsetOperator <- function(name = NULL, cohortIds, cohortCombinationOperator, negate, windows = list(), startWindow = NULL, endWindow = NULL) {
  subset <- CohortSubsetOperator$new()
  subset$name <- name
  subset$cohortIds <- cohortIds
  subset$cohortCombinationOperator <- cohortCombinationOperator
  subset$negate <- negate

  # Start and end windows must always have subset anchor values set to support backwards compatibility
  if (!is.null(startWindow) || !is.null(endWindow)) {
    warning("Arguments 'startWindow' and 'endWindow' is deprecated. Use 'windows' instead.")
  }

  if (!is.null(startWindow)) {
    startWindow$subsetAnchor <- "cohortStart"
    windows[[length(windows) + 1]] <- startWindow
  }

  if (!is.null(endWindow)) {
    endWindow$subsetAnchor <- "cohortEnd"
    windows[[length(windows) + 1]] <- endWindow
  }

  subset$windows <- windows
  subset
}

# DemographicSubsetOperator ------------------------------
#' @title Demographic Subset Operator
#' @description
#' Operators for subsetting a cohort by demographic criteria
#'
#' @export
DemographicSubsetOperator <- R6::R6Class(
  classname = "DemographicSubsetOperator",
  inherit = SubsetOperator,
  private = list(
    queryBuilder = DemographicSubsetQb,
    suffixStr = "Demo",
    .ageMin = 0,
    .ageMax = 99999,
    .gender = NULL,
    .race = NULL,
    .ethnicity = NULL
  ),
  public = list(
    #' @description List representation of object
    toList = function() {
      objRepr <- super$toList()
      if (length(private$.ageMin)) {
        objRepr$ageMin <- safeUnbox(private$.ageMin)
      }
      if (length(private$.ageMax)) {
        objRepr$ageMax <- safeUnbox(private$.ageMax)
      }
      if (!is.null(private$.gender)) {
        objRepr$gender <- private$.gender
      }
      if (!is.null(private$.race)) {
        objRepr$race <- private$.race
      }
      if (!is.null(private$.ethnicity)) {
        objRepr$ethnicity <- private$.ethnicity
      }

      objRepr
    },
    #' Map gender concepts to names
    #' @param mapping               optional list of mappings for concept id to nouns
    #' @returns char vector
    mapGenderConceptsToNames = function(mapping = list(
                                          "8507" = "males",
                                          "8532" = "females",
                                          "0" = "unknown gender"
                                        )) {
      conceptMap <- c()
      if (length(self$gender)) {
        for (x in self$gender) {
          mp <- ifelse(!is.null(mapping[[as.character(x)]]), mapping[[as.character(x)]], paste("gender concept:", x))
          if (is.null(mp)) {
            mp <- x
          }
          conceptMap <- c(conceptMap, mp)
        }
      }

      return(conceptMap)
    },

    #' Get auto generated name
    #' @description name generated from subset operation properties
    #'
    #' @return character
    getAutoGeneratedName = function() {
      nameString <- ""

      if (!is.null(self$gender)) {
        nameString <- paste0(nameString, "", paste(self$mapGenderConceptsToNames(), collapse = ", "))
      }

      if (length(self$ageMin) && self$ageMin > 0) {
        if (nameString != "") {
          nameString <- paste0(nameString, " ")
        }

        nameString <- paste0(nameString, "aged ", self$ageMin)

        if (length(self$ageMax) && self$ageMax < 99999) {
          nameString <- paste0(nameString, " - ")
        } else {
          nameString <- paste0(nameString, "+")
        }
      }

      if (length(self$ageMax) && self$ageMax < 99999) {
        if (length(self$ageMin) && self$ageMin == 0) {
          if (nameString != "") {
            nameString <- paste0(nameString, " ")
          }

          nameString <- paste0(nameString, "aged <=")
        }
        nameString <- paste0(nameString, self$ageMax)
      }

      if (!is.null(private$.race)) {
        if (nameString != "") {
          nameString <- paste0(nameString, ", ")
        }

        nameString <- paste0(nameString, "race: ", paste(self$race, collapse = ", "))
      }
      if (!is.null(private$.ethnicity)) {
        if (nameString != "") {
          nameString <- paste0(nameString, ", ")
        }

        nameString <- paste0(nameString, "ethnicity: ", paste(self$ethnicity, collapse = ", "))
      }

      return(nameString)
    },

    #' @description json serialized representation of object
    toJSON = function() {
      .toJSON(self$toList())
    },


    #' @description Compare Subset to another
    #' @param criteria DemographicSubsetOperator instance
    isEqualTo = function(criteria) {
      checkmate::assertR6(criteria, "DemographicSubsetOperator")
      return(all(
        self$ageMin == criteria$ageMin,
        self$ageMax == criteria$ageMax,
        self$getGender() == criteria$getGender(),
        self$getRace() == criteria$getRace(),
        self$getEthnicity() == criteria$getEthnicity()
      ))
    },


    #' @description Gender getter - used when constructing SQL to default
    #' NULL to an empty string
    getGender = function() {
      if (is.null(private$.gender)) {
        return("")
      } else {
        return(private$.gender)
      }
    },

    #' @description Race getter - used when constructing SQL to default
    #' NULL to an empty string
    getRace = function() {
      if (is.null(private$.race)) {
        return("")
      } else {
        return(private$.race)
      }
    },

    #' @description Ethnicity getter - used when constructing SQL to default
    #' NULL to an empty string
    getEthnicity = function() {
      if (is.null(private$.ethnicity)) {
        return("")
      } else {
        return(private$.ethnicity)
      }
    }
  ),
  active = list(
    #' @field    ageMin Int between 0 and 99999 - minimum age
    ageMin = function(ageMin) {
      if (missing(ageMin)) {
        return(private$.ageMin)
      }
      checkmate::assertInt(ageMin, lower = 0, upper = min(self$ageMax, 99999))
      private$.ageMin <- ageMin
      return(self)
    },
    #' @field  ageMax  Int between 0 and 99999 - maximum age
    ageMax = function(ageMax) {
      if (missing(ageMax)) {
        return(private$.ageMax)
      }
      checkmate::assertInt(ageMax, lower = max(0, self$ageMin), upper = 99999)
      private$.ageMax <- ageMax
      return(self)
    },
    #' @field gender vector of gender concept IDs
    gender = function(gender) {
      if (missing(gender)) {
        return(private$.gender)
      }
      checkmate::assertVector(gender, null.ok = TRUE)
      private$.gender <- gender
      return(self)
    },
    #' @field race character string denoting race
    race = function(race) {
      if (missing(race)) {
        return(private$.race)
      }
      checkmate::assertVector(race, null.ok = TRUE)
      private$.race <- race
      return(self)
    },
    #' @field ethnicity character string denoting ethnicity
    ethnicity = function(ethnicity) {
      if (missing(ethnicity)) {
        return(private$.ethnicity)
      }
      checkmate::assertVector(ethnicity, null.ok = TRUE)
      private$.ethnicity <- ethnicity
      return(self)
    }
  )
)

# createDemographicSubset ------------------------------
#' Create createDemographicSubset Subset operator
#' @export
#' @family subsets
#' @param name         Optional char name
#' @param ageMin       The minimum age
#' @param ageMax       The maximum age
#' @param gender       Gender demographics - concepts - 0, 8532, 8507, 0, "female", "male".
#'                     Any string that is not "male" or "female" (case insensitive) is converted to gender concept 0.
#'                     https://athena.ohdsi.org/search-terms/terms?standardConcept=Standard&domain=Gender&page=1&pageSize=15&query=
#'                     Specific concept ids not in this set can be used but are not explicitly validated
#' @param race         Race demographics - concept ID list
#' @param ethnicity    Ethnicity demographics - concept ID list
createDemographicSubsetOperator <- function(name = NULL, ageMin = 0, ageMax = 99999, gender = NULL, race = NULL, ethnicity = NULL) {
  mapGenderCodes <- function(x) {
    if (length(x) > 1) {
      retValue <- c()
      for (i in x) {
        retValue <- c(retValue, mapGenderCodes(i))
      }
      return(retValue)
    }
    if (is.character(x)) {
      x <- tolower(x)
      if (x == "male") {
        return(8507)
      } else if (x == "female") {
        return(8532)
      }
      return(0)
    }
    return(x)
  }

  if (!is.null(gender)) {
    gender <- mapGenderCodes(gender)
  }

  subset <- DemographicSubsetOperator$new()
  subset$name <- name
  subset$ageMin <- ageMin
  subset$ageMax <- ageMax
  subset$gender <- gender
  subset$race <- race
  subset$ethnicity <- ethnicity

  subset
}


# LimitSubsetOperator ------------------------------
#' @title Limit Subset Operator
#' @export
#' @description operator to apply limiting subset operations (e.g. washout periods, calendar ranges or earliest entries)
#'
LimitSubsetOperator <- R6::R6Class(
  classname = "LimitSubsetOperator",
  inherit = SubsetOperator,
  private = list(
    queryBuilder = LimitSubsetQb,
    suffixStr = "Limit to",
    .priorTime = 0,
    .followUpTime = 0,
    .minimumCohortDuration = 0,
    .maximumCohortDuration = NULL,
    .limitTo = character(0),
    .calendarStartDate = NULL,
    .calendarEndDate = NULL
  ),
  public = list(
    #' Get auto generated name
    #' @description name generated from subset operation properties
    #'
    #' @return character
    getAutoGeneratedName = function() {
      nameString <- ""

      if (self$limitTo != "all") {
        nameString <- paste0(nameString, tolower(SqlRender::camelCaseToTitleCase(self$limitTo)), " occurence")
      } else {
        nameString <- paste0(nameString, "occurs")
      }

      if (self$priorTime > 0) {
        nameString <- paste0(nameString, " with at least ", self$priorTime, " days prior observation")
      }

      if (self$followUpTime > 0) {
        if (self$priorTime > 0) {
          nameString <- paste(nameString, "and")
        } else {
          nameString <- paste(nameString, "with at least")
        }
        nameString <- paste(nameString, self$followUpTime, "days follow up observation")
      }

      if (!is.null(self$calendarStartDate)) {
        nameString <- paste(nameString, "after", self$calendarStartDate)
      }

      if (!is.null(self$calendarEndDate)) {
        if (!is.null(self$calendarStartDate)) {
          nameString <- paste(nameString, "and")
        }
        nameString <- paste(nameString, "before", self$calendarEndDate)
      }

      if (self$minimumCohortDuration > 0) {
        nameString <- paste(nameString, "lasting at least", self$minimumCohortDuration, "days")
      }

      if (!is.null(self$maximumCohortDuration)) {
        nameString <- paste(nameString, "lasting at most", self$maximumCohortDuration, "days")
      }

      return(nameString)
    },
    #' To List
    #' @description List representation of object
    toList = function() {
      objRef <- super$toList()
      objRef$priorTime <- safeUnbox(private$.priorTime)
      objRef$followUpTime <- safeUnbox(private$.followUpTime)
      objRef$minimumCohortDuration <- safeUnbox(private$.minimumCohortDuration)
      objRef$maximumCohortDuration <- safeUnbox(private$.maximumCohortDuration)
      objRef$limitTo <- safeUnbox(private$.limitTo)
      objRef$calendarStartDate <- safeUnbox(private$.calendarStartDate)
      objRef$calendarEndDate <- safeUnbox(private$.calendarEndDate)

      objRef
    }
  ),
  active = list(
    #' @field priorTime             minimum washout time in days
    priorTime = function(priorTime) {
      if (missing(priorTime)) {
        return(private$.priorTime)
      }
      checkmate::assertInt(priorTime, lower = 0, upper = 99999)
      private$.priorTime <- priorTime
      self
    },
    #' @field followUpTime            minimum required follow up time in days
    followUpTime = function(followUpTime) {
      if (missing(followUpTime)) {
        return(private$.followUpTime)
      }

      checkmate::assertInt(followUpTime, lower = 0, upper = 99999)
      private$.followUpTime <- followUpTime
      self
    },
    #' @field minimumCohortDuration            minimum cohort duration time in days
    minimumCohortDuration = function(duration) {
      if (missing(duration)) {
        return(private$.minimumCohortDuration)
      }

      checkmate::assertInt(duration, lower = 0, upper = 99999)
      private$.minimumCohortDuration <- duration
      self
    },
    #' @field maximumCohortDuration            maximum cohort duration time in days
    maximumCohortDuration = function(duration) {
      if (missing(duration)) {
        return(private$.maximumCohortDuration)
      }
      if (is.null(duration)) {
        private$.maximumCohortDuration <- NULL
        return(self)
      } else {
        checkmate::assertInt(duration, lower = 0, upper = 99999)
        private$.maximumCohortDuration <- duration
        self
      }
    },
    #' @field limitTo     character one of:
    #'                              "firstEver" - only first entry in patient history
    #'                              "earliestRemaining" - only first entry after washout set by priorTime
    #'                              "latestRemaining" -  the latest remaining after washout set by followUpTime
    #'                              "lastEver" - only last entry in patient history inside
    #'
    #'                          Note, when using firstEver and lastEver with follow up and washout, patients with events
    #'                          outside this will be censored.
    #'
    limitTo = function(limitTo) {
      if (missing(limitTo)) {
        return(private$.limitTo)
      }

      checkmate::assertCharacter(limitTo)

      # maintain support for old versions
      if (limitTo == "") {
        limitTo <- "all"
      }

      checkmate::assertChoice(limitTo, choices = c("all", "firstEver", "earliestRemaining", "latestRemaining", "lastEver"))
      private$.limitTo <- limitTo
      self
    },
    #' @field calendarStartDate            The calendar start date for limiting by date
    calendarStartDate = function(calendarStartDate) {
      if (missing(calendarStartDate)) {
        return(private$.calendarStartDate)
      }

      if (is.character(calendarStartDate)) {
        if (calendarStartDate == "") {
          calendarStartDate <- NULL
        } else {
          calendarStartDate <- lubridate::date(calendarStartDate)
        }
      }
      checkmate::assertDate(calendarStartDate, null.ok = TRUE)

      if (length(calendarStartDate) && is.na(calendarStartDate)) {
        stop("Must provide a valid date, not NA")
      }

      private$.calendarStartDate <- calendarStartDate
      self
    },
    #' @field calendarEndDate            The calendar end date for limiting by date
    calendarEndDate = function(calendarEndDate) {
      if (missing(calendarEndDate)) {
        return(private$.calendarEndDate)
      }

      if (is.character(calendarEndDate)) {
        if (calendarEndDate == "") {
          calendarEndDate <- NULL
        } else {
          calendarEndDate <- lubridate::date(calendarEndDate)
        }
      }
      checkmate::assertDate(calendarEndDate, null.ok = TRUE)

      if (length(calendarEndDate) && is.na(calendarEndDate)) {
        stop("Must provide a valid date, not NA")
      }

      private$.calendarEndDate <- calendarEndDate
      self
    }
  )
)

# createLimitSubset ------------------------------
#' Create Limit Subset Operator
#' @description
#' Subset cohorts using specified limit criteria
#' @export
#' @family subsets
#' @param name                  Name of operation
#' @param priorTime             Required prior observation window (specified as a positive integer)
#' @param followUpTime          Required post observation window (specified as a positive integer)
#' @param minimumCohortDuration Required cohort duration length (specified as a positive integer)
#' @param maximumCohortDuration Optional: maximum cohort duration length (specified as a positive integer), defaults to NULL
#' @param limitTo           character one of:
#'                              "firstEver" - only first entry in patient history
#'                              "earliestRemaining" - only first entry after washout set by priorTime
#'                              "latestRemaining" -  the latest remaining after washout set by followUpTime
#'                              "lastEver" - only last entry in patient history inside
#'
#'                          Note, when using firstEver and lastEver with follow up and washout, patients with events
#'                          outside this will be censored. The "firstEver" and "lastEver" are applied first.
#'                          The "earliestRemaining" and "latestRemaining" are applied after all other limit
#'                          criteria are applied (i.e. after applying prior/post time and calendar time).
#' @param calendarEndDate       Start date to allow period (e.g. 2015/1/1)
#' @param calendarStartDate     End date to allow periods (e.g. 2020/1/1/)
createLimitSubsetOperator <- function(name = NULL,
                                      priorTime = 0,
                                      followUpTime = 0,
                                      minimumCohortDuration = 0,
                                      maximumCohortDuration = NULL,
                                      limitTo = "all",
                                      calendarStartDate = NULL,
                                      calendarEndDate = NULL) {
  if (limitTo == "" || is.null(limitTo)) {
    limitTo <- "all"
  }

  if (minimumCohortDuration == 0 && priorTime == 0 & followUpTime == 0 & limitTo == "all" &
    is.null(maximumCohortDuration) & is.null(calendarStartDate) & is.null(calendarEndDate)) {
    stop("No limit criteria specified")
  }

  subset <- LimitSubsetOperator$new()
  subset$name <- name
  subset$priorTime <- priorTime
  subset$followUpTime <- followUpTime
  subset$minimumCohortDuration <- minimumCohortDuration
  subset$maximumCohortDuration <- maximumCohortDuration
  subset$limitTo <- limitTo
  subset$calendarStartDate <- calendarStartDate
  subset$calendarEndDate <- calendarEndDate

  subset
}

#' Create Limit Subset Operator
#' @description
#' Subset cohorts using specified limit criteria.
#' deprecated This function is deprecated. Please use `createLimitSubsetOperator()` instead.
#' @param ... Arguments passed to the underlying operator.
#' @export
createLimitSubset <- function(...) {
  .Deprecated("createLimitSubsetOperator")
  createLimitSubsetOperator(...)
}

#' Create Cohort Subset Operator
#' @description
#' Subset cohorts using specified limit criteria.
#' deprecated This function is deprecated. Please use `createCohortSubsetOperator()` instead.
#' @param ... Arguments passed to the underlying operator.
#' @export
createCohortSubset <- function(...) {
  .Deprecated("createCohortSubsetOperator")
  createCohortSubsetOperator(...)
}

#' Create Demographic Subset Operator
#' @description
#' Subset cohorts using specified limit criteria.
#' deprecated This function is deprecated. Please use `createDemographicSubsetOperator()` instead.
#' @param ... Arguments passed to the underlying operator.
#' @export
createDemographicSubset <- function(...) {
  .Deprecated("createDemographicSubsetOperator")
  createDemographicSubsetOperator(...)
}
