#' @name PC_CoFM
#' @title Perform PCA-based Factor Estimation for CoFM
#' @description
#' This function performs Principal Component Analysis (PCA) on the correlation matrix
#' of the data to estimate factor loadings and uniquenesses. It is designed to work with
#' data generated by the \code{\link{CoFM}} function and calculates error metrics (MSE and
#' relative loss) by comparing estimates against the true parameters.
#'
#' @param data A matrix or data frame of input data (n x p). Usually the \code{$data} output
#'   from \code{CoFM}.
#' @param m Integer. The number of principal components (factors) to retain.
#' @param A Matrix. The true factor loadings matrix (p x m). Usually \code{$True_Params$A}
#'   from \code{CoFM}.
#' @param D Matrix. The true uniquenesses matrix (p x p). Usually \code{$True_Params$D}
#'   from \code{CoFM}.
#' @return A list containing:
#' \item{A2}{Estimated factor loadings matrix.}
#' \item{D2}{Estimated uniquenesses matrix.}
#' \item{MSESigmaA}{Mean Squared Error for factor loadings.}
#' \item{MSESigmaD}{Mean Squared Error for uniquenesses.}
#' \item{LSigmaA}{Relative loss metric for factor loadings.}
#' \item{LSigmaD}{Relative loss metric for uniquenesses.}
#'
#' @export
#'
#' @examples
#' # Examples should be fast and reproducible for CRAN checks
#' set.seed(123)
#'
#' # 1. Generate toy data using CoFM
#' sim_result <- CoFM(n = 200, p = 6, m = 2, type = "Clayton", param = 2.0)
#'
#' # 2. Extract true parameters and observed data
#' true_A <- sim_result$True_Params$A
#' true_D <- sim_result$True_Params$D
#' obs_data <- sim_result$data
#'
#' # 3. Apply PC method to estimate parameters and compute errors
#' pc_result <- PC_CoFM(data = obs_data, m = 2, A = true_A, D = true_D)
#'
#' # 4. Inspect results
#' pc_result$MSESigmaA
#' pc_result$MSESigmaD
#' head(pc_result$A2)
PC_CoFM <- function(data, m, A, D) {

  # 1. Input Validation
  if (!is.matrix(data) && !is.data.frame(data)) {
    stop("Data must be a matrix or data frame.")
  }
  X <- as.matrix(data)
  p <- ncol(X)

  if (!is.numeric(m) || length(m) != 1L || is.na(m) || m <= 0 || m > p) {
    stop("m must be a positive integer and cannot exceed the number of variables (ncol(data)).")
  }
  m <- as.integer(m)

  if (!is.matrix(A) || nrow(A) != p || ncol(A) != m) {
    stop("A must be a matrix of dimension p x m, where p = ncol(data) and m is the number of factors.")
  }
  if (!is.matrix(D) || nrow(D) != p || ncol(D) != p) {
    stop("D must be a matrix of dimension p x p, where p = ncol(data).")
  }

  # 2. Standardize data
  X <- scale(X)

  # 3. Correlation Matrix
  S <- stats::cor(X)
  # For correlation matrix, diag(S) should be 1, but keep general form.

  # 4. Eigen decomposition (symmetric = TRUE improves stability)
  eig <- base::eigen(S, symmetric = TRUE)

  # Ensure sorted descending (defensive)
  idx <- order(eig$values, decreasing = TRUE)
  eig_values <- eig$values[idx]
  eig_vectors <- eig$vectors[, idx, drop = FALSE]

  # 5. Loadings estimate: Ahat = V_m * sqrt(Lambda_m)
  rowname <- paste0("X", seq_len(p))
  colname <- paste0("Factor", seq_len(m))
  Ahat <- matrix(0, nrow = p, ncol = m, dimnames = list(rowname, colname))

  for (i in seq_len(m)) {
    val <- max(0, eig_values[i]) # numerical safety
    Ahat[, i] <- sqrt(val) * eig_vectors[, i]
  }

  # 6. Uniquenesses: Dhat = diag(S) - diag(Ahat Ahat')
  h2 <- diag(Ahat %*% t(Ahat))
  Dhat <- diag(S) - h2
  Dhat <- pmax(Dhat, 0) # clamp small negative values due to numerical error
  Dhat_mat <- diag(Dhat, nrow = p)

  # 7. Error metrics (Frobenius norms)
  MSESigmaA <- matrixcalc::frobenius.norm(Ahat - A)^2 / (p^2)
  MSESigmaD <- matrixcalc::frobenius.norm(Dhat_mat - D)^2 / (p^2)

  LSigmaA <- matrixcalc::frobenius.norm(Ahat - A)^2 / matrixcalc::frobenius.norm(A)^2
  LSigmaD <- matrixcalc::frobenius.norm(Dhat_mat - D)^2 / matrixcalc::frobenius.norm(D)^2

  list(
    A2 = Ahat,
    D2 = Dhat_mat,
    MSESigmaA = MSESigmaA,
    MSESigmaD = MSESigmaD,
    LSigmaA = LSigmaA,
    LSigmaD = LSigmaD
  )
}
