#' @name FanPC_CoFM
#' @title Perform Factor Analysis via Principal Component (FanPC) for CoFM
#' @description
#' This function estimates factor loadings and uniquenesses using a principal-component
#' (FanPC) approach. It then compares these estimates with the true parameters (A and D)
#' to calculate Mean Squared Errors (MSE) and relative loss metrics. This is designed to
#' work with data generated by the \code{\link{CoFM}} function.
#'
#' @param data A matrix or data frame of input data (n x p). Usually the \code{$data}
#'   output from \code{CoFM}.
#' @param m Integer. The number of principal components (factors) to extract.
#' @param A Matrix. The true factor loadings matrix (p x m). Usually \code{$True_Params$A}
#'   from \code{CoFM}.
#' @param D Matrix. The true uniquenesses matrix (p x p). Usually \code{$True_Params$D}
#'   from \code{CoFM}.
#' @return A list containing:
#' \item{AF}{Estimated factor loadings matrix (p x m).}
#' \item{DF}{Estimated uniquenesses matrix (p x p).}
#' \item{MSESigmaA}{Mean Squared Error for factor loadings.}
#' \item{MSESigmaD}{Mean Squared Error for uniquenesses.}
#' \item{LSigmaA}{Relative loss metric for factor loadings.}
#' \item{LSigmaD}{Relative loss metric for uniquenesses.}
#'
#' @export
#'
#' @examples
#' # Examples should be fast and reproducible for CRAN checks
#' set.seed(123)
#'
#' # 1. Generate toy data using CoFM
#' sim_result <- CoFM(n = 200, p = 6, m = 2, type = "Clayton", param = 2.0)
#'
#' # 2. Extract true parameters and observed data
#' true_A <- sim_result$True_Params$A
#' true_D <- sim_result$True_Params$D
#' obs_data <- sim_result$data
#'
#' # 3. Apply FanPC and compute error metrics
#' fanpc_result <- FanPC_CoFM(data = obs_data, m = 2, A = true_A, D = true_D)
#'
#' # 4. Inspect results
#' fanpc_result$MSESigmaA
#' fanpc_result$MSESigmaD
#' head(fanpc_result$AF)
FanPC_CoFM <- function(data, m, A, D) {

  # 1. Input validation
  if (!is.matrix(data) && !is.data.frame(data)) {
    stop("Data must be a matrix or data frame.")
  }
  X <- as.matrix(data)
  p <- ncol(X)

  if (!is.numeric(m) || length(m) != 1L || is.na(m) || m <= 0 || m > p) {
    stop("m must be a positive integer and cannot exceed the number of variables (ncol(data)).")
  }
  m <- as.integer(m)

  if (!is.matrix(A) || nrow(A) != p || ncol(A) != m) {
    stop("A must be a matrix of dimension p x m, where p = ncol(data) and m is the number of factors.")
  }
  if (!is.matrix(D) || nrow(D) != p || ncol(D) != p) {
    stop("D must be a matrix of dimension p x p, where p = ncol(data).")
  }

  # 2. Standardize
  X <- scale(X)

  # 3. Correlation matrix
  SigmahatF <- stats::cor(X)

  # 4. Eigen decomposition (symmetric = TRUE for stability)
  eig <- base::eigen(SigmahatF, symmetric = TRUE)

  # Sort eigenvalues/vectors descending (defensive)
  ind <- order(eig$values, decreasing = TRUE)
  lambdahat <- eig$values[ind]
  Q <- eig$vectors[, ind, drop = FALSE]

  # 5. Estimated loadings (PCA-style): AF = Q_m * sqrt(lambda_m)
  lam_use <- pmax(lambdahat[1:m], 0)
  AF <- Q[, 1:m, drop = FALSE] %*% diag(sqrt(lam_use), nrow = m)

  # 6. Estimated uniquenesses matrix
  hF <- diag(AF %*% t(AF))
  DF <- diag(SigmahatF) - hF
  DF <- pmax(DF, 0)                # clamp small numerical negatives
  DF_mat <- diag(DF, nrow = p)

  # 7. Errors (Frobenius norms)
  # Note: matrixcalc is in Imports, so safe to call directly
  MSESigmaA <- matrixcalc::frobenius.norm(AF - A)^2 / (p^2)
  MSESigmaD <- matrixcalc::frobenius.norm(DF_mat - D)^2 / (p^2)

  LSigmaA <- matrixcalc::frobenius.norm(AF - A)^2 / matrixcalc::frobenius.norm(A)^2
  LSigmaD <- matrixcalc::frobenius.norm(DF_mat - D)^2 / matrixcalc::frobenius.norm(D)^2

  list(
    AF = AF,
    DF = DF_mat,
    MSESigmaA = MSESigmaA,
    MSESigmaD = MSESigmaD,
    LSigmaA = LSigmaA,
    LSigmaD = LSigmaD
  )
}
