#' plotComp
#'
#' This function plots the specified mixed model so that it can be visualized.
#' It utilizes the gamma function to allow approximations of the poisson
#' distributions, allowing for a smooth plot.
#'
#' @param comps List of components to be plotted. Output from fitModels.
#' @param compName Name of the component to plot
#' @param saveRes Whether or not to save results. Default is F.
#' @param saveDir Where to save plots, must be provided if using saveDir
#' @param runName Used to add a runName to the file output. Default is "".
#' @keywords component plot
#' @import ggplot2
#' @export
#' @return Plots the components to allow visualization
#' @examples 
#' \donttest{plotComp(compsExp, compName = "segsize")}
plotComp = function(comps,compName,saveRes=FALSE,saveDir=NULL,runName = ""){
  
  pois = FALSE
  plotTitle = paste(compName,"Components")
  xlabel = switch (compName,
                  "segsize"     = "Segment size",
                  "bp10MB"      = "Breakpoint count per 10MB",
                  "osCN"        = "Number of Oscillations",
                  "changepoint" = "Copy-number change point",
                  "copynumber"  = "Copy-number",
                  "bpchrarm"    = "Breakpoint count per chr arm",
                  "ploidy"      = "Ploidy")
  
  #Grabs parameters and finds range based on whether it is pois or not
  plotparam = comps[[compName]]
  if (!is.null(nrow(plotparam))) {
    range = seq(0,max(plotparam[1,]+(3*plotparam[2,])),0.1)
    numParam = ncol(plotparam)
  }
  else {
    range = seq(0,max(plotparam*2),.1)
    numParam = length(plotparam)
    pois = T
  }
  
  #Generates the sample data for the plot
  data = data.frame()
  for(i in 1:numParam)
  {
    if (pois){
      yVals = sapply(range,function(x){(exp(-plotparam[i])*plotparam[i]^x)/gamma(x+1)})
    }
    else {
      yVals = stats::dnorm(range,plotparam[1,i],plotparam[2,i])
    }
    yVals[is.infinite(yVals) | is.na(yVals)] = 0
    yVals[yVals>3] = 3
    comp = cbind(val=yVals,x=range,comp=i)
    data = rbind(data,comp)
  }
  data$comp=as.factor(data$comp)
  
  #Scaling function used to have consistent labels on the plot
  scaleFUN <- function(x) sprintf("%.2f", x) 
  
  #Generates the plot and prints it out
  plot = ggplot(data,aes_string(x="x",y="val",color="comp")) + geom_line(size=0.25) +
    scale_colour_discrete(labels=paste0(rep("Comp",numParam),1:numParam),
                          name="Components") +
    scale_y_continuous(labels = scaleFUN) +
    xlab(xlabel) + ylab("") +
    ggtitle(plotTitle)
  print(plot)
  
  if (runName != "")
    runName = paste0(runName,"_")
  
  if(saveRes){
    if (is.null(saveDir)){
	stop('No saveDir provided for saveRes. Please set saveDir.')
    }
    ggsave(paste0(saveDir,runName,compName,"Comps.tiff"),plot,width = 4,
           height = 3, units = "in", dpi = 300,)
  }
}
