#' @rdname MHSU_status_basic
#' @title Concurrent Mental Health and Substance Use status detection in North American Healthcare Administrative Databases
#' @description Concurrent Mental Health and Substance Use status is detected in North American Healthcare Administrative Databases  using clinician's parameters of interest such as minimum number of hospital visits, minimum number of physician services, time lag between them and plausible ICD diagnostics. It is assumed that all input data dates fall into the maximum time span between mental health status and substance use status.
#' @param inputdata  a dataframe including columns: `ClientID`,  `VisitDate`,  `Diagnostic_H`, and `Diagnostic_P`
#' @param n_MHH minimum number of potential mental health related hospital visits
#' @param n_MHP minimum number of potential mental health related medical service physician visits
#' @param n_SUH minimum number of potential substance use related hospital visits
#' @param n_SUP minimum number of potential substance use related medical service physician visits
#' @param t_MH maximum time lag (in days) between all hospital visits and all medical service physician visits
#' @param t_SU maximum time lag (in days) between all hospital visits and all medical service physician visits
#' @param t_MHSU the maximum time span (in days) between mental health status and substance use status
#' @param ICD_MH plausible list of Mental Health status diagnostic codes
#' @param ICD_SU plausible list of Substance Use status diagnostic codes
#' @returns a dataframe matrix with  Clients' ID, earliest dates of Substance Use/Mental Health, latest dates of Substance Use/Mental Health, Mental Health status, Substance Use status, and, Concurrent Mental Health and Substance Use status
#'
#' @references
#' Khan S. (2017). Concurrent mental and substance use disorders in Canada. Health reports, 28(8), 3–8, Ottawa, ON, Canada, PMID: 29044442. https://pubmed.ncbi.nlm.nih.gov/29044442/
#' Canadian Institute for Health Information. (2022). Canadian coding standards for version 2022 ICD-10-CA and CCI. Canadian Institute for Health Information. Ottawa, Ontario, Canada. https://www.cihi.ca/en/version-2022-icd-10-cacci-classifications-canadian-coding-standards-and-related-products
#' Centers for Disease Control and Prevention. (2024). International classification of diseases, tenth revision, clinical modification (ICD-10-CM) 2024. National Center for Health Statistics. Atlanta, GA, United States. https://www.cdc.gov/nchs/icd/icd-10-cm/index.html
#' World Health Organization. (2019). International classification of diseases for mortality and morbidity statistics (11th Revision). Geneva, Switzerland: World Health Organization.  https://icd.who.int.
#' Casillas, S. M., Scholl, L., Mustaquim, D., & Vivolo-Kantor, A. (2022). Analysis of trends and usage of ICD-10-CM discharge diagnosis codes for poisonings by fentanyl, tramadol, and other synthetic narcotics in emergency department data. Addictive Behaviors Reports, 16, 100464. Atlanta, GA, USA. https://doi.org/10.1016/j.abrep.2022.100464
#' Keen, C., Kinner, S. A., Young, J. T., Jang, K., Gan, W., Samji, H., Zhao, B., Krausz, M., & Slaunwhite, A. (2021). Prevalence of co‐occurring mental illness and substance use disorder and association with overdose: a linked data cohort study among residents of British Columbia, Canada. Addiction, 117(1), 129–140. https://doi.org/10.1111/add.15580
#' Lavergne, M. R., Loyal, J. P., Shirmaleki, M., Kaoser, R., Nicholls, T., Schütz, C. G., Vaughan, A., Samji, H., Puyat, J. H., Kaulius, M., Jones, W., & Small, W. (2022). The relationship between outpatient service use and emergency department visits among people treated for mental and substance use disorders: Analysis of population-based administrative data in British Columbia, Canada. BMC Health Services Research, 22(1), 477. https://doi.org/10.1186/s12913-022-07759-z
#'
#' @examples
#' data(SampleRWD)
#' myexample <- SampleRWD[, c(1:4)]
#' SampleMHSU_1 <- MHSU_status_basic(myexample,
#' n_MHH=1, n_MHP=1, n_SUH=1, n_SUP=1, t_MH=60, t_SU=60, t_MHSU=365,
#' ICD_MH=c("F060","F063","F064", "F067"),
#' ICD_SU=c("F100","T4041","F120","F140"))
#' head(SampleMHSU_1)
#'
#' @import dplyr
#' @import tidyr
#' @import purrr
#'
#' @export
#'
MHSU_status_basic <- function(inputdata, n_MHH, n_MHP, n_SUH, n_SUP, t_MH, t_SU,t_MHSU, ICD_MH, ICD_SU) {
  # Check if the input data is empty
  if (nrow(inputdata) == 0) {
    return(data.frame(
      ClientID = integer(),
      earliestdate_MH = as.Date(character()),
      latestdate_MH = as.Date(character()),
      earliestdate_SU = as.Date(character()),
      latestdate_SU = as.Date(character()),
      MH_status = character(),
      SU_status = character(),
      MHSU_status = character()
    ))
  }
  # Step 1: Use the ancillary function g_MH to get MH-related data
  mh_data <- MH_status(inputdata=inputdata, t_MH=t_MH, n_MHH=n_MHH, n_MHP=n_MHP, ICD_MH=ICD_MH)

  # Step 2: Use the ancillary function g_SU to get SU-related data
  su_data <- SU_status(inputdata=inputdata, t_SU=t_SU, n_SUH=n_SUH, n_SUP=n_SUP, ICD_SU=ICD_SU)

  # Step 3: Merge the results from g_MH and g_SU by ClientID
  combined_data <- mh_data %>%
    full_join(su_data, by = "ClientID") %>%
    mutate(
      earliestdate_MHSU = pmin(as.Date(.data$earliestdate_MH), as.Date(.data$latestdate_MH),
                               as.Date(.data$earliestdate_SU), as.Date(.data$latestdate_SU), na.rm = TRUE),
      latestdate_MHSU = pmax(as.Date(.data$earliestdate_MH), as.Date(.data$latestdate_MH),
                             as.Date(.data$earliestdate_SU), as.Date(.data$latestdate_SU), na.rm = TRUE),
      deltadate_MHSU = as.numeric(.data$latestdate_MHSU - .data$earliestdate_MHSU),
      MHSU_status = ifelse(MH_status == "YES" & SU_status == "YES" & .data$deltadate_MHSU <= t_MHSU, "YES", "NO")
    )


  # Step 4: Select and return the required columns
  result <- combined_data %>%
    select(.data$ClientID, .data$earliestdate_MH, .data$latestdate_MH, .data$earliestdate_SU, .data$latestdate_SU, MH_status, SU_status, .data$MHSU_status)

  return(result)
}

