% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/arguments.R
\name{arguments}
\alias{arguments}
\title{Shared documentation for arguments}
\arguments{
\item{model}{character string (case-insensitive) indicating the model to be
fitted to the data. Must be one of the options listed in the Section
\link[=arguments.model]{Supported Models}.}

\item{n}{the sample size of the output time series \code{yt} after burn-in
(simulation only). Default is \code{n = 1}.}

\item{nnew}{optional; the number of out-of sample predicted values required
(extract and fit only). Default is \code{nnew = 0}.}

\item{burn}{the length of the \sQuote{burn-in} period (simulation only).
Default is \code{burn = 0}. The first \code{burn} values of the time series are
discarded.}

\item{yt}{numeric vector with the observed time series (extract and fit
only). Missing values (NA's) are not allowed.}

\item{y.start}{optional; an initial value for \eqn{Y_t} (to initialize
recursions when \eqn{t < 1}). Default is \code{y.start = NULL}, in which case,
the recursion assumes that \eqn{Y_t = g_{12}^{-1}(0)}, for \eqn{t < 1}.
Only relevant if \eqn{p > 0}.}

\item{rho}{the quantile being considered in the conditional distribution of
\eqn{Y_t} (only present in Kumaraswamy and Unit-Weibull based models). It
can be any positive number between 0 and 1. Default is \code{rho = 0.5}, which
corresponds to the median.}

\item{y.lower}{the lower limit for the Kumaraswamy density support. Default
is \code{y.lower = 0}.}

\item{y.upper}{the upper limit for the Kumaraswamy density support. Default
is \code{y.upper = 1}.}

\item{vt.start}{optional; an initial value for  \eqn{\vartheta_t} (to
initialize recursions when \eqn{t < 1}). Default is \code{vt.start = NULL}, in
which case, the recursion assumes that \eqn{\vartheta_t = g_{22}^{-1}(0)},
for \eqn{t < 1}. Only relevant if \eqn{\nu} is time-varying and \eqn{p_2 >
  0}.}

\item{e2.start}{optional; an initial value for  \eqn{g_{23}(e_{1t})} (to
initialize recursions when \eqn{t < 1}). Default is \code{e2.start = NULL}, in
which case, the recursion assumes that \eqn{e_{1t} = g_{23}^{-1}(0)}, for
\eqn{t < 1}. Only relevant if \eqn{\nu} is time-varying and \eqn{q_2 > 0}
or \eqn{d_2 > 0}.}

\item{xreg}{optional; external regressors. Can be specified as a vector, a
matrix or a list. Default is \code{xreg = NULL}. For details, see the Section
\link[=arguments.regressors]{Regressors format}.}

\item{xnew}{optional; \code{nnew} new observations of the external regressors
(extract and fit only). Follows the same format is the same as \code{xreg}.
Default is \code{xnew = NULL}.}

\item{xreg.start}{optional; initial value for the regressors (to initialize
recursion). Can be specified as a vector or a list. Default is
\code{xreg.start = NULL}, in which case, the average of the first \eqn{p}
values (AR order) is used. Only relevant if \code{xreg} is provided, \code{xregar = TRUE} and \eqn{p > 0}. For details, see the Section
\link[=arguments.regressors]{Regressors format}.}

\item{xregar}{a length 1 or 2 logical vector indicating whether \code{xreg} should
be included in the AR recursion for each part of the model. Default is
\code{xregar = TRUE}. Only relevant if \eqn{p > 0}. If a single value is
provided and \eqn{\nu} is time-varying, the same option is assumed for both
parts of the model. See the Section \sQuote{The BTSR structure} in
\link{btsr-package} for details.}

\item{inf}{a length 1 or 2 integer vector given the truncation point for
infinite sums. Default is \code{inf = 1000}. See the Section \link[=arguments.order]{Model Order} for details.}

\item{p}{optional; a length 1 or 2 integer vector given the order of the AR
polynomial (extract and fit only). Default is \code{p = NULL}. See the Section
\link[=arguments.order]{Model Order} for details.}

\item{q}{optional; a length 1 or 2 integer vector given the order of the MA
polynomial (extract and fit only). Default is \code{q = NULL}. See the Section
\link[=arguments.order]{Model Order} for details.}

\item{d}{a length 1 or 2 logical vector indicating whether the long memory
parameter \eqn{d} should be included in the model either as a fixed or
non-fixed parameter (fit only). If \code{d = FALSE}, internally the value of the
parameter \code{d} is fixed as 0. In this case, if \code{start} or \code{fixed.values}
include \code{d}, the value provided by the user is ignored.  If \eqn{\nu} is
time-varying and a single value is provided it is assumed that \eqn{d_1 =
  d_2 = d}.}

\item{ignore.start}{optional; logical value indicating whether the argument
\code{start} should be ignored (fit only). If starting values are not provided,
the function uses the default values and \code{ignore.start} is ignored. In case
starting values are provided and \code{ignore.start = TRUE}, those starting
values are ignored and recalculated. The default is \code{ignore.start = FALSE}.
Partial starting values are not allowed.}

\item{start}{optional; a list with the starting values for the non-fixed
coefficients of the model (fit only). The default is \code{start = NULL}, in
which case the function \link{coefs.start} is used internally to obtain starting
values for the parameters. For details on the expected format and the
arguments that can be passed through  \code{coefs}, see the Section \link[=arguments.coefs]{Model coefficients}.}

\item{coefs}{a list with the coefficients of the model (simulation and
extraction only). The default is \code{coefs = NULL}. For details on the
expected format and the arguments that can be passed through \code{coefs}, see
the Section \link[=arguments.coefs]{Model coefficients}.}

\item{lags}{optional; a list with the lags (integer values) that the entries
in \code{coefs} or \code{start} correspond to (extract and fit only). The default is
\code{lags = NULL}, in which the \code{lags} are computed from the \code{fixed.lags}
argument (if provided). When components are missing or empty in both,
\code{lags} and \code{fixed.lags}, the default behavior is to include all lags based
on \code{nreg = ncol(xreg)}, \code{p}, and \code{q}. For details, see the Section \link[=arguments.coefs]{Model coefficients}.}

\item{fixed.values}{optional; a list with the values of the coefficients that
are fixed (extract and fit only). The default is \code{fixed.values = NULL}. See
the Section \link[=arguments.coefs]{Model coefficients}.}

\item{fixed.lags}{optional; a list with the lags (integer values) that the
fixed values in \code{fixed.values} correspond to (extract and fit only). The
default is \code{fixed.lags = NULL}. For missing components, fixed values will
are set based on \code{lags}.}

\item{lower}{optional; list with the lower bounds for the parameters (fit
only). Default is \code{lower = NULL}. The default is to assume that the
parameters have no lower bound except for \code{nu}, for which de default is 0.
Only the bounds for bounded parameters need to be specified. The format of
\code{lower} and the arguments that can be passed through this list are the same
as the ones for \code{start}.}

\item{upper}{optional; list with the upper bounds for the parameters (fit
only). Default is \code{upper = NULL}. The default is to assume that the
parameters have no upper bound. Only the bounds for bounded parameters need
to be specified. The format of \code{lower} and the arguments that can be passed
through this list are the same as the ones for \code{start}.}

\item{map}{a non-negative integer from 1 to 5 corresponding to the map
function. Default is \code{map = 4}. See the Section \link[=arguments.map]{The map function}.}

\item{error.scale}{either 0 or 1; the scale for the error term. Default is
\code{error.scale = 1} (predictive scale).}

\item{linkg}{link functions. Can be specified as a character, two-character
vector or a named list. The corresponding text strings for currently
available links are listed in \link{link.btsr}. Default values depend on the
model. For some models default values override user specifications. See
the Section \link[=arguments.link]{Link defaults} for details.}

\item{linkh}{a character indicating which link must be associated to the
chaotic process.  See the Section \sQuote{The BTSR structure} in
\link{btsr-package} for details and \link{link.btsr} for valid links. Default is
\code{linkh = "linear"}.}

\item{configs.linkg}{a list with two elements, \code{ctt} and \code{power}, which
define the constant \eqn{a} and the exponent \eqn{b} in the link function
\eqn{g(x) = a x^b}. Each element can be specified as a numeric value, a
vector of size 2 or a named list. The default is \code{configs.linkg = NULL}.
See the Section \link[=arguments.link]{Link defaults}  for details.}

\item{configs.linkh}{a list with extra configurations for the link \eqn{h}.
For now, only used if \code{linkh = "linear"} or \code{"polynomial"}. Default is
\code{configs.linkh = list(ctt = 1, power = 1)}.}

\item{m}{a non-negative integer indicating the starting time for the sum of
the partial log-likelihood, given by \eqn{\ell = \sum_{t = m+1}^n \ell_t}
(extract and fit only). Default is \code{m = 0}. For details, see the Section
\link[=arguments.loglik]{The log-likelihood}.}

\item{llk}{logical; indicates whether the value of the log-likelihood
function should be returned (extract and fit only). Default is \code{llk =
  TRUE}.}

\item{sco}{logical; indicates whether the score vector should be returned
(extract and fit only). Default is \code{sco = FALSE}.}

\item{info}{logical; indicates whether the information matrix should be
returned (extract and fit only). Default is \code{info = FALSE}. For the fitting
function, \code{info} is automatically set to \code{TRUE} when \code{report = TRUE}.}

\item{extra}{logical, if \code{TRUE} the matrices and vectors used to calculate
the score vector and the information matrix are returned (extract and fit
only). Default is \code{extra = FALSE}. Ignored by BARC models.}

\item{control}{a list with configurations to be passed to the optimization
subroutines (fit only). Default is \code{control = NULL}. Missing arguments will
receive default values. For details, see \link{fit.control}.}

\item{report}{logical; indicates whether the summary from the fitted model
should be be printed (fit only). Default is \code{report = TRUE}, in which case
\code{info} is automatically set to \code{TRUE}.}

\item{complete}{logical; if FALSE returns only \code{yt}, else returns additional
components (simulation only). Default is \code{complete = FALSE}.}

\item{debug}{logical, if \code{TRUE} the output from FORTRAN is return (for
debugging purposes). Default is \code{debug = FALSE}.}

\item{...}{further arguments passed to the internal functions. See, for
instance, \link{summary.btsr} for details.}
}
\description{
This is the common documentation for all parameter in BTSR package.

The package handles function arguments in two compatible formats
\itemize{
\item \strong{Legacy structure} (pre-1.0.0). Used for models with fixed or
no \eqn{\nu} parameter. Automatically converted to the new format when
processed.

\item \strong{New structure} (1.0.0+). Required for models with
time-varying \eqn{\nu} parameter.
}
All functions accept both formats seamlessly, ensuring backward compatibility.
The internal processing automatically standardizes to the new structure.
}
\section{Supported Models}{
Internally, all models are handled by the same function and all models can be
obtained from the more general case \code{"*ARFIMAV"}. When a particular model
(e.g. \code{"BREG"} or \code{"BARMA"}) is invoked some default values are assumed.

The following table summarizes the available distributions and the
corresponding string to generate each model type. The character \code{V} at
the end of the string indicates that \eqn{\nu} is time-varying.

\if{html}{\out{<div class="sourceCode">}}\preformatted{+--------------+--------+------------+---------+-----------+---------+
| Distribution | i.i.d. | Regression | Short   | Long      | Chaotic |
|              | sample |            | Memory  | Memory    |         |
+--------------+--------+------------+---------+-----------+---------+
| Beta         | BETA   | BREG       | BARMA   | BARFIMA   | BARC    |
|              |        | BREGV      | BARMAV  | BARFIMAV  |         |
+--------------+--------+------------+---------+-----------+---------+
| Gamma        | GAMMA  | GREG       | GARMA   | GARFIMA   |         |
|              |        | GREGV      | GARMAV  | GARFIMAV  |         |
+--------------+--------+------------+---------+-----------+---------+
| Kumaraswamy  | KUMA   | KREG       | KARMA   | KARFIMA   |         |
|              |        | KREGV      | KARMAV  | KARFIMAV  |         |
+--------------+--------+------------+---------+-----------+---------+
| Matsuoka     | MATSU  | MREG       | MARMA   | MARFIMA   |         |
+--------------+--------+------------+---------+-----------+---------+
| Unit-Lindley | UL     | ULREG      | ULARMA  | ULARFIMA  |         |
+--------------+--------+------------+---------+-----------+---------+
| Unit-Weibull | UW     | UWREG      | UWARMA  | UWARFIMA  |         |
|              |        | UWREGV     | UWARMAV | UWARFIMAV |         |
+--------------+--------+------------+---------+-----------+---------+
}\if{html}{\out{</div>}}
}

\section{Default values}{
All models are special cases of the general \code{"*ARFIMAV"} structure. When a
specific model is selected via \code{model = "NAME"}, the package automatically
applies these default configurations (any parameter that does not appear in
the equations below is ignored)

\strong{i.i.d samples} (e.g., BETA, GAMMA,...)
\deqn{
 \eta_{1t} = \alpha_1 = \mu, \quad \eta_{2t} = \alpha_2  = \nu.
}
Fixed

\if{html}{\out{<div class="sourceCode">}}\preformatted{p <- q <- d <- 0
xreg <- NULL
linkg <- list(g11 = "linear", g2 = "linear",
              g21 = "linear", g23 = "linear")
}\if{html}{\out{</div>}}

\strong{Regression models with \eqn{\nu_t} constant over time} (e.g., BREG,
GREG,...)
\deqn{
 \eta_{1t} =
 g_{11}(\mu_t) = \alpha_1 + \boldsymbol{X}_{1t}'\boldsymbol{\beta}_1,
 \quad
 \eta_{2t} = \alpha_2 = \nu.
 }
Fixed

\if{html}{\out{<div class="sourceCode">}}\preformatted{p <- q <- d <- 0
xreg <- list(part1 = "user's regressors", part2 = NULL)
linkg <- list(g11 = "user's choice", g12 = "linear",
              g2 = "linear", g21 = "linear", g23 = "linear")
}\if{html}{\out{</div>}}

\strong{Regression models with \eqn{\nu_t} varying on time} (e.g. BREGV,
GREGV)
\deqn{
  \eta_{1t}  =
  g_{11}(\mu_t) = \alpha_1 + \boldsymbol{X}_{1t}'\boldsymbol{\beta}_1,
  \quad
  \eta_{2t}  =
  g_{21}(g_2(\nu_t)) = \alpha_2 + \boldsymbol{X}_{2t}'\boldsymbol{\beta}_2.
 }
Fixed

\if{html}{\out{<div class="sourceCode">}}\preformatted{p <- q <- d <- 0
linkg <- list(g11 = "user's choice", g12 = "linear",
              g2 = "user's choice", g21 = "user's choice",
              g22 = "linear", g23 = "linear")
}\if{html}{\out{</div>}}

\strong{Short-memory models with \eqn{\nu} constant over time (ARMA-like)} (e.g. BARMA,
GARMA,...)
\deqn{
  \begin{aligned}
  \eta_{1t} & =
  g_{11}(\mu_t) = \alpha_1 + \boldsymbol{X}_{1t}'\boldsymbol{\beta}_1 +
  \sum_{i=1}^{p_1} \phi_{1i}\bigl(g_{12}(Y_{t-i})-
  I_{X_1}\boldsymbol{X}_{1(t-i)}'\boldsymbol{\beta}_1\bigr) +
  \sum_{k=1}^{q_1} \theta_{1k} e_{1,t-k}, \\
  \eta_{2t} &  = \alpha_2 = \nu.
  \end{aligned}
  }
Fixed

\if{html}{\out{<div class="sourceCode">}}\preformatted{d <- 0
xreg <- list(part1 = "user's regressors", part2 = NULL)
linkg <- list(g11 = "user's choice", g12 = "user's choice",
              g2 = "linear", g21 = "linear", g23 = "linear")
}\if{html}{\out{</div>}}

\strong{Short-memory models with \eqn{\nu_t} varying on time} (e.g. BARMAV,
GARMAV,...)
\deqn{
  \begin{aligned}
  \eta_{1t} & =
  g_{11}(\mu_t) =\alpha_1 + \boldsymbol{X}_{1t}'\boldsymbol{\beta}_1 +
  \sum_{i=1}^{p_1} \phi_{1i}\big(g_{12}(Y_{t-i})-
  I_{X_1}\boldsymbol{X}_{1(t-i)}'\boldsymbol{\beta}_1\big) +
  \sum_{k=1}^{q_1} \theta_{1k} r_{t-k},\\
  \vartheta_t & = g2(\nu_t)\\
  \eta_{2t} &
  = g_{21}(\vartheta_t) =\alpha_2 + \boldsymbol{X}_{2t}' \boldsymbol{\beta}_2
  + \sum_{i=1}^{p_2} \phi_{2i}\big(g_{22}(\vartheta_{t-i})-
  I_{X_2}\boldsymbol{X}_{2(t-i)}'\boldsymbol{\beta}_2\big) +
  \sum_{k=1}^{q_2} \theta_{2k}  g_{23}(e_{1,t-k}).
  \end{aligned}
 }
Fixed

\if{html}{\out{<div class="sourceCode">}}\preformatted{d <- 0
}\if{html}{\out{</div>}}

\strong{Long-memory models with \eqn{\nu} constant over time (ARFIMA-like
models)} (e.g. BARFIMA, GARFIMA,...)
\deqn{
 \begin{aligned}
 \eta_{1t} & =
 g_{11}(\mu_t) =\alpha_1 + \boldsymbol{X}_{1t}'\boldsymbol{\beta}_1 +
 \sum_{i=1}^{p_1} \phi_{1i}\big(g_{12}(Y_{t-i})-
 I_{X_1}\boldsymbol{X}_{1(t-i)}'\boldsymbol{\beta}_1\big) +
 \sum_{k=1}^\infty c_{1k} r_{t-k},\\
 \eta_{2t} &  =\alpha_2 =  \nu.
 \end{aligned}
}
Fixed

\if{html}{\out{<div class="sourceCode">}}\preformatted{p <- c("user's p", 0)
q <- c("user's q", 0)
d <- c("user's d", 0)
xreg <- list(part1 = "user's regressors", part2 = NULL)
linkg <- list(g11 = "user's choice", g12 = "user's choice",
              g2 = "linear", g21 = "linear", g23 = "linear")
}\if{html}{\out{</div>}}
}

\section{Reproducing Models from the Literature}{
This section summarizes how to replicate well-known time series models from
the literature using the BTSR package. For each model type, we provide the
necessary parameter settings and references to the original publications.
These configurations act as templates, helping users correctly apply the
package to reproduce results or extend established models.

Key arguments (e.g., \code{error.scale}, \code{xregar}, \code{y.lower},
\code{y.upper}, \code{rho}) should be set to match the specifications in the
referenced articles. While we focus on the \verb{btsr.*} functions (see
\link{BTSR.functions}), all models can also be implemented using the corresponding
parent model functions (for details, see \link{BTSR.parent.models}).

\strong{i.i.d. samples}: The arguments \code{error.scale} and \code{xregar} are
ignored.
\itemize{
\item Beta distribution with parameters  \code{shape1} and \code{shape2} compatible
with the one from \link{rbeta}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "BETA"
alpha = shape1/(shape1 + shape2)
nu = shape1 + shape2
}\if{html}{\out{</div>}}

\item Gamma distribution with parameters \code{shape} and \code{scale} compatible with
the one from \link{rgamma}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "GAMMA"
alpha = shape*scale
nu = shape
}\if{html}{\out{</div>}}

\item Kumaraswamy distribution with shape parameters \code{shape1} and \code{shape2}
(respectively denoted by \eqn{a} and \eqn{b} in
\insertCite{kumaraswamy1980;nobrackets}{BTSR}):

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "KUMA"
alpha = (y.lower - y.upper)*(1 - (1-rho)^1/shape2)*1/shape1 + y.lower
nu = shape1
}\if{html}{\out{</div>}}

\strong{Warning:} Choose \eqn{\mu},  \eqn{\nu} and \eqn{\rho} carefully
since \eqn{|\log(1-\rho)| >> |\log(1 - \mu^\nu)|} may cause numerical
instability.

\item Matsuoka distribution with shape parameter \code{shape}
\insertCite{matsuoka2024}{BTSR}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "MATSU"
alpha = (shape/(shape+1))^(3/2)
}\if{html}{\out{</div>}}

\item Unit-Lindley distribution with parameter \code{theta}
\insertCite{mazucheli2018}{BTSR}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "UL"
alpha = 1/(1 + theta)
}\if{html}{\out{</div>}}

\item Unit-Weibull distribution with parameter \code{mu}, \code{beta} and \code{tau} from
\insertCite{mazucheli2019}{BTSR}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "UW"
alpha = mu
nu = beta
rho = tau
}\if{html}{\out{</div>}}

}

\strong{Regression models}: the argument \code{error.scale} and all entries but
\code{g11} in \code{linkg} are ignored
\itemize{
\item Beta regression \insertCite{ferrari2004}{BTSR}: \code{model = "BREG"}

\item Kumaraswamy regression \insertCite{mitnikbaek2013}{BTSR}:
\code{model = "KREG"}.

\item Unit-Lindley regression \insertCite{mazucheli2018}{BTSR}:
\code{model = "ULREG"}.

\item Unit-Weibull regression \insertCite{mazucheli2019}{BTSR}:
\code{model = "UWREG"}.
}

\strong{ARMA-like models}
\itemize{
\item BARMA model \insertCite{rocha2009,rocha2017}{BTSR}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "BARMA"
error.scale = 1
xregar = TRUE
}\if{html}{\out{</div>}}

\item KARMA model \insertCite{bayer2017}{BTSR}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "KARMA"
error.scale = 1
xregar = TRUE
y.lower = 0
y.upper = 1
rho = 0.5
}\if{html}{\out{</div>}}

\item GARMA model \insertCite{prass2025}{BTSR}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "GARMA"
error.scale = 0
}\if{html}{\out{</div>}}

\item MARMA model \insertCite{pumi2025matsu}{BTSR}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "MARMA"
error.scale = 1
xregar = TRUE
}\if{html}{\out{</div>}}

\item ULARMA model \insertCite{pumi2025ul}{BTSR}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "ULARMA"
error.scale = 1
xregar = TRUE
}\if{html}{\out{</div>}}

}

\strong{ARFIMA-like models}
\itemize{
\item BARFIMA model \insertCite{pumi2019}{BTSR}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{model = "BARFIMA"
error.scale = 1
xregar = TRUE
d = TRUE (for fitting)
}\if{html}{\out{</div>}}

}

\strong{Chaotic models}
\itemize{
\item BARC model \insertCite{pumi2021}{BTSR}: set \code{model = "BARC"} and
\code{error.scale = 1}.
}
}

\section{Regressors format}{
In-sample (\code{xreg}) and out-of-sample values (\code{xnew}) for regressors can be
provided in two formats
\itemize{
\item \strong{Legacy structure:} a vector or matrix. Internally \code{xreg} is
converted to \code{xreg = list(part1 = xreg, part2 = NULL)}. The same applies to
\code{xnew}

\item \strong{New structure:} a list with elements \code{part1} (regressors for
first model component) and \code{part2} (regressors for second model component),
each being a vector or matrix.
}

\code{xreg.start} can be provided in two formats
\itemize{
\item \strong{Legacy structure:} a vector with initial values for each
regressor. Internally \code{xreg.start} is converted to
\code{xreg.start = list(part1 = xreg.start, part2 = NULL)}.

\item \strong{New structure:} a list with elements \code{part1} and \code{part2},
each a vector of initial values for the respective regressors.
}

The following rules apply to \code{xreg}, \code{xnew} and \code{xreg.start}
\itemize{
\item if \code{model} corresponds to a case where \eqn{\nu} is constant over time
(e.g., \code{model = "BARMA"}), \code{part2} is ignored.

\item For simulation, regressors must include \code{n + burn} observations.

\item For model fitting, parameter initialization, or component extraction,
the number of regressor observations must match the length of the observed
time series \code{yt}.

\item When \code{xreg = NULL} or \code{nnew = 0}, \code{xnew} is ignored. If \code{nnew > 0} and
the number of regressors in \code{xnew} does not match \code{xreg} an error message is
issued.

\item If starting values for \code{xreg} are not provided and \eqn{p_i > 0} for
the \eqn{i}th part of the model, the default behavior is to assume
\deqn{
  \displaystyle\boldsymbol{X}_t =
  \frac{1}{p_i}\sum_{k = 1}^{p_i} \boldsymbol{X}_k,
 \quad \text{for } t< 1.
 }
}
}

\section{Model Order}{
The coefficients \eqn{\{c_{ik}\}_{k\geq 0}} are defined through the relation
(see the section \sQuote{The BTSR Structure} in \link{btsr-package})
\deqn{
c_i(z) := (1-L)^{-d_i}\theta_i(z) = \sum_{k = 0}^\infty c_{ik}z^k, \quad i \in \{1,2\}.
}
where \eqn{\theta_i(z) = \sum_{k = 0}^{q_i} \theta_{ik}z^k} is the moving
average characteristic polynomial, with order \eqn{q_i}. For practical
purposes, the following approximation is used
\deqn{
c_i(z) \approx \sum_{k = 0}^{K_i} c_{ik}z^k,
}
for some \eqn{K_i} sufficiently large. \cr

\code{inf} corresponds to the truncation point for all infinite sums using the
coefficients \eqn{\{c_{ik}\}_{k\geq 0}}, \eqn{i \in \{1,2\}}, including
samples generation and derivatives calculation. It can be provided as either
a single integer (legacy format) or a length 2 integer vector (new format)
specifying the trunction points for \code{part1}/\code{part2}. If \eqn{\nu} is
time-varying and a single value is provided the same value is used for both
parts. When \eqn{d = 0}, Fortran automatically sets \code{inf} to \eqn{q} (MA
order). \cr

By default \code{p} and \code{q} are set to \code{NULL}, in which case their values are
computed internally, based on the size of the argument \code{phi} and \code{theta},
respectively, in the lists of coefficients (or staring values), fixed lags,
and fixed values. For fitting purposes, if \code{p} (analogously, \code{q}) and \code{start}
are both \code{NULL}, an error message is issued. These parameters can be
provided as either a single integer (legacy format) or a length 2 integer
vector (new format) specifying orders for \code{part1}/\code{part2}. If \eqn{\nu} is
time-varying and a single value of \code{p} (analogously, \code{q}) is provided it is
assumed that \eqn{p_1 = p_2 = p} (analogously, \eqn{q_1 = q_2 = q}).
}

\section{Model coefficients}{
\code{start}, \code{coefs}, \code{fixed.values}, \code{lags} and \code{fixed.lags} can be specified in
one of two ways
\itemize{
\item \strong{Legacy structure:} a list with optional components \code{alpha},
\code{beta}, \code{phi}, \code{theta}, \code{d}, \code{u0} (BARC only) and required argument \code{nu}
(except for one-parameter models such as ULARMA and MARMA).

\item \strong{New structure:} a list with elements \code{part1} and \code{part2}, each
being a list with with optional components \code{alpha}, \code{beta}, \code{phi}, \code{theta},
\code{d} and  \code{u0} (BARC only).
}

The optional arguments in this lists are
\itemize{
\item \code{alpha}: a numeric value corresponding to the intercept. For i.i.d.
corresponds to the mean of the distribution.

\item \code{beta}: a vector of coefficients corresponding to the regressors in
\code{xreg}.

\item \code{phi}: a vector of autoregressive coefficients.

\item \code{theta}: for BARC models, this is the parameter for the map function
(see \link{BARC.functions} for details). For any other model, this is a vector
of moving average coefficients corresponding to the MA order.

\item\code{d}: a numeric value corresponding to the long memory parameter.

\item \code{u0}: a numeric value in the interval \eqn{(0,1)}, corresponding to
the value of the random variable \eqn{U_0}. See \link{BARC.functions} for
details.

\item\code{nu}: distribution related parameter, usually the dispersion.
}

The following rules apply for these lists and their arguments.\cr

\emph{Simulation:}
\itemize{
\item Passing \code{coefs} as an empty list will result in an error message.

\item \code{start} and \code{fixed.values} (consequently, \code{fixed.lags}) are not used.

\item If \code{xreg} is provided but \code{coefs} does not include a \code{beta} argument,
an error message is issued.

\item \code{phi} must be a vector of length \eqn{p} (the AR order), meaning all
coefficients must be provided, including zeros.

\item \code{theta} (non-BARC models) must be a vector of length \eqn{q} (the MA
order), meaning all coefficients must be provided, including zeros.
}

\emph{Extraction:}
\itemize{
\item One dimensional parameters (e.g. \code{alpha}) that do not appear in
\code{coefs} are assume to be fixed.

\item An error message will be issued if both \code{coefs} and \code{fixed.values} are
both empty.

\item If \eqn{\nu} is not constant over time and \code{nu} is missing in both
\code{coefs} and \code{fixed.values}, an error message is issued (except for
one-parameter models such as ULARMA and MARMA). Ignored if the new format
is used.
}

\emph{Fitting:}
\itemize{
\item One dimensional parameters (e.g. \code{alpha}) cannot appear in both
\code{start} and \code{fixed.values}.

\item \code{coefs} is not used.
}

\emph{Extraction and fitting:}
\itemize{
\item Coefficients may include both fixed lags (with values in
\code{fixed.values}) and non-fixed lags (with values in \code{coefs} or \code{start}).

\item \code{lags} and \code{fixed.lags} are complementary. Either suffices, or mix
them (e.g., \code{lags} for some parameters, \code{fixed.lags} for others).

\item For one dimensional parameters, the lag is obviously always 1 and can
be suppressed when the parameter added to the \code{fixed.values} list.

\item For extraction, if \code{coefs = NULL}, one dimensional parameters that do
not appear in \code{fixed.values} are assumed to be non-fixed. The same goes for
fitting when \code{start = NULL} or \code{ignore.start = TRUE}.

\item If \code{coefs}/\code{start} is provided, one dimensional parameters that do not
appear in this list are assumed to be fixed.

\item By default, if a given vector has fixed lags and the corresponding
entry in \code{fixed.values} is empty, the fixed values are set as zero.

\item If parameter values are provided in \code{coefs}, \code{start} or \code{fixed.values}
and the size of the vector is not the same as the dimension of the
parameters, either \code{lags} or \code{fixed.lags} must also be provided.
}
}

\section{The map function}{
The map function \eqn{T:[0,1] \to [0,1]} in BARC models is a dynamical
system, i.e., a function, potentially depending on a \eqn{r}-dimensional
vector of parameters \eqn{\theta}. As for today, for all implemented maps,
\eqn{r = 1}.

Available choices are
\itemize{
\item \code{map = 1}, \eqn{\theta = k}, for \eqn{k} integer greater or equal to 2.
\deqn{T(u) = (ku)(\text{mod } 1)}

\item \code{map = 2}, \eqn{0 \le \theta \le 1}
\deqn{T(u) = \dfrac{u}{\theta}I( u < \theta) +
  \theta\dfrac{(u - \theta)}{(1 - \theta)}I(u \ge \theta)}

\item \code{map = 3} (logistic map), \eqn{ 0 \le \theta \le 4},
\deqn{T(u) = \theta(1-\theta)}

\item \code{map = 4} (Manneville-Pomeau map), \eqn{0 < \theta < 1}
\deqn{T(u) = (u + u^{1+\theta})(\text{mod } 1)}

\item \code{map = 5} (Lasota-Mackey's map),
\deqn{T(u) = \dfrac{u}{(1 - u)}I(u \le 0.5) + (2u - 1)I(u > 0.5)}
}
}

\section{Link defaults}{
\code{linkh} and \code{configs.linkh} only apply to BARC models.\cr

\code{linkg} can be specified in one of two ways
\itemize{
\item \strong{Legacy structure:} a character or two-character vector. If
only one string is provided, the same link name is used for \code{g11} and \code{g12}.
Internally, this structure is automatically converted to the new format with
\code{g2 = g21 = g22 = g23 = "linear"}.

\item \strong{New structure:} a named list with optional elements (order is
irrelevant) \code{g11}, \code{g12}, \code{g2}, \code{g21}, \code{g22} and \code{g23}. These links apply,
respectively, to \eqn{\mu_t}, \eqn{Y_t} (in the AR recursion or part 1),
\eqn{\nu_t}, \eqn{\vartheta_t = g_2(\nu_t)}, \eqn{\vartheta_t} (in the AR
recursion of part 2) and \eqn{e_{1t}} (to build the error term in part 2).
}
For models that do not have the \eqn{\nu} parameter, the links \code{g2}, \code{g21},
\code{g22} and \code{g23} are set to \code{"linear"} for compatibility with Fortran
subroutines.

Missing entries in the \code{linkg} list follow these rules
\itemize{
\item If either \code{g11} or \code{g12} is missing (but not both), internally it is
set \code{g12 = g11}.

\item If both \code{g11} and \code{g12} are missing, use the default values for the
particular model (see below).

\item If \code{phi = NULL} for part 1, \code{g12} is not required, hence set to
\code{"linear"} and ignored in Fortran.

\item If \code{phi = NULL} for part 2, \code{g22} is not required, hence set to
\code{"linear"} and ignored in Fortran.

\item If either \code{g21} or \code{g22} is missing (but not both), internally it is
set \code{g22 = g21}.

\item If both \code{g21} and \code{g22} are missing, use the default values for the
particular model (see below).
}

Default \code{linkg} values are model-dependent (based on the string provided with
\code{model}):
\itemize{
\item For all models where \eqn{\nu} is constant over time:\cr
internally, \code{g2}, \code{g21}, and \code{g22} are forced to \code{"linear"}, with \eqn{a = 1}.\cr
\strong{Overrides any user specifications}.

\item iid samples:\cr
\strong{Overrides any user specifications}.\cr
\code{linkg = "linear"} (with \eqn{a = 1}). Internally converted to

\if{html}{\out{<div class="sourceCode">}}\preformatted{linkg = list(g11 = "linear", g12 = "linear", g2 = "linear"
             g21 = "linear", g22 = "linear", g23 = "linear")
}\if{html}{\out{</div>}}

\item BARFIMA, KARFIMA, ULARFIMA, UWARFIMA:\cr
\code{linkg = "logit"}. Internally converted to

\if{html}{\out{<div class="sourceCode">}}\preformatted{linkg = list(g11 = "logit", g12 = "logit", g2 = "linear"
             g21 = "linear", g22 = "linear", g23 = "linear")
}\if{html}{\out{</div>}}

\item GARFIMA:\cr
\code{linkg = "log"}. Internally converted to

\if{html}{\out{<div class="sourceCode">}}\preformatted{linkg = list(g11 = "log", g12 = "log", g2 = "linear"
             g21 = "linear", g22 = "linear", g23 = "linear")
}\if{html}{\out{</div>}}

\item MARFIMA:\cr
\code{linkg = "cloglog"}. Internally converted to

\if{html}{\out{<div class="sourceCode">}}\preformatted{linkg = list(g11 = "cloglog", g12 = "cloglog", g2 = "linear"
             g21 = "linear", g22 = "linear", g23 = "linear")
}\if{html}{\out{</div>}}

\item BARFIMAV, GARFIMAV, KARFIMAV, UWARFIMAV:

\code{g11} and \code{g12} have the same default values as the particular model where
\eqn{\nu} is constant over time.

\code{g2 = "default"}, meaning that \code{g2} is set as the the default link for
the model.
\itemize{
\item For BARFIMAV \code{"default" = SIP} with \eqn{a = b = 1}.
\item For GARFIMAV \code{"default" = SIP} with \eqn{a = 0} and \eqn{b = 1}.
\item For remaining models \code{"default" = "linear"} with \eqn{a = 1}.
}

\code{g21} depends on the model.
\itemize{
\item For BARFIMAV \code{g21 = "logit"}
\item For any other model \code{g21 = "log"}.
}

For \code{g22}, the default is to assume  \code{g22 = g21}.

Finally, \verb{g23 = "polynomial}, with \eqn{a = 1} and \eqn{b = 2} (set in
\code{configs.link})

\item Particular cases (e.g., BREG, BREGV) inherit defaults from parent
models (except iid samples).
}

\code{configs.linkg} if provided, it must be provided as a list with optional
elements, \code{ctt} and \code{power}, which define the constant \eqn{a} and the
exponent \eqn{b} in the link function \eqn{g(x) = a x^b}. Each element in
this list can be specified in one of two ways
\itemize{
\item \strong{Legacy structure:} a numeric value (applied uniformly across
all linear links) or a numeric vector of length 2, which will be associated
to \code{g11} and \code{g12}.

\item \strong{New structure:} a named list with optional elements (order is
irrelevant) \code{g11}, \code{g12}, \code{g2}, \code{g21}, \code{g22} and \code{g23}.
}
For now, the arguments \code{ctt} and \code{power} are only used when the link function
is \code{"linear"} or \code{"polynomial"}. If \code{NULL}, default is to assume that \code{ctt}
and \code{power} are both equal to 1 for all links.
}

\section{The log-likelihood}{
Let \eqn{\boldsymbol\gamma = (\boldsymbol \rho', \boldsymbol \lambda')'} be
the vector of unknown parameters in the model where
\itemize{
\item \eqn{\boldsymbol\rho} is the vector of unknown parameters in part 1
\item \eqn{\boldsymbol\lambda} is the vector of unknown parameters in part 2.
}
The \strong{log-likelihood function}, conditioned on a set of initial conditions
\eqn{\mathcal{F}_m} is given by
\deqn{
\ell(\boldsymbol\gamma) = \sum_{t = m+1}^n \ell_t =
 \displaystyle\sum_{t=m+1}^n\log\!\big(f(Y_t \mid \mathcal{F}_{t-1},
 \boldsymbol{\gamma})\big).
}

For simplicity of notation assume \eqn{m = 0}. The \strong{score vector}
\eqn{U(\boldsymbol\gamma) = \big(U_{\boldsymbol\rho}(\boldsymbol\gamma)',
U_{\boldsymbol\lambda}(\boldsymbol\gamma)'\big)'}
can be written as
\deqn{
U_{\boldsymbol\rho}(\boldsymbol\gamma) = D_{\boldsymbol\rho}' T_1\boldsymbol h_1 +
M_{\boldsymbol\rho}' T_2\boldsymbol h_2  \qquad \mbox{and} \qquad
U_{\boldsymbol\lambda}(\boldsymbol\gamma) = D_{\boldsymbol\lambda}' T_2\boldsymbol h_2,
}
where
\itemize{
\item \eqn{D_{\boldsymbol\rho}}, \eqn{D_{\boldsymbol\lambda}} and
\eqn{M_{\boldsymbol\rho}} are the matrices for which the \eqn{(i,j)}th
elements are given, respectively, by
\deqn{
  [D_{\boldsymbol\rho}]_{i,j} = \dfrac{\partial \eta_{1i}}{\partial \rho_j},
  \quad
  [D_{\boldsymbol\lambda}]_{i,j} =\dfrac{\partial \eta_{2i}}{\partial \lambda_j}
  \quad \mbox{and} \quad
  [M_{\boldsymbol\rho}]_{i,j} = \dfrac{\partial \eta_{2i}}{\partial \rho_j},
}

\item \eqn{T_1} and \eqn{T_2} are diagonal matrices given by
\deqn{
T_1 = \mathrm{diag}\bigg\{\dfrac{\partial \mu_1}{\partial \eta_{1t}},\dots,
\dfrac{\partial \mu_n}{\partial \eta_{1n}}\bigg\},
\quad
T_2 = \mathrm{diag}\bigg\{\dfrac{\partial \nu_1}{\partial \eta_{2t}},\dots,
\dfrac{\partial \nu_n}{\partial \eta_{2n}}\bigg\},
}

\item \eqn{\boldsymbol{h}_1} and \eqn{\boldsymbol{h}_2} are the vectors
defined by
\deqn{
 \boldsymbol{h}_1 = \bigg(\dfrac{\partial \ell_1}{\partial \mu_1}, \cdots,
 \dfrac{\partial \ell_n}{\partial \mu_n}\bigg)' \quad \mbox{and} \quad
 \boldsymbol{h}_2 = \bigg(\dfrac{\partial \ell_1}{\partial \nu_1}, \cdots,
 \dfrac{\partial \ell_n}{\partial \nu_n}\bigg)'.
}
}
For the models implemented so far, \eqn{\partial\eta_{1t}/\partial\lambda_j =
0} so that we don't need a matrix for these derivatives.

The \strong{conditional Fisher information matrix} for
\eqn{\boldsymbol\gamma} is given by
\deqn{
K_n(\boldsymbol\gamma) =
\begin{pmatrix}
K_{\boldsymbol\rho,\boldsymbol\rho} & K_{\boldsymbol\rho,\boldsymbol\lambda}\\
K_{\boldsymbol\lambda,\boldsymbol\rho}& K_{\boldsymbol\lambda,\boldsymbol\lambda}
\end{pmatrix}
}
with
\deqn{
\begin{aligned}
K_{\boldsymbol\rho,\boldsymbol\rho} &=
 D'_{\boldsymbol \rho}T_1E_\mu T_1 D_{\boldsymbol \rho}
 + M'_{\boldsymbol \rho}T_2E_{\mu\nu}T_1 D_{\boldsymbol \rho}
 + D'_{\boldsymbol \rho}T_1E_{\mu\nu} T_2 M_{\boldsymbol \rho}
 + M'_{\boldsymbol \rho}T_2 E_\nu T_2 M_{\boldsymbol \rho}\\
K_{\boldsymbol\rho,\boldsymbol\lambda} &= K_{\boldsymbol\lambda,\boldsymbol\rho}'=
 D_{\boldsymbol \rho}' T_1E_{\mu\nu}T_2D_{\boldsymbol \lambda}
 + M_{\boldsymbol \rho}' T_2 E_\nu T_2  D_{\boldsymbol \lambda},\\
K_{\boldsymbol\lambda,\boldsymbol\lambda} &=
 D_{\boldsymbol \lambda}' T_2E_\nu T_2D_{\boldsymbol \lambda}
\end{aligned}
}
where \eqn{E_\mu}, \eqn{E_{\mu\nu}} and  \eqn{E_\nu}
are diagonal matrices for which the \eqn{(t,t)}th element is given by
\deqn{
[E_\mu ]_{t,t} =
-\mathbb{E}\bigg(\dfrac{\partial^2 \ell_t}{\partial \mu_t^2} \bigg|
\mathcal{F} _{t-1} \bigg), \quad
[E_{\mu\nu}]_{t,t} =
-\mathbb{E}\bigg(\dfrac{\partial^2 \ell_t}{\partial\mu_t\partial \nu_t}
\bigg| \mathcal{F} _{t-1} \bigg) \quad \mbox{and} \quad
[E_\nu]_{t,t} =
- \mathbb{E}\bigg(\dfrac{\partial^2 \ell_t}{ \partial \nu_t^2} \bigg|
\mathcal{F} _{t-1} \bigg).
}
}

\references{
\insertAllCited{}
}
\seealso{
\link{BTSR.model.defaults}: function to print default settings for a
specified model
}
\keyword{internal}
