






#![no_std]

















































use core::mem::size_of;
use core::ptr::NonNull;



#[inline(always)]
pub fn ptrdistance<T>(start: *const T, end: *const T) -> usize {
    let size = size_of::<T>();
    if size == 0 {
        (end as usize).wrapping_sub(start as usize)
    } else {
        (end as usize - start as usize) / size
    }
}


pub trait PointerExt : Copy {
    unsafe fn offset(self, i: isize) -> Self;

    unsafe fn add(self, i: usize) -> Self {
        self.offset(i as isize)
    }

    unsafe fn sub(self, i: usize) -> Self {
        self.offset((i as isize).wrapping_neg())
    }




    #[inline(always)]
    unsafe fn pre_inc(&mut self) -> Self {
        *self = self.offset(1);
        *self
    }




    #[inline(always)]
    unsafe fn post_inc(&mut self) -> Self {
        let current = *self;
        *self = self.offset(1);
        current
    }




    #[inline(always)]
    unsafe fn pre_dec(&mut self) -> Self {
        *self = self.offset(-1);
        *self
    }




    #[inline(always)]
    unsafe fn post_dec(&mut self) -> Self {
        let current = *self;
        *self = self.offset(-1);
        current
    }


    #[inline(always)]
    unsafe fn inc(&mut self) {
        *self = self.offset(1);
    }


    #[inline(always)]
    unsafe fn dec(&mut self) {
        *self = self.offset(-1);
    }


    #[inline(always)]
    unsafe fn stride_offset(self, s: isize, index: usize) -> Self {
        self.offset(s * index as isize)
    }
}

impl<T> PointerExt for *const T {
    #[inline(always)]
    unsafe fn offset(self, i: isize) -> Self {
        self.offset(i)
    }


    #[inline]
    unsafe fn add(self, i: usize) -> Self {
        self.add(i)
    }

    #[inline]
    unsafe fn sub(self, i: usize) -> Self {
        self.sub(i)
    }
}

impl<T> PointerExt for *mut T {
    #[inline(always)]
    unsafe fn offset(self, i: isize) -> Self {
        self.offset(i)
    }

    #[inline]
    unsafe fn add(self, i: usize) -> Self {
        self.add(i)
    }

    #[inline]
    unsafe fn sub(self, i: usize) -> Self {
        self.sub(i)
    }
}








impl<T> PointerExt for NonNull<T> {
    #[inline(always)]
    unsafe fn offset(self, i: isize) -> Self {
        NonNull::new_unchecked(self.as_ptr().offset(i))
    }
}


#[cfg(test)]
mod tests {
    use super::PointerExt;
    use core::ptr::NonNull;

    #[test]
    fn it_works() {
        unsafe {
            let mut xs = [0; 16];
            let mut ptr = xs.as_mut_ptr();
            let end = ptr.offset(4);
            let mut i = 0;
            while ptr != end {
                *ptr.post_inc() = i;
                i += 1;
            }
            assert_eq!(&xs[..8], &[0, 1, 2, 3, 0, 0, 0, 0]);
        }
    }

    #[test]
    fn nonnull() {
        unsafe {
            let mut xs = [0; 16];
            let mut ptr = NonNull::new(xs.as_mut_ptr()).unwrap();
            let end = ptr.offset(4);
            let mut i = 0;
            while ptr != end {
                *ptr.post_inc().as_ptr() = i;
                i += 1;
            }
            assert_eq!(&xs[..8], &[0, 1, 2, 3, 0, 0, 0, 0]);
        }
    }

    #[test]
    fn nonnull_sub() {
        unsafe {

            let mut xs = [0; 16];
            let mut ptr = xs.as_mut_ptr().add(xs.len());
            let nptr = NonNull::new(xs.as_mut_ptr()).unwrap();
            let mut nend = nptr.add(xs.len());
            let mut i = 0;
            while nptr != nend {
                nend = nend.sub(1);
                ptr = ptr.sub(1);
                assert_eq!(nend.as_ptr(), ptr);
                *nend.as_ptr() = i;
                i += 1;
            }
            assert_eq!(&xs[..8], &[15, 14, 13, 12, 11, 10, 9, 8]);
        }
    }
}
