










use crate::Rng;
use core::iter;
#[cfg(feature = "alloc")]
use alloc::string::String;



















pub trait Distribution<T> {

    fn sample<R: Rng + ?Sized>(&self, rng: &mut R) -> T;



































    fn sample_iter<R>(self, rng: R) -> DistIter<Self, R, T>
    where
        R: Rng,
        Self: Sized,
    {
        DistIter {
            distr: self,
            rng,
            phantom: ::core::marker::PhantomData,
        }
    }


















    fn map<F, S>(self, func: F) -> DistMap<Self, F, T, S>
    where
        F: Fn(T) -> S,
        Self: Sized,
    {
        DistMap {
            distr: self,
            func,
            phantom: ::core::marker::PhantomData,
        }
    }
}

impl<'a, T, D: Distribution<T>> Distribution<T> for &'a D {
    fn sample<R: Rng + ?Sized>(&self, rng: &mut R) -> T {
        (*self).sample(rng)
    }
}








#[derive(Debug)]
pub struct DistIter<D, R, T> {
    distr: D,
    rng: R,
    phantom: ::core::marker::PhantomData<T>,
}

impl<D, R, T> Iterator for DistIter<D, R, T>
where
    D: Distribution<T>,
    R: Rng,
{
    type Item = T;

    #[inline(always)]
    fn next(&mut self) -> Option<T> {



        Some(self.distr.sample(&mut self.rng))
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        (usize::max_value(), None)
    }
}

impl<D, R, T> iter::FusedIterator for DistIter<D, R, T>
where
    D: Distribution<T>,
    R: Rng,
{
}

#[cfg(features = "nightly")]
impl<D, R, T> iter::TrustedLen for DistIter<D, R, T>
where
    D: Distribution<T>,
    R: Rng,
{
}






#[derive(Debug)]
pub struct DistMap<D, F, T, S> {
    distr: D,
    func: F,
    phantom: ::core::marker::PhantomData<fn(T) -> S>,
}

impl<D, F, T, S> Distribution<S> for DistMap<D, F, T, S>
where
    D: Distribution<T>,
    F: Fn(T) -> S,
{
    fn sample<R: Rng + ?Sized>(&self, rng: &mut R) -> S {
        (self.func)(self.distr.sample(rng))
    }
}





#[cfg(feature = "alloc")]
pub trait DistString {

    fn append_string<R: Rng + ?Sized>(&self, rng: &mut R, string: &mut String, len: usize);


    #[inline]
    fn sample_string<R: Rng + ?Sized>(&self, rng: &mut R, len: usize) -> String {
        let mut s = String::new();
        self.append_string(rng, &mut s, len);
        s
    }
}

#[cfg(test)]
mod tests {
    use crate::distributions::{Distribution, Uniform};
    use crate::Rng;

    #[test]
    fn test_distributions_iter() {
        use crate::distributions::Open01;
        let mut rng = crate::test::rng(210);
        let distr = Open01;
        let mut iter = Distribution::<f32>::sample_iter(distr, &mut rng);
        let mut sum: f32 = 0.;
        for _ in 0..100 {
            sum += iter.next().unwrap();
        }
        assert!(0. < sum && sum < 100.);
    }

    #[test]
    fn test_distributions_map() {
        let dist = Uniform::new_inclusive(0, 5).map(|val| val + 15);

        let mut rng = crate::test::rng(212);
        let val = dist.sample(&mut rng);
        assert!((15..=20).contains(&val));
    }

    #[test]
    fn test_make_an_iter() {
        fn ten_dice_rolls_other_than_five<R: Rng>(
            rng: &mut R,
        ) -> impl Iterator<Item = i32> + '_ {
            Uniform::new_inclusive(1, 6)
                .sample_iter(rng)
                .filter(|x| *x != 5)
                .take(10)
        }

        let mut rng = crate::test::rng(211);
        let mut count = 0;
        for val in ten_dice_rolls_other_than_five(&mut rng) {
            assert!((1..=6).contains(&val) && val != 5);
            count += 1;
        }
        assert_eq!(count, 10);
    }

    #[test]
    #[cfg(feature = "alloc")]
    fn test_dist_string() {
        use core::str;
        use crate::distributions::{Alphanumeric, DistString, Standard};
        let mut rng = crate::test::rng(213);

        let s1 = Alphanumeric.sample_string(&mut rng, 20);
        assert_eq!(s1.len(), 20);
        assert_eq!(str::from_utf8(s1.as_bytes()), Ok(s1.as_str()));

        let s2 = Standard.sample_string(&mut rng, 20);
        assert_eq!(s2.chars().count(), 20);
        assert_eq!(str::from_utf8(s2.as_bytes()), Ok(s2.as_str()));
    }
}
