\name{hr.mcp}
\alias{hr.mcp}
\title{
Minimum convex polygon home ranges
}
\description{
Computes and (optionally) plots a minimum convex polygon home range.  Includes provisions for excluding points and polygon clipping.
}
\usage{
hr.mcp(x, y=NULL, n.min=50, method=c("median","mean"), maxiter=1000, id=NA, domain=NA, exclude=0.00, clip.to=NA, plot=TRUE,...)}
\arguments{
  \item{x}{
A matrix or data frame (default), or a numeric vector of coordinates
}
  \item{y}{
Either \code{NULL} (default) or a numeric vector of coordinates
}
  \item{n.min}{
Minimum number of qualifying locations required for an estimate.  If fewer than \code{n.min} locations qualify, the function will return \code{NA} and a warning.
}
  \item{method}{
Character string specifying the method used to identify outlying points.  Default is to use distances from the spatial median of points (\code{"median"}).  Distance from arithmetic means of coordinates (barycenter) may also be specified (\code{"mean"}).
}
\item{maxiter}{
Maximum number of iterations for calculation of spatial median
}
  \item{id}{
Optional identifier, e.g., for individual animal
}
  \item{domain}{
Either a matrix of coordinates, a list, or an object of class \code{"gpc.poly"}.  Controls inclusion of data in calculations.  
}
  \item{exclude}{
Proportion of outlying points to exclude from calculations.
}
  \item{clip.to}{
Either a matrix of coordinates, a list, or an object of class \code{"gpc.poly"}.  Controls clipping of home ranges.
}
  \item{plot}{
Produce a diagnostic plot?  Default is \code{TRUE}.
}
  \item{\dots}{
Not implemented
}
}
\details{
This function was motivated primarily by a need to incorporate "hard" boundaries and "holes" into home-range estimates.  For example, one might wish to 1) restrict interest to the portion of a home range encompassed by a national park, or 2) exclude spurious locations of an island dwelling mammal "found" at sea.  The function also features a robust method for identifying outlying locations and relies on simple data structures that integrate well with other programming tools. 

If \code{domain} is specified, only data that fall within polygons that are not "holes" will be included in home range calculations.  At present, \code{domain} may be an object of class \code{"gpc.poly"}, a matrix of coordinates describing a polygon, or a list with 1 component for each polygon and 1 component for each hole.  If list input is used, each component of the list must contain 3 elements named "x" (a numeric vector of coordinates), "y" (another numeric vector of coordinates), and "hole."  See \code{\link{gpc.poly-class}}.

If \code{clip.to} is specified, the home range returned will be the intersection of the home range with the area described by \code{clip.to}.  As for \code{domain}, the area described by \code{clip.to} may consist of any number of polygons and "holes."  At present, \code{clip.to} may be an object of class \code{"gpc.poly"}, a matrix of coordinates describing a polygon, or a list with 1 component for each polygon and 1 component for each hole.  If list input is used, each component of the list must contain 3 elements named "x" (a numeric vector of coordinates), "y" (another numeric vector of coordinates), and "hole."  See \code{\link{gpc.poly-class}}.

Using the spatial median to identify outlying points is robust to effects of extreme outliers but computationally slow and requires package \code{ICSNP}.  Using the location described by arithmetic means of coordinates is less robust but faster.  

If \code{spatial.median} fails to converge, try increasing the number of iterations.  Some data used to test \code{hr.mcp} required 1000-2000 iterations.

Order of operations: 1) Data that are not within the domain are dropped, 2) points furthest from the spatial median or barycenter are excluded, 3) the resulting home range is clipped.

All points are shown in plots.  Points excluded by \code{domain} are shown in light gray.

Plots generated by \code{\link{hr.mcp}} are intended for diagnostic use. A plotting method provided for objects of class \code{"hr.mcp"} (\code{\link{plot.hr.mcp}}) can be used in conjunction with other plotting tools to provide complete control.
}
\value{
Returns a list of class \code{hr.mcp} with the following components:
  \item{id }{Optional identifier}
  \item{exclude }{Proportion of points excluded }
  \item{n}{Sample sizes; total number of points, number within domain, number excluded, number used in calculations}
  \item{mcp}{An object of class \code{gpc.poly}}
  \item{area}{Area of home range polygon(s), not including area of holes}
}
\author{
Glen A. Sargeant\cr
U.S. Geological Survey\cr
Northern Prairie Wildlife Research Center\cr
\email{glen_sargeant@usgs.gov}
}
\note{
%%  ~~further notes~~
}

\section{Warning }{
Polygons should not overlap.  Holes should not traverse polygon boundaries.  These issues can be addressed with functions in package \pkg{gpclib}, which can be used to find intersections, unions, and differences of polygons.

Use good judgement when clipping data or home ranges so results support biologically sensible interpretations!
}

\seealso{
\code{\link{plot.hr.mcp}}
}
\examples{

#Matrix of locations
data(xy.elk)

#Boundary of study area
data(wica.gpc)
str(wica.gpc)

#A "hole" in the study area
data(hole.gpc)
str(hole.gpc)

#Study area with hole
data(wica.hole.gpc)
str(wica.hole.gpc)

par(mfrow=c(2,2))

#No-frills 95% mcp
hr.mcp(x=xy.elk, exclude=0.05)

#MCP computed from all locations,
#then clipped
hr.mcp(x=xy.elk,clip.to=wica.gpc)

#Area
a.park <- hr.mcp(x=xy.elk,clip.to=wica.gpc)$area

#Area, but now clipped to study area
#and with hole clipped out
a.clip <- hr.mcp(x=xy.elk,clip.to=wica.hole.gpc)$area

#Area of hole; note that it is negative
a.hole <- area.poly(hole.gpc)

(a.park+a.hole)==a.clip

#Home range
#Domain restricted to exclude locations outside
#study area and within a "hole;" result clipped
hr.mcp(x=xy.elk, domain=wica.hole.gpc, clip.to=wica.hole.gpc)
}