% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_KM.R
\name{estimate_KM}
\alias{estimate_KM}
\title{Wrapper for Kaplan Meier Time-to-Event analysis}
\usage{
estimate_KM(data = NULL, strata = NULL, CNSR = "CNSR", AVAL = "AVAL", ...)
}
\arguments{
\item{data}{The name of the dataset for Time-to-Event analysis based on the Analysis Data Model (ADaM) principles. The dataset is expected to have
one record per subject per analysis parameter. Rows in which the analysis variable (AVAL) or the sensor variable (CNSR) contain NA, are removed during analysis.}

\item{strata}{Character vector, representing the strata for Time-to-Event analysis. When NULL, an overall analysis is performed.
Default is NULL.}

\item{CNSR}{Censor for Time-to-Event analysis. Default is "CNSR", as per CDISC ADaM guiding principles.}

\item{AVAL}{Analysis value for Time-to-Event analysis. Default is "AVAL", as per CDISC ADaM guiding principles.}

\item{...}{additional arguments passed on to the ellipsis of the call \code{survival::survfit.formula(data = data, formula = Surv(AVAL, 1-CNSR) ~ strata), ...)} .
Use \code{?survival::survfit.formula} and \code{?survival::survfitCI} for more information.}
}
\value{
survfit object, extended by elements PARAM/PARAMCD, ready for downstream processing in estimation or visualization functions and methods.
}
\description{
This function is a wrapper around \code{survival::survfit.formula} to perform a Kaplan-Meier analysis, assuming right-censored data.
The function expects that the data has been filtered on the parameter (PARAM/PARAMCD) of interest.
Alternatively, PARAM/PARAMCD can be used in the \code{strata} argument. \cr
The result is an object of class \code{survfit} which can be used in downstream functions and methods that rely on the \code{survfit} class.
By default:
\itemize{
\item{The Kaplan Meier estimate is estimated directly (stype = 1).}
\item{The cumulative hazard is estimated using the Nelson-Aalen estimator (ctype = 1): H.tilde = cumsum(x$n.event/x$n.risk).
The MLE (H.hat(t) = -log(S.hat(t))) can't be requested.}
\item{A two-sided pointwise 0.95 confidence interval is estimated using a log transformation (conf.type = "log").}
}
}
\examples{

## No stratification
visR::estimate_KM(data = adtte)

## Stratified Kaplan-Meier analysis by `TRTP`
visR::estimate_KM(data = adtte, strata = "TRTP")

## Stratified Kaplan-Meier analysis by `TRTP` and `SEX`
visR::estimate_KM(data = adtte, strata = c("TRTP", "SEX"))

## Stratification with one level
visR::estimate_KM(data = adtte, strata = "PARAMCD")

## Analysis on subset of adtte
visR::estimate_KM(data = adtte[adtte$SEX == "F", ])

## Modify the default analysis by using the ellipsis
visR::estimate_KM(data = adtte, strata = NULL, 
  type = "kaplan-meier", conf.int = FALSE, timefix = TRUE)

## Example working with non CDISC data
head(survival::veteran)

# convert time and censoring data to ADaM variables 
# convert censoring status to CDISC principles
veteran_adam <- survival::veteran \%>\%
 dplyr::mutate(AVAL = time, 
               CNSR = dplyr::if_else(status == 1, 0, 1) 
 )

visR::estimate_KM(data = veteran_adam, strata = "trt")
}
\references{
\url{https://github.com/therneau/survival}
}
\seealso{
\code{\link[survival]{survfit.formula} \link[survival]{survfitCI}}
}
