#' run.vardict.sample
#'
#' @description
#' Run VarDict on a sample. Idea: have a low-level function that simply submits job to Perl, after BAM paths have been found. 
#' and output paths already have been decided upon
#'
#' @param tumour.bam 
#'	Path to tumour sample BAM file.
#' @param sample.id
#'	Sample ID for labelling
#' @param paired 
#'	Logical indicating whether to do variant calling with a matched normal.
#' @param proton 
#'	Logical indicating whether the data was generated by proton sequencing. Defaults to False (i.e. Illumina)
#' @param normal.bam
#' 	Path to normal BAM file if \code{paired = TRUE}
#' @param output.directory 
#'	Path to output directory
#' @param output.filename 
#'	Name of resulting VCF file (defaults to SAMPLE_ID.vcf)
#' @param code.directory 
#'	Path to directory where code should be stored
#' @param log.directory 
#'	Path to directory where log files should be stored
#' @param config.file 
#'	Path to config file
#' @param job.dependencies 
#'	Vector with names of job dependencies
#' @param job.name 
#'	Name of job to be submitted
#' @param job.group 
#'	Group job should belong to
#' @param verify.options 
#'	Logical indicating whether to run verify.varitas.options
#' @param quiet 
#'	Logical indicating whether to print command to screen rather than submit it to the system. Defaults to false, useful for debugging.
#'
#'
#' 
#'
run.vardict.sample <- function(
	tumour.bam,
	sample.id,
	paired,
	proton = FALSE,
	normal.bam = NULL, 
	output.directory = NULL,
	output.filename = NULL,
	code.directory = NULL,
	log.directory = NULL,
	config.file = NULL,
	job.dependencies = NULL,
	job.name = NULL,
	job.group = NULL,
	quiet = FALSE, 
	verify.options = !quiet
	) {
	
	### INPUT TESTS ###########################################################

	if(paired && is.null(normal.bam)) {
		stop('paired is set to true but no normal sample BAM has been supplied.');
	}	
	
	if(!paired && !is.null(normal.bam)) {
		stop('Getting mixed signals: paired is set to false but a normal sample BAM has been supplied.');
	}
	
	### MAIN ##################################################################
		
	if( verify.options ) {
		verify.varitas.options(
		  varitas.options = config.file, 
			stages.to.run = 'calling', 
			variant.callers = 'vardict'
			);
	}



	if(is.null(config.file)) {
		config.file <- save.config();
	}

	# sort out flags
	flags <- c();
	
	if(paired) {
		flags <- c('paired', flags);
	}
	
	if(proton) { 
		flags <- c('proton', flags);
	}
	

	script <- system.file('perl', 'run_vardict.pl', package = get.varitas.options('pkgname') );

	command <- make.command.line.call(
		main.command = c('perl', script), 
		options = c(
			'tumour_bam' = tumour.bam,
			'normal_bam' = normal.bam,
			'sample_id' = sample.id,
			'config_file' = config.file,
			'output_directory' = output.directory,
			'output_filename' = output.filename,
			'code_directory' = code.directory,
			'log_directory' = log.directory,
			'job_dependencies' = job.dependencies,
			'job_name' = job.name, 
			'job_group' = job.group
			),
		flags = flags
		);

	if(quiet) { 
	    cat(command, '\n');
	} else { 
		system(command);
	}
		
}

