% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_run_modify.R
\docType{package}
\name{umx}
\alias{umx}
\title{Functions for Structural Equation Modeling in OpenMx}
\description{
\code{umx} allows you to more easily build, run, modify, and report structural models,
building on the OpenMx package.
All core functions are organized into families, so they are easier to find
(see "families" below under \strong{See Also})

All the functions have full-featured and well commented examples, some even have \emph{figures},
so use the help, even if you think it won't help :-)
Have a look, for example at \code{\link[=umxRAM]{umxRAM()}}

Check out NEWS about new features at \code{news(package = "umx")}
}
\details{
Introductory working examples are below. You can run all demos with demo(umx)
When I have a vignette, it will be: vignette("umx", package = "umx")

There is a helpful blog at \url{https://tbates.github.io}

(Only) if you want the bleeding-edge version:

devtools::install_github("tbates/umx")
}
\examples{
require("umx")
data(demoOneFactor)
manifests = names(demoOneFactor)
m1 = umxRAM("One Factor", data = demoOneFactor, type="cov",
	umxPath("G", to = manifests),
	umxPath(var = manifests),
	umxPath(var = "G"  , fixedAt= 1)
)

# umx added informative labels, created starting values, 
# Ran your model (if autoRun is on), and displayed a brief summary
# including a comparison if you modified a model...!

# umxSummary generates journal-ready fit information.
# We can choose std=T for standardized parameters and can also
# filter out some types of parameter (e.g. means or residuals)

umxSummary(m1, std = TRUE, residuals=FALSE)

# parameters() flexibly retrieves model coefficients. 
# For example just G-loadings greater than |.3| and rounded to 2-digits.
parameters(m1, thresh="above", b=.3, pattern = "G_to.*", digits = 2)

# (The built-in coef works as for lm etc.)
coef(m1)

# ==================
# = Model updating =
# ==================
# umxModify modifies, renames, re-runs, and compares a model
# Can we set the loading of x1 on G to zero? (nope...)
m2 = umxModify(m1, "G_to_x1", name = "no_effect_of_g_on_X1", comparison = TRUE)

# note1: umxSetParameters can do this with some additional flexibility
# note2 "comparison = TRUE" above is the same as calling 
# umxCompare, like this
umxCompare(m1, m2)


# ========================
# = Confidence intervals =
# ========================

# umxSummary() will show these, but you can also use the confint() function
confint(m1) # OpenMx's SE-based confidence intervals


\dontrun{
# umxConfint formats everything you need nicely, and allows adding CIs (with parm=)
umxConfint(m1, parm = 'all', run = TRUE) # likelihood-based CIs

# And make a Figure and open in browser
plot(m1, std = TRUE)

# If you just want the .dot code returned set file = NA
plot(m1, std = TRUE, file = NA)
}

}
\references{
\itemize{
\item \url{https://github.com/tbates/umx}
}
}
\seealso{
Other Teaching and testing Functions: 
\code{\link{tmx_genotypic_effect}()},
\code{\link{tmx_is.identified}()}

Other Core Model Building Functions: 
\code{\link{umxMatrix}()},
\code{\link{umxModify}()},
\code{\link{umxPath}()},
\code{\link{umxRAM}()},
\code{\link{umxSuperModel}()}

Other Model Summary and Comparison: 
\code{\link{umxCompare}()},
\code{\link{umxEquate}()},
\code{\link{umxMI}()},
\code{\link{umxReduce}()},
\code{\link{umxSetParameters}()},
\code{\link{umxSummary}()}

Other Reporting Functions: 
\code{\link{umxAPA}()},
\code{\link{umxFactorScores}()},
\code{\link{umxGetParameters}()},
\code{\link{umxParameters}()},
\code{\link{umx_aggregate}()},
\code{\link{umx_time}()}

Other Plotting functions: 
\code{\link{plot.MxLISRELModel}()},
\code{\link{plot.MxModelTwinMaker}()},
\code{\link{plot.MxModel}()},
\code{\link{umxPlotACEcov}()},
\code{\link{umxPlotACEv}()},
\code{\link{umxPlotACE}()},
\code{\link{umxPlotCP}()},
\code{\link{umxPlotDoC}()},
\code{\link{umxPlotFun}()},
\code{\link{umxPlotGxEbiv}()},
\code{\link{umxPlotGxE}()},
\code{\link{umxPlotIP}()},
\code{\link{umxPlotSexLim}()},
\code{\link{umxPlotSimplex}()}

Other Super-easy helpers: 
\code{\link{umxEFA}()},
\code{\link{umxMendelianRandomization}()}

Other Twin Modeling Functions: 
\code{\link{power.ACE.test}()},
\code{\link{umxACEcov}()},
\code{\link{umxACEv}()},
\code{\link{umxACE}()},
\code{\link{umxCP}()},
\code{\link{umxDoCp}()},
\code{\link{umxDoC}()},
\code{\link{umxGxE_window}()},
\code{\link{umxGxEbiv}()},
\code{\link{umxGxE}()},
\code{\link{umxIP}()},
\code{\link{umxReduceACE}()},
\code{\link{umxReduceGxE}()},
\code{\link{umxReduce}()},
\code{\link{umxRotate.MxModelCP}()},
\code{\link{umxSexLim}()},
\code{\link{umxSimplex}()},
\code{\link{umxSummarizeTwinData}()},
\code{\link{umxSummaryACEv}()},
\code{\link{umxSummaryACE}()},
\code{\link{umxSummaryDoC}()},
\code{\link{umxSummaryGxEbiv}()},
\code{\link{umxSummarySexLim}()},
\code{\link{umxSummarySimplex}()},
\code{\link{umxTwinMaker}()}

Other Twin Data functions: 
\code{\link{umx_long2wide}()},
\code{\link{umx_make_TwinData}()},
\code{\link{umx_make_twin_data_nice}()},
\code{\link{umx_residualize}()},
\code{\link{umx_scale_wide_twin_data}()},
\code{\link{umx_wide2long}()}

Other Get and set: 
\code{\link{umx_get_checkpoint}()},
\code{\link{umx_get_options}()},
\code{\link{umx_set_auto_plot}()},
\code{\link{umx_set_auto_run}()},
\code{\link{umx_set_checkpoint}()},
\code{\link{umx_set_condensed_slots}()},
\code{\link{umx_set_cores}()},
\code{\link{umx_set_data_variance_check}()},
\code{\link{umx_set_optimization_options}()},
\code{\link{umx_set_optimizer}()},
\code{\link{umx_set_plot_file_suffix}()},
\code{\link{umx_set_plot_format}()},
\code{\link{umx_set_plot_use_hrbrthemes}()},
\code{\link{umx_set_separator}()},
\code{\link{umx_set_silent}()},
\code{\link{umx_set_table_format}()}

Other Check or test: 
\code{\link{umx_check_names}()},
\code{\link{umx_is_class}()},
\code{\link{umx_is_endogenous}()},
\code{\link{umx_is_exogenous}()},
\code{\link{umx_is_numeric}()},
\code{\link{umx_is_ordered}()}

Other Data Functions: 
\code{\link{umxFactor}()},
\code{\link{umxHetCor}()},
\code{\link{umx_as_numeric}()},
\code{\link{umx_cont_2_quantiles}()},
\code{\link{umx_lower2full}()},
\code{\link{umx_make_MR_data}()},
\code{\link{umx_make_TwinData}()},
\code{\link{umx_make_fake_data}()},
\code{\link{umx_make_raw_from_cov}()},
\code{\link{umx_polychoric}()},
\code{\link{umx_polypairwise}()},
\code{\link{umx_polytriowise}()},
\code{\link{umx_read_lower}()},
\code{\link{umx_read_prolific_demog}()},
\code{\link{umx_rename}()},
\code{\link{umx_reorder}()},
\code{\link{umx_score_scale}()},
\code{\link{umx_select_valid}()},
\code{\link{umx_stack}()}

Other File Functions: 
\code{\link{dl_from_dropbox}()},
\code{\link{umx_file_load_pseudo}()},
\code{\link{umx_make_sql_from_excel}()},
\code{\link{umx_move_file}()},
\code{\link{umx_open}()},
\code{\link{umx_rename_file}()},
\code{\link{umx_write_to_clipboard}()}

Other String Functions: 
\code{\link{umx_explode_twin_names}()},
\code{\link{umx_explode}()},
\code{\link{umx_grep}()},
\code{\link{umx_names}()},
\code{\link{umx_paste_names}()},
\code{\link{umx_rot}()},
\code{\link{umx_str_chars}()},
\code{\link{umx_str_from_object}()},
\code{\link{umx_trim}()}

Other Miscellaneous Stats Helpers: 
\code{\link{FishersMethod}()},
\code{\link{SE_from_p}()},
\code{\link{geometric_mean}()},
\code{\link{harmonic_mean}()},
\code{\link{oddsratio}()},
\code{\link{reliability}()},
\code{\link{umxCov2cor}()},
\code{\link{umxHetCor}()},
\code{\link{umxWeightedAIC}()},
\code{\link{umx_apply}()},
\code{\link{umx_cor}()},
\code{\link{umx_means}()},
\code{\link{umx_r_test}()},
\code{\link{umx_round}()},
\code{\link{umx_scale}()},
\code{\link{umx_var}()}

Other Miscellaneous Utility Functions: 
\code{\link{install.OpenMx}()},
\code{\link{libs}()},
\code{\link{qm}()},
\code{\link{umxLav2RAM}()},
\code{\link{umxModelNames}()},
\code{\link{umxRAM2Lav}()},
\code{\link{umxVersion}()},
\code{\link{umx_array_shift}()},
\code{\link{umx_find_object}()},
\code{\link{umx_lower.tri}()},
\code{\link{umx_msg}()},
\code{\link{umx_open_CRAN_page}()},
\code{\link{umx_pad}()},
\code{\link{umx_print}()}

Other datasets: 
\code{\link{Fischbein_wt}},
\code{\link{GFF}},
\code{\link{docData}},
\code{\link{iqdat}},
\code{\link{us_skinfold_data}}

Other Advanced Model Building Functions: 
\code{\link{umxAlgebra}()},
\code{\link{umxFixAll}()},
\code{\link{umxJiggle}()},
\code{\link{umxRun}()},
\code{\link{umxThresholdMatrix}()},
\code{\link{umxUnexplainedCausalNexus}()},
\code{\link{xmuLabel}()},
\code{\link{xmuValues}()}

Other xmu internal not for end user: 
\code{\link{umxModel}()},
\code{\link{umxRenameMatrix}()},
\code{\link{umx_APA_pval}()},
\code{\link{umx_fun_mean_sd}()},
\code{\link{umx_get_bracket_addresses}()},
\code{\link{umx_make}()},
\code{\link{umx_standardize}()},
\code{\link{umx_string_to_algebra}()},
\code{\link{xmuHasSquareBrackets}()},
\code{\link{xmuLabel_MATRIX_Model}()},
\code{\link{xmuLabel_Matrix}()},
\code{\link{xmuLabel_RAM_Model}()},
\code{\link{xmuMI}()},
\code{\link{xmuMakeDeviationThresholdsMatrices}()},
\code{\link{xmuMakeOneHeadedPathsFromPathList}()},
\code{\link{xmuMakeTwoHeadedPathsFromPathList}()},
\code{\link{xmuMaxLevels}()},
\code{\link{xmuMinLevels}()},
\code{\link{xmuPropagateLabels}()},
\code{\link{xmuRAM2Ordinal}()},
\code{\link{xmuTwinSuper_Continuous}()},
\code{\link{xmuTwinSuper_NoBinary}()},
\code{\link{xmuTwinUpgradeMeansToCovariateModel}()},
\code{\link{xmu_CI_merge}()},
\code{\link{xmu_CI_stash}()},
\code{\link{xmu_DF_to_mxData_TypeCov}()},
\code{\link{xmu_PadAndPruneForDefVars}()},
\code{\link{xmu_bracket_address2rclabel}()},
\code{\link{xmu_cell_is_on}()},
\code{\link{xmu_check_levels_identical}()},
\code{\link{xmu_check_needs_means}()},
\code{\link{xmu_check_variance}()},
\code{\link{xmu_clean_label}()},
\code{\link{xmu_data_missing}()},
\code{\link{xmu_data_swap_a_block}()},
\code{\link{xmu_describe_data_WLS}()},
\code{\link{xmu_dot_make_paths}()},
\code{\link{xmu_dot_make_residuals}()},
\code{\link{xmu_dot_maker}()},
\code{\link{xmu_dot_move_ranks}()},
\code{\link{xmu_dot_rank_str}()},
\code{\link{xmu_extract_column}()},
\code{\link{xmu_get_CI}()},
\code{\link{xmu_lavaan_process_group}()},
\code{\link{xmu_make_TwinSuperModel}()},
\code{\link{xmu_make_bin_cont_pair_data}()},
\code{\link{xmu_make_mxData}()},
\code{\link{xmu_match.arg}()},
\code{\link{xmu_name_from_lavaan_str}()},
\code{\link{xmu_path2twin}()},
\code{\link{xmu_path_regex}()},
\code{\link{xmu_print_algebras}()},
\code{\link{xmu_rclabel_2_bracket_address}()},
\code{\link{xmu_safe_run_summary}()},
\code{\link{xmu_set_sep_from_suffix}()},
\code{\link{xmu_show_fit_or_comparison}()},
\code{\link{xmu_simplex_corner}()},
\code{\link{xmu_standardize_ACEcov}()},
\code{\link{xmu_standardize_ACEv}()},
\code{\link{xmu_standardize_ACE}()},
\code{\link{xmu_standardize_CP}()},
\code{\link{xmu_standardize_IP}()},
\code{\link{xmu_standardize_RAM}()},
\code{\link{xmu_standardize_SexLim}()},
\code{\link{xmu_standardize_Simplex}()},
\code{\link{xmu_start_value_list}()},
\code{\link{xmu_starts}()},
\code{\link{xmu_summary_RAM_group_parameters}()},
\code{\link{xmu_twin_add_WeightMatrices}()},
\code{\link{xmu_twin_check}()},
\code{\link{xmu_twin_get_var_names}()},
\code{\link{xmu_twin_make_def_means_mats_and_alg}()},
\code{\link{xmu_twin_upgrade_selDvs2SelVars}()}
}
\concept{Advanced Model Building Functions}
\concept{Check or test}
\concept{Core Model Building Functions}
\concept{Data Functions}
\concept{File Functions}
\concept{Get and set}
\concept{Miscellaneous Stats Helpers}
\concept{Miscellaneous Utility Functions}
\concept{Model Summary and Comparison}
\concept{Plotting functions}
\concept{Reporting Functions}
\concept{String Functions}
\concept{Summary Functions}
\concept{Super-easy helpers}
\concept{Teaching and testing Functions}
\concept{Twin Data functions}
\concept{Twin Modeling Functions}
\concept{datasets}
\concept{xmu internal not for end user}
\concept{zAdvanced Helpers}
