% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tsesimp.R
\name{tsesimp}
\alias{tsesimp}
\title{The Simple Two-Stage Estimation (TSE) Method for Treatment
Switching}
\usage{
tsesimp(
  data,
  id = "id",
  stratum = "",
  time = "time",
  event = "event",
  treat = "treat",
  censor_time = "censor_time",
  pd = "pd",
  pd_time = "pd_time",
  swtrt = "swtrt",
  swtrt_time = "swtrt_time",
  base_cov = "",
  base2_cov = "",
  aft_dist = "weibull",
  strata_main_effect_only = TRUE,
  recensor = TRUE,
  admin_recensor_only = TRUE,
  swtrt_control_only = TRUE,
  alpha = 0.05,
  ties = "efron",
  offset = 1,
  boot = TRUE,
  n_boot = 1000,
  seed = NA
)
}
\arguments{
\item{data}{The input data frame that contains the following variables:
\itemize{
\item \code{id}: The subject id.
\item \code{stratum}: The stratum.
\item \code{time}: The survival time for right censored data.
\item \code{event}: The event indicator, 1=event, 0=no event.
\item \code{treat}: The randomized treatment indicator, 1=treatment,
0=control.
\item \code{censor_time}: The administrative censoring time. It should
be provided for all subjects including those who had events.
\item \code{pd}: The disease progression indicator, 1=PD, 0=no PD.
\item \code{pd_time}: The time from randomization to PD.
\item \code{swtrt}: The treatment switch indicator, 1=switch, 0=no switch.
\item \code{swtrt_time}: The time from randomization to treatment switch.
\item \code{base_cov}: The baseline covariates (excluding treat).
\item \code{base2_cov}: The baseline and secondary baseline
covariates (excluding swtrt).
}}

\item{id}{The name of the id variable in the input data.}

\item{stratum}{The name(s) of the stratum variable(s) in the input data.}

\item{time}{The name of the time variable in the input data.}

\item{event}{The name of the event variable in the input data.}

\item{treat}{The name of the treatment variable in the input data.}

\item{censor_time}{The name of the censor_time variable in the input data.}

\item{pd}{The name of the pd variable in the input data.}

\item{pd_time}{The name of the pd_time variable in the input data.}

\item{swtrt}{The name of the swtrt variable in the input data.}

\item{swtrt_time}{The name of the swtrt_time variable in the input data.}

\item{base_cov}{The names of baseline covariates (excluding
treat) in the input data for the outcome Cox model.}

\item{base2_cov}{The names of secondary baseline covariates
(excluding swtrt) in the input data for the AFT model for
post-progression survival.}

\item{aft_dist}{The assumed distribution for time to event for the AFT
model. Options include "exponential", "weibull" (default),
"loglogistic", and "lognormal".}

\item{strata_main_effect_only}{Whether to only include the strata main
effects in the AFT model. Defaults to \code{TRUE}, otherwise all
possible strata combinations will be considered in the AFT model.}

\item{recensor}{Whether to apply recensoring to counterfactual
survival times. Defaults to \code{TRUE}.}

\item{admin_recensor_only}{Whether to apply recensoring to administrative
censoring times only. Defaults to \code{TRUE}. If \code{FALSE},
recensoring will be applied to the actual censoring times for dropouts.}

\item{swtrt_control_only}{Whether treatment switching occurred only in
the control group. The default is \code{TRUE}.}

\item{alpha}{The significance level to calculate confidence intervals.
The default value is 0.05.}

\item{ties}{The method for handling ties in the Cox model, either
"breslow" or "efron" (default).}

\item{offset}{The offset to calculate the time to event, PD, and
treatment switch. We can set \code{offset} equal to 1 (default),
1/30.4375, or 1/365.25 if the time unit is day, month, or year.}

\item{boot}{Whether to use bootstrap to obtain the confidence
interval for hazard ratio. Defaults to \code{TRUE}.}

\item{n_boot}{The number of bootstrap samples.}

\item{seed}{The seed to reproduce the bootstrap results. The default is
missing, in which case, the seed from the environment will be used.}
}
\value{
A list with the following components:
\itemize{
\item \code{psi}: The estimated causal parameter for the control group.
\item \code{psi_CI}: The confidence interval for \code{psi}.
\item \code{psi_CI_type}: The type of confidence interval for \code{psi},
i.e., "AFT model" or "bootstrap".
\item \code{logrank_pvalue}: The two-sided p-value of the log-rank test
for an intention-to-treat (ITT) analysis.
\item \code{cox_pvalue}: The two-sided p-value for treatment effect based on
the Cox model.
\item \code{hr}: The estimated hazard ratio from the Cox model.
\item \code{hr_CI}: The confidence interval for hazard ratio.
\item \code{hr_CI_type}: The type of confidence interval for hazard ratio,
either "Cox model" or "bootstrap".
\item \code{data_aft}: A list of input data for the AFT model by treatment
group.
\item \code{fit_aft}: A list of fitted AFT models by treatment group.
\item \code{data_outcome}: The input data for the outcome Cox model.
\item \code{fit_outcome}: The fitted outcome Cox model.
\item \code{settings}: A list with the following components:
\itemize{
\item \code{aft_dist}: The distribution for time to event for the AFT
model.
\item \code{strata_main_effect_only}: Whether to only include the strata
main effects in the AFT model.
\item \code{recensor}: Whether to apply recensoring to counterfactual
survival times.
\item \code{admin_recensor_only}: Whether to apply recensoring to
administrative censoring times only.
\item \code{swtrt_control_only}: Whether treatment switching occurred
only in the control group.
\item \code{alpha}: The significance level to calculate confidence
intervals.
\item \code{ties}: The method for handling ties in the Cox model.
\item \code{offset}: The offset to calculate the time to event, PD, and
treatment switch.
\item \code{boot}: Whether to use bootstrap to obtain the confidence
interval for hazard ratio.
\item \code{n_boot}: The number of bootstrap samples.
\item \code{seed}: The seed to reproduce the bootstrap results.
}
\item \code{psi_trt}: The estimated causal parameter for the experimental
group if \code{swtrt_control_only} is \code{FALSE}.
\item \code{psi_trt_CI}: The confidence interval for \code{psi_trt} if
\code{swtrt_control_only} is \code{FALSE}.
\item \code{hr_boots}: The bootstrap hazard ratio estimates if \code{boot} is
\code{TRUE}.
\item \code{psi_boots}: The bootstrap \code{psi} estimates if \code{boot} is
\code{TRUE}.
\item \code{psi_trt_boots}: The bootstrap \code{psi_trt} estimates if
\code{boot} is \code{TRUE} and \code{swtrt_control_only} is
\code{FALSE}.
}
}
\description{
Obtains the causal parameter estimate of the AFT model
for switching after disease progression and the hazard ratio estimate
of the outcome Cox model to adjust for treatment switching.
}
\details{
We use the following steps to obtain the hazard ratio estimate
and confidence interval had there been no treatment switching:
\itemize{
\item Fit an AFT model to post-progression survival data to estimate
the causal parameter \eqn{\psi} based on the patients
in the control group who had disease progression.
\item Derive the counterfactual survival times for control patients
had there been no treatment switching.
\item Fit the Cox proportional hazards model to the observed survival times
for the experimental group and the counterfactual survival times
for the control group to obtain the hazard ratio estimate.
\item If bootstrapping is used, the confidence interval and corresponding
p-value for hazard ratio are calculated based on a t-distribution with
\code{n_boot - 1} degrees of freedom.
}
}
\examples{

library(dplyr)

# the eventual survival time
shilong1 <- shilong \%>\%
  arrange(bras.f, id, tstop) \%>\%
  group_by(bras.f, id) \%>\%
  slice(n()) \%>\%
  select(-c("ps", "ttc", "tran"))

# the last value of time-dependent covariates before pd
shilong2 <- shilong \%>\%
  filter(pd == 0 | tstart <= dpd) \%>\%
  arrange(bras.f, id, tstop) \%>\%
  group_by(bras.f, id) \%>\%
  slice(n()) \%>\%
  select(bras.f, id, ps, ttc, tran)

# combine baseline and time-dependent covariates
shilong3 <- shilong1 \%>\%
  left_join(shilong2, by = c("bras.f", "id"))

# apply the two-stage method
fit1 <- tsesimp(
  data = shilong3, id = "id", time = "tstop", event = "event",
  treat = "bras.f", censor_time = "dcut", pd = "pd",
  pd_time = "dpd", swtrt = "co", swtrt_time = "dco",
  base_cov = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c",
                "pathway.f"),
  base2_cov = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c",
                "pathway.f", "ps", "ttc", "tran"),
  aft_dist = "weibull", alpha = 0.05,
  recensor = TRUE, swtrt_control_only = FALSE, offset = 1,
  boot = FALSE)

c(fit1$hr, fit1$hr_CI)

}
\references{
Nicholas R Latimer, KR Abrams, PC Lambert, MK Crowther, AJ Wailoo,
JP Morden, RL Akehurst, and MJ Campbell.
Adjusting for treatment switching in randomised controlled trials - A
simulation study and a simplified two-stage method.
Statistical Methods in Medical Research. 2017;26(2):724-751.
}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
