% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kin.R
\name{fin.kin}
\alias{fin.kin}
\title{Tracking of fin-like extensions of body contours}
\usage{
fin.kin(
  x,
  fin.pos = NULL,
  smooth.n = 50,
  tip.ang = 10,
  smoothing = "loess",
  x.bins = 0.2,
  ml.smooth = 0.25
)
}
\arguments{
\item{x}{a data frame or matrix with 'x' and 'y' data as columns.}

\item{fin.pos}{numeric, a vector of length 2 indicating the start and end of the contour region that contains the fins of interest as a proportion of the body length.}

\item{smooth.n}{numeric, the number of smoothing operations undertaken by \link{coo_smooth} on the contour described by 'x'.}

\item{tip.ang}{the minimum angle, in degrees, that defines tip of each fin. See Details.}

\item{smoothing}{character, the midline smoothing method, either 'loess' or 'spline'.}

\item{x.bins}{numeric, when less than or equal to 1, the proportion of contour coordinates to sample for midline estimation. If greater than 1, the absolute number of equally spaced x values from which to compute the midline. See Details.}

\item{ml.smooth}{numeric the smoothing value for the midline. If \code{smoothing} is set to 'loess', passed to 'span' value for \code{\link{loess}}. If \code{smoothing} is set to 'spline', passed to 'spar' value for \code{\link{smooth.spline}}}
}
\value{
A list with the following components:

\code{body} a data table consisting of x,y coordinates of the body contour

\code{fin} a data table describing the contour of the fins consisting of the following:

\itemize{
\item x,y coordinates within the range of \code{fin.pos}

\item 'ang': the angle formed by each coordinate and its adjacent points.
\item 'fin': fin side, 'L' or 'R'
\item 'y.pred': predicted y values according to \code{lm()} from start to end of fins. 
}

\code{fin.pts} a data table describing fin position consisting of the following:

\itemize{
\item  x,y coordinates of the fin tips, start, and end within the range of \code{fin.pos}.
\item 'ang': the angle formed by the coordinates and their adjacent points.
\item 'pos': description  of the coordinates' positions, 'start', 'end' or 'tip'.
}

\code{comp} a data table describing the composite contour of the body minus the fins.
\itemize{
\item  x,y coordinates of the body except the range of x values within \code{fin.pos}. These values take on a straight line described by the prediction of \code{lm()} based on the start and end of the fin. See Details.
}

\code{midline} a data table describing the estimated 
\itemize{
\item 'x': the mean x position within the bin.
\item 'x.bin': the x bin in \code{\link{cut}} notation.
\item 'y.m': the y midpoint at the bind and mean x value.
\item 'ml.pred': the y midline value according to the smoothing parameters.
}
}
\description{
Estimates the amplitudes of regions along a body contour that are protruding. Useful in computing paired-fin amplitudes from contour data produced from  \link{kin.simple} and \link{kin.search}. Also computes a smoothed midline based on the body outline with the fin region removed.
}
\details{
The algorithm assumes a left-right orientation, i.e., the head of the contour is left. If otherwise oriented, contour can be flipped with \code{\link{coo_flipx}} and \code{\link{coo_flipy}} after converting contour to class \code{coo}.

 \code{tip.angle} is used to define the tip of the fin, assuming that the tip of the fin is pointed and, for a sufficiently smoothed fin contour, will have contour edges that form the highest angles within the fin region defined by \code{fin.pos}. Low values of \code{smooth.n} ($<$5) should be avoided if the contour is jagged, perhaps due to digitization.
 
In addition to fin amplitude and contour extraction, also produces a composite contour of the body minus the fin area described by \code{fin.pos}. Fin contours are replaced by a simple linear prediction constructed from the coordinates of the first and last values covered by \code{fin.pos}. The result is a straight line between the start and end of each fin. From this composite body contour, a midline prediction is made based on the method indicated by \code{smoothing} and number of points indicated by \code{x.bins}. 
 
 \code{x.bins} controls the bin size of x values used to estimate the midline. From these bins, mean x and the range of y is calculated. The midpoint at each mean x is then calculated from the mid point of y. When less then 1, \code{x.bins} values approaching 1 may result in poor a midline as x values on one side of the contour may not have corresponding identical values on the other. Values closer to 0 will result in fewer points but a more robust midline. Higher \code{smooth.n} values will also result in a more robust midline estimation (but also a loss of contour information).
}
\examples{
###plot pectoral-fin amplitudes of a swimming sunfish
\dontrun{
require(ggplot2)

#download example avi video
f <- "https://github.com/ckenaley/exampledata/blob/master/sunfish_pect.avi?raw=true"
download.file(f,"sunfish.avi")

#extract images with ffmpeg opereations and reduce them to 600 px wide with a filter
filt.red <- " -vf scale=600:-1 " #filter
vid.to.images2(vid.path="sunfish.avi",filt = filt.red) #extract

#number of frames
fr <- length(list.files("images"))
#extract contours and other data
kin <- kin.simple(image.dir = "images",frames=c(1:fr),thr=0.9,ant.per = 0.25)
#fin amplitudes by frame with data.table
fin.pos <- c(0.25,.5)
fin.dat <- kin$cont[, { f <- fin.kin(data.frame(x=x,y=y),fin.pos =fin.pos);
list(amp=f$amp$amp,fin=f$amp$fin,amp.bl=f$amp$amp.bl)},by=list(frame)]
p <- ggplot(dat=fin.dat,aes(x=frame,y=amp,col=fin))+geom_line()+theme_classic(15)
print(p)


## plot body and fin contours of frame 1
cont <- data.frame(x=kin$cont[frame==2,list(x,y)]$x,y=kin$cont[frame==2,list(y)]$y)
fins <- fin.kin(cont,fin.pos =fin.pos,x.bins=100)

#plot body contour and fins 
p <- qplot(data=fins$body,x=x,y=y)+geom_point(data=fins$fin,aes(x,y),col="red",size=3)
p+geom_point(data=fins$fin.pts,aes(x,y,shape=pos))+xlim(c(0,kin$dim[1]))+ylim(c(0,kin$dim[2]))

#plot body contour minus fins and the body midline
p <- qplot(data=fins$comp,x=x,y=y)+geom_point(data=fins$midline,aes(x,ml.pred),col="red",size=2)
p+xlim(c(0,kin$dim[1]))+ylim(c(0,kin$dim[2]))

}



}
\seealso{
\code{\link{kin.simple}}, \code{\link{kin.LDA}}, \code{\link{kin.search}}, \code{\link{efourier}}, \code{\link{coo_angle_edges}}, \code{\link{coo_smooth}}, \code{\link{loess}}, \code{\link{smooth.spline}}
}
