#' Atomic Type Casting Without Stripping Attributes
#'
#' @description
#' Atomic type casting in R is generally performed using the functions
#' \link[base]{as.logical}, \link[base]{as.integer},
#' \link[base]{as.double}, \link[base]{as.character},
#' \link[base]{as.complex}, and \link[base]{as.raw}. \cr
#' \cr
#' Converting an object between atomic types using these functions
#' strips the object of its attributes,
#' including attributes such as names and dimensions. \cr
#' \cr
#' The functions provided here by the 'tinycodet' package
#' preserve all attributes - except the "class" attribute. \cr
#' \cr
#' The functions are as follows: \cr
#'
#'  * \code{as_bool()}: converts object to atomic type \code{logical} (\code{TRUE, FALSE, NA}).
#'  * \code{as_int()}: converts object to atomic type \code{integer}.
#'  * \code{as_dbl()}: converts object to atomic type \code{double} (AKA decimal numbers).
#'  * \code{as_chr()}: converts object to atomic type \code{character}.
#'  * \code{as_cplx()}: converts object to atomic type \code{complex}.
#'  * \code{as_raw()}:converts object to atomic type \code{raw}.
#'
#'
#' @param x vector, matrix, array
#' (or a similar object where all elements share the same type).
#' @param ... further arguments passed to or from other methods.
#'
#' @returns
#' The converted object.
#'
#' @seealso \link{tinycodet_safer}
#'
#' @examples
#' x <- c(rep(0, 2), seq(0, 2.5, by=0.5)) |> matrix(ncol=2)
#' colnames(x) <- c("one", "two")
#' attr(x, "test") <- "test"
#' print(x)
#'
#' # notice that in all following, attributes (except class) are conserved:
#' as_bool(x)
#' as_int(x)
#' as_dbl(x)
#' as_chr(x)
#'
#' 
#'
#'

#' @name atomic_conversions
NULL


#' @rdname atomic_conversions
#' @export
as_bool <- function(x, ...) {
  temp.attr <- attributes(x)
  temp.attr[["class"]] <- NULL
  out <- as.logical(x, ...)
  attributes(out) <- temp.attr
  return(out)
}


#' @rdname atomic_conversions
#' @export
as_int <- function(x, ...) {
  temp.attr <- attributes(x)
  temp.attr[["class"]] <- NULL
  out <- as.integer(x, ...)
  attributes(out) <- temp.attr
  return(out)
}


#' @rdname atomic_conversions
#' @export
as_dbl <- function(x, ...) {
  temp.attr <- attributes(x)
  temp.attr[["class"]] <- NULL
  out <- as.double(x, ...)
  attributes(out) <- temp.attr
  return(out)
}


#' @rdname atomic_conversions
#' @export
as_chr <- function(x, ...) {
  temp.attr <- attributes(x)
  temp.attr[["class"]] <- NULL
  out <- as.character(x, ...)
  attributes(out) <- temp.attr
  return(out)
}


#' @rdname atomic_conversions
#' @export
as_cplx <- function(x, ...) {
  temp.attr <- attributes(x)
  temp.attr[["class"]] <- NULL
  out <- as.complex(x, ...)
  attributes(out) <- temp.attr
  return(out)
}


#' @rdname atomic_conversions
#' @export
as_raw <- function(x, ...) {
  temp.attr <- attributes(x)
  temp.attr[["class"]] <- NULL
  out <- as.raw(x, ...)
  attributes(out) <- temp.attr
  return(out)
}