% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/publish_DSC_via_WebService.R
\name{publish_DSC_via_WebService}
\alias{publish_DSC_via_WebService}
\title{Publish a Data Stream Clustering Task via a Web Service}
\usage{
publish_DSC_via_WebService(
  dsc,
  port,
  task_file = NULL,
  serializer = "csv",
  serve = TRUE,
  background = TRUE,
  debug = FALSE
)
}
\arguments{
\item{dsc}{A character string that creates a DSC.}

\item{port}{port used to serve the task.}

\item{task_file}{name of the plumber task script file.}

\item{serializer}{method used to serialize the data. By default \code{csv} (comma separated values)
is used. Other methods are \code{json} and \code{rds} (see \link[plumber:serializers]{plumber::serializer_csv}).}

\item{serve}{if \code{TRUE}, then a task file is written and a server started, otherwise,
only a plumber task file is written.}

\item{background}{logical; start a background process?}

\item{debug}{if \code{TRUE}, then the service is started locally and a web client is started to explore the interface.}
}
\value{
a \link[processx:process]{processx::process} object created with \code{\link[callr:r_bg]{callr::r_bg()}} which runs the plumber server
in the background. The process can be stopped with \code{rp$kill()} or by killing the process
using the operating system with the appropriate PID. \code{rp$get_result()} can
be used to check for errors in the server process (e.g., when it terminates
unexpectedly).
}
\description{
Uses the package \link{plumber} to publish a data stream task as a web service.
}
\details{
The function writes a plumber task script file and starts the web server to serve
the content of the stream using the endpoints
\itemize{
\item GET \verb{/info}
\item POST \verb{/update} requires the data to be uploaded as a file in csv format (see Examples section).
\item GET \verb{/get_centers} with parameter \code{type} (see \code{\link[=get_centers]{get_centers()}}).
\item GET \verb{/get_weights} with parameter \code{type} (see \code{\link[=get_weights]{get_weights()}}).
}

Supported serializers are \code{csv} (default), \code{json}, and \code{rds}.

APIs generated using plumber can be easily deployed. See: \href{https://www.rplumber.io/articles/hosting.html}{Hosting}. By setting a \code{task_file} and \code{serve = FALSE} a plumber
task script file is generated that can deployment.
}
\examples{
# find a free port
port <- httpuv::randomPort()
port

# Deploy a clustering process listening for data on the port
rp1 <- publish_DSC_via_WebService("DSC_DBSTREAM(r = .05)", port = port)
rp1

# look at ? DSC_WebService for a convenient interface. 
# Here we we show how to connect to the port and send data manually.
library(httr)

# the info verb returns some basic information about the clusterer.
resp <- RETRY("GET", paste0("http://localhost:", port, "/info"))
d <- content(resp, show_col_types = FALSE)
d

# create a local data stream and send it to the clusterer using the update verb.
dsd <- DSD_Gaussians(k = 3, d = 2, noise = 0.05)

tmp <- tempfile()
stream::write_stream(dsd, tmp, n = 500, header = TRUE)
resp <- POST(paste0("http://localhost:", port, "/update"), 
  body = list(upload = upload_file(tmp)))
unlink(tmp)
resp

# retrieve the cluster centers using the get_centers verb
resp <- GET(paste0("http://localhost:", port, "/get_centers"))
d <- content(resp, show_col_types = FALSE)
head(d)

plot(dsd, n = 100)
points(d, col = "red", pch = 3, lwd = 3)

# kill the process.
rp1$kill()
rp1

# Debug the interface (run the service and start a web interface)
if (interactive())
  publish_DSC_via_WebService("DSC_DBSTREAM(r = .05)", 
         port = port, debug = TRUE)
}
\seealso{
Other WebService: 
\code{\link{DSC_WebService}()},
\code{\link{DSD_ReadWebService}()},
\code{\link{publish_DSD_via_WebService}()}

Other dsc: 
\code{\link{DSC_WebService}()}
}
\concept{WebService}
\concept{dsc}
