% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/std_selected.R, R/std_selected_boot.R
\name{std_selected}
\alias{std_selected}
\alias{std_selected_boot}
\title{Standardize Variables in a Regression Model}
\usage{
std_selected(lm_out, to_scale = NULL, to_center = NULL, to_standardize = NULL)

std_selected_boot(
  lm_out,
  to_scale = NULL,
  to_center = NULL,
  to_standardize = NULL,
  conf = 0.95,
  nboot = 100,
  boot_args = NULL,
  save_boot_est = TRUE,
  full_output = FALSE,
  do_boot = TRUE
)
}
\arguments{
\item{lm_out}{The output from \code{\link[=lm]{lm()}}.}

\item{to_scale}{The terms to be rescaled by standard deviation,
specified by a formula as in \code{\link[=lm]{lm()}}. For example, if the terms to be scaled
are \code{x1} and \code{x3}, use \code{~ x1 + x3}. No need to specify the
interaction
term. To scale the outcome variable, list it on the \emph{right hand side}
as a predictor.
Specify only the original variables. If \code{NULL}, then no terms
will be rescaled by their standard deviations. Variables that are not
numeric will be ignored. Default is \code{NULL}.}

\item{to_center}{The terms to be mean centered, specified by a formula
as in \code{\link[=lm]{lm()}}. For example, if the terms to be centered
is \code{x1} and \code{x3}, use \code{~ x1 + x3}. No need to specify the
interaction term. To center  the outcome variable, list it on the
\emph{right hand side}
as a predictor.
Specify only the original variables. If \code{NULL}, then no term
will be centered. Default is \code{NULL}.}

\item{to_standardize}{The terms to be standardized, specified by a formula
as in \code{\link[=lm]{lm()}}. For example, if the terms to be standardized
is \code{x1} and \code{x3}, use \code{~ x1 + x3}. No need to specify the
interaction term. To standardize the outcome variable, list it on the
\emph{right hand side}
as a predictor.
Specify only the original variables.
This is a shortcut to \code{to_center} and \code{to_scale}. Listing a variable
in \code{to_standardize} is equivalent to listing this variable in
both \code{to_center} and \code{to_scale}.
Default is \code{NULL}.}

\item{conf}{The level of confidence for the confidence interval.
Default is .95.}

\item{nboot}{The number of bootstrap samples. Default is 100.}

\item{boot_args}{A named list of arguments to be passed to \code{\link[boot:boot]{boot::boot()}}.
Default is \code{NULL}.}

\item{save_boot_est}{If \code{TRUE}, the default, the bootstrap estimates will
be saved in the element
\code{boot_est} of the output.}

\item{full_output}{Whether the full output from \code{\link[boot:boot]{boot::boot()}} is returned.
Default is \code{FALSE}. If \code{TRUE}, the full output from
\code{\link[boot:boot]{boot::boot()}} will be saved in the element \code{boot_out}
of the output.}

\item{do_boot}{Whether bootstrapping confidence intervals will be formed.
Default is \code{TRUE}. If \code{FALSE}, all arguments related to
bootstrapping will be ignored.}
}
\value{
The updated \code{\link[=lm]{lm()}} output, with the class \code{std_selected} added. It will be
treated as a usual \code{\link[=lm]{lm()}} object by most functions. These are the major
additional element in the list:
\itemize{
\item \code{scaled_terms}: If not \code{NULL}, a character vector of the variables scaled.
\item \code{centered_terms}: If not \code{NULL}, a character vector of the variables mean-centered.
\item \code{scaled_by}: A numeric vector of the scaling factors for all the variables in
the model. The value is 1 for terms not scaled.
\item \code{centered_by}: A numeric vector of the numbers used for centering for all
the variables in the model. The value is 0 for
terms not centered.
\item \code{std_selected_call}: The original call.
\item \code{lm_out_call}: The call in \code{lm_out}.
}

Like \code{\link[=std_selected]{std_selected()}}, \code{\link[=std_selected_boot]{std_selected_boot()}} returns the updated \code{\link[=lm]{lm()}}
output, with the class \code{std_selected} added. The output of \code{\link[=std_selected_boot]{std_selected_boot()}}
contain these additional elements in the list:
\itemize{
\item \code{boot_ci}: A data frame of the bootstrap confidence intervals of the
regression coefficient.
\item \code{nboot}: The number of bootstrap samples requested.
\item \code{conf}: The level of confidence, in proportion.
\item \code{boot_est}: A matrix of the bootstrap estimates of the regression coefficients.
The number of rows equal to \code{nboot}, and the number of columns
equal to the number of terms in the regression model.
\item \code{std_selected_boot_call}: The call to \code{std_selected_boot()}.
\item \code{boot_out}: If available, the original output from \code{boot::boot()}.
}
}
\description{
Standardize, mean center, or scale by standard deviation
selected variables in a regression model and refit the model
}
\details{
\code{\link[=std_selected]{std_selected()}} was originally developed to compute the standardized
moderation effect and the standardized coefficients for other predictors
given an \code{\link[=lm]{lm()}} output (Cheung, Cheung, Lau, Hui, & Vong, 2022).
It has been extended such that users can specify
which variables in a regression model are to be mean-centered and/or
rescaled by
their standard deviations. If the model has one or more interaction terms,
they will be formed after the transformation, yielding the correct
standardized solution for a moderated regression model. Moreover,
categorical predictors will be automatically skipped in mean-centering
and rescaling.

Standardization is conducted when a variable is mean-centered and then
rescaled by its standard deviation. Therefore, if the goal is to get the
standardized solution of a moderated regression, users
just instruct the function to standardize all non-categorical variables
in the regression model.

\code{\link[=std_selected_boot]{std_selected_boot()}} is a wrapper of \code{\link[=std_selected]{std_selected()}}. It calls
\code{\link[=std_selected]{std_selected()}}
once
for each bootstrap sample, and then computes the nonparametric
bootstrap
percentile confidence intervals (Cheung, Cheung, Lau, Hui, & Vong, 2022).

If \code{do_boot} is \code{FALSE}, then the object it returns is identical to that
by \code{\link[=std_selected]{std_selected()}}.

This function intentionally does not have an argument for setting the seed
for
random number. Users are recommended to set the seed, e.g., using
\code{\link[=set.seed]{set.seed()}}
before calling it, to ensure reproducibility.
}
\section{Functions}{
\itemize{
\item \code{std_selected()}: The base function to center or
scale selected variables in a regression model

\item \code{std_selected_boot()}: A wrapper of \code{\link[=std_selected]{std_selected()}} that forms
nonparametric bootstrap confidence intervals.

}}
\examples{

# Load a sample data set

dat <- test_x_1_w_1_v_1_cat1_n_500
head(dat)

# Do a moderated regression by lm
lm_raw <- lm(dv ~ iv*mod + v1 + cat1, dat)
summary(lm_raw)

# Mean center mod only
lm_cw <- std_selected(lm_raw, to_center = ~ mod)
summary(lm_cw)

# Mean center mod and iv
lm_cwx <- std_selected(lm_raw, to_center = ~ mod + iv)
summary(lm_cwx)

# Standardize both mod and iv
lm_stdwx <- std_selected(lm_raw, to_scale = ~ mod + iv,
                               to_center = ~ mod + iv)
summary(lm_stdwx)

# Standardize all variables except for categorical variables.
# Interaction terms are formed after standardization.
lm_std <- std_selected(lm_raw, to_scale = ~ .,
                               to_center = ~ .)
summary(lm_std)

# Use to_standardize as a shortcut
lm_stdwx2 <- std_selected(lm_raw, to_standardize = ~ mod + iv)
# The results are the same
coef(lm_stdwx)
coef(lm_stdwx2)
all.equal(coef(lm_stdwx), coef(lm_stdwx2))



dat <- test_x_1_w_1_v_1_cat1_n_500
head(dat)

# Do a moderated regression by lm
lm_raw <- lm(dv ~ iv*mod + v1 + cat1, dat)
summary(lm_raw)
# Standardize all variables as in std_selected above, and compute the
# nonparametric bootstrapping percentile confidence intervals.
set.seed(87053)
lm_std_boot <- std_selected_boot(lm_raw,
                                 to_scale = ~ .,
                                 to_center = ~ .,
                                 conf = .95,
                                 nboot = 100)
# In real analysis, nboot should be at least 2000.
summary(lm_std_boot)

# Use to_standardize as a shortcut
set.seed(87053)
lm_std_boot2 <- std_selected_boot(lm_raw,
                                  to_standardize = ~ .,
                                  conf = .95,
                                  nboot = 100)
# The results are the same
confint(lm_std_boot)
confint(lm_std_boot2)
all.equal(confint(lm_std_boot), confint(lm_std_boot2))


}
\references{
Cheung, S. F., Cheung, S.-H., Lau, E. Y. Y., Hui, C. H., & Vong, W. N.
(2022) Improving an old way to measure moderation effect in standardized
units. \emph{Health Psychology}, \emph{41}(7), 502-505.
\doi{10.1037/hea0001188}
}
\author{
Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}
}
