% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxph_methods.R
\name{standardize_coxph}
\alias{standardize_coxph}
\title{Regression standardization in Cox proportional hazards models}
\usage{
standardize_coxph(
  formula,
  data,
  values,
  times,
  measure = c("survival", "rmean"),
  clusterid,
  ci_level = 0.95,
  ci_type = "plain",
  contrasts = NULL,
  family = "gaussian",
  reference = NULL,
  transforms = NULL
)
}
\arguments{
\item{formula}{The formula which is used to fit the model for the outcome.}

\item{data}{The data.}

\item{values}{A named list or data.frame specifying the variables and values
at which marginal means of the outcome will be estimated.}

\item{times}{A vector containing the specific values of \eqn{T} at
which to estimate the standardized survival function.}

\item{measure}{Either "survival" to estimate the survival function at times
or "rmean" for the restricted mean survival up to the largest of times.}

\item{clusterid}{An optional string containing the name of a cluster identification variable
when data are clustered.}

\item{ci_level}{Coverage probability of confidence intervals.}

\item{ci_type}{A string, indicating the type of confidence intervals.
Either "plain", which gives untransformed intervals, or "log", which gives
log-transformed intervals.}

\item{contrasts}{A vector of contrasts in the following format:
If set to \code{"difference"} or \code{"ratio"}, then \eqn{\psi(x)-\psi(x_0)}
or \eqn{\psi(x) / \psi(x_0)} are constructed, where \eqn{x_0} is a reference
level specified by the \code{reference} argument. Has to be \code{NULL}
if no references are specified.}

\item{family}{The family argument which is used to fit the glm model for the outcome.}

\item{reference}{A vector of reference levels in the following format:
If \code{contrasts} is not \code{NULL}, the desired reference level(s). This
must be a vector or list the same length as \code{contrasts}, and if not named,
it is assumed that the order is as specified in contrasts.}

\item{transforms}{A vector of transforms in the following format:
If set to \code{"log"}, \code{"logit"}, or \code{"odds"}, the standardized
mean \eqn{\theta(x)} is transformed into \eqn{\psi(x)=\log\{\theta(x)\}},
\eqn{\psi(x)=\log[\theta(x)/\{1-\theta(x)\}]}, or
\eqn{\psi(x)=\theta(x)/\{1-\theta(x)\}}, respectively.
If the vector is \code{NULL}, then \eqn{\psi(x)=\theta(x)}.}
}
\value{
An object of class \code{std_surv}.
This is basically a list with components estimates and covariance estimates in \code{res}
Results for transformations, contrasts, references are stored in \code{res_contrasts}.
The output contains estimates for contrasts and confidence intervals for all
combinations of transforms and reference levels.
Obtain numeric results in a data frame with the \link{tidy} function.
}
\description{
\code{standardize_coxph} performs regression standardization in Cox proportional
hazards models at specified values of the exposure over the sample
covariate distribution. Let \eqn{T}, \eqn{X}, and \eqn{Z} be the survival
outcome, the exposure, and a vector of covariates, respectively.
\code{standardize_coxph} fits a Cox proportional hazards model and the Breslow estimator
of the baseline hazard in order to estimate the
standardized survival function \eqn{\theta(t,x)=E\{S(t|X=x,Z)\}} when \code{measure = "survival"} or the standardized restricted mean survival up to time t \eqn{\theta(t, x) = E\{\int_0^t S(u|X = x, Z) du\}} when \code{measure = "rmean"}, where
\eqn{t} is a specific value of \eqn{T}, \eqn{x} is a specific value of
\eqn{X}, and the expectation is over the marginal distribution of \eqn{Z}.
}
\details{
\code{standardize_coxph} fits the Cox proportional hazards model
\deqn{\lambda(t|X,Z)=\lambda_0(t)\exp\{h(X,Z;\beta)\}.}
Breslow's estimator of the cumulative baseline hazard
\eqn{\Lambda_0(t)=\int_0^t\lambda_0(u)du} is used together with the partial
likelihood estimate of \eqn{\beta} to obtain estimates of the survival
function \eqn{S(t|X=x,Z)} if \code{measure = "survival"}:
\deqn{\hat{S}(t|X=x,Z)=\exp[-\hat{\Lambda}_0(t)\exp\{h(X=x,Z;\hat{\beta})\}].}
For each \eqn{t} in the \code{t} argument and for each \eqn{x} in the
\code{x} argument, these estimates are averaged across all subjects (i.e.
all observed values of \eqn{Z}) to produce estimates
\deqn{\hat{\theta}(t,x)=\sum_{i=1}^n \hat{S}(t|X=x,Z_i)/n,} where \eqn{Z_i}
is the value of \eqn{Z} for subject \eqn{i}, \eqn{i=1,...,n}.  The variance
for \eqn{\hat{\theta}(t,x)} is obtained by the sandwich formula.

If \code{measure = "rmean"}, then \eqn{\Lambda_0(t)=\int_0^t\lambda_0(u)du}
is used together with the partial
likelihood estimate of \eqn{\beta} to obtain estimates of the restricted mean survival
up to time t: \eqn{\int_0^t S(u|X=x,Z) du} for each element of \code{times}. The estimation
and inference is done using the method described in Chen and Tsiatis 2001.
Currently, we can only estimate the difference in RMST for a single binary
exposure. Two separate Cox models are fit for each level of the exposure,
which is expected to be coded as 0/1.
}
\note{
Standardized survival functions are sometimes referred to as (direct)
adjusted survival functions in the literature.

\code{standardize_coxph/standardize_parfrailty} does not currently handle time-varying exposures or
covariates.

\code{standardize_coxph/standardize_parfrailty} internally loops over all values in the \code{t} argument.
Therefore, the function will usually be considerably faster if
\code{length(t)} is small.

The variance calculation performed by \code{standardize_coxph} does not condition on
the observed covariates \eqn{\bar{Z}=(Z_1,...,Z_n)}. To see how this
matters, note that
\deqn{var\{\hat{\theta}(t,x)\}=E[var\{\hat{\theta}(t,x)|\bar{Z}\}]+var[E\{\hat{\theta}(t,x)|\bar{Z}\}].}
The usual parameter \eqn{\beta} in a Cox proportional hazards model does not
depend on \eqn{\bar{Z}}. Thus, \eqn{E(\hat{\beta}|\bar{Z})} is independent
of \eqn{\bar{Z}} as well (since \eqn{E(\hat{\beta}|\bar{Z})=\beta}), so that
the term \eqn{var[E\{\hat{\beta}|\bar{Z}\}]} in the corresponding variance
decomposition for \eqn{var(\hat{\beta})} becomes equal to 0. However,
\eqn{\theta(t,x)} depends on \eqn{\bar{Z}} through the average over the
sample distribution for \eqn{Z}, and thus the term
\eqn{var[E\{\hat{\theta}(t,x)|\bar{Z}\}]} is not 0, unless one conditions on
\eqn{\bar{Z}}. The variance calculation by Gail and Byar (1986) ignores this
term, and thus effectively conditions on \eqn{\bar{Z}}.
}
\examples{


require(survival)
set.seed(7)
n <- 300
Z <- rnorm(n)
Zbin <- rbinom(n, 1, .3)
X <- rnorm(n, mean = Z)
T <- rexp(n, rate = exp(X + Z + X * Z)) # survival time
C <- rexp(n, rate = exp(X + Z + X * Z)) # censoring time
fact <- factor(sample(letters[1:3], n, replace = TRUE))
U <- pmin(T, C) # time at risk
D <- as.numeric(T < C) # event indicator
dd <- data.frame(Z, Zbin, X, U, D, fact)
fit.std.surv <- standardize_coxph(
  formula = Surv(U, D) ~ X + Z + X * Z,
  data = dd,
  values = list(X = seq(-1, 1, 0.5)),
  times = 1:5
)
print(fit.std.surv)
plot(fit.std.surv)
tidy(fit.std.surv)

fit.std <- standardize_coxph(
  formula = Surv(U, D) ~ X + Zbin + X * Zbin + fact,
  data = dd,
  values = list(Zbin = 0:1),
  times = 1.5,
  measure = "rmean",
  contrast = "difference",
  reference = 0
)
print(fit.std)
tidy(fit.std)

}
\references{
Chang I.M., Gelman G., Pagano M. (1982). Corrected group prognostic curves
and summary statistics. \emph{Journal of Chronic Diseases} \bold{35},
669-674.

Gail M.H. and Byar D.P. (1986). Variance calculations for direct adjusted
survival curves, with applications to testing for no treatment effect.
\emph{Biometrical Journal} \bold{28}(5), 587-599.

Makuch R.W. (1982). Adjusted survival curve estimation using covariates.
\emph{Journal of Chronic Diseases} \bold{35}, 437-443.

Sjölander A. (2016). Regression standardization with the R-package stdReg.
\emph{European Journal of Epidemiology} \bold{31}(6), 563-574.

Sjölander A. (2018). Estimation of causal effect measures with the R-package
stdReg. \emph{European Journal of Epidemiology} \bold{33}(9), 847-858.

Chen, P. Y., Tsiatis, A. A. (2001). Causal inference on the difference of the restricted mean lifetime between two groups. \emph{Biometrics}, \bold{57}(4), 1030-1038.
}
\author{
Arvid Sjölander, Adam Brand, Michael Sachs
}
