% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitTD.R
\name{fitTD}
\alias{fitTD}
\title{Fit Single Trial Mixed Model}
\usage{
fitTD(
  TD,
  trials = names(TD),
  design = NULL,
  traits,
  what = c("fixed", "random"),
  covariates = NULL,
  useCheckId = FALSE,
  spatial = FALSE,
  engine = NA,
  control = NULL,
  progress = FALSE,
  ...
)
}
\arguments{
\item{TD}{An object of class \code{\link{TD}}.}

\item{trials}{A character vector specifying the trials for modeling.}

\item{design}{A character string specifying the experimental design. Either
"ibd" (incomplete block design), "res.ibd" (resolvable incomplete block
design), "rcbd" (randomized complete block design), "rowcol" (row column
design) or "res.rowcol" (resolvable row column design).}

\item{traits}{A character vector specifying the traits for modeling.}

\item{what}{A character vector specifying whether "genotype" should
be fitted as "fixed" or "random" effect. If not specified, both models
are fitted.}

\item{covariates}{A character vector specifying covariates to be fitted as
extra fixed effects in the model.}

\item{useCheckId}{Should checkId be used as a fixed effect in the model?\cr
If \code{TRUE}, \code{TD} has to contain a column 'checkId'.}

\item{spatial}{Should spatial models be tried? Spatial models can
only be fitted with SpATS and asreml. If SpATS is used for modeling, only
spatial models can be fitted and spatial is always set to \code{TRUE}. If
asreml is used, fitting spatial models is optional.}

\item{engine}{A string specifying the name of the mixed modeling engine to
use, either "SpATS", "lme4" or "asreml." For spatial models, "SpaTS" is used
as default, for other models "lme4".}

\item{control}{An optional list with control parameters to be passed to the
actual fitting functions. Currently \code{nSeg} and \code{nestDiv} are valid
parameters when fitting a model using SpATS. They pass a value to nseg and
nest.div in \code{\link[SpATS]{PSANOVA}} respectively. For \code{nSeg} also a
named list can be supplied containing values for nSeg per trial.\cr
\code{criterion} is a valid parameter when fitting a spatial model using
asreml. It may be used to pass a goodness-of-fit criterion for comparing
different spatial models. See also in details. Other parameters are ignored.}

\item{progress}{Should the progress of the modeling be printed. If
\code{TRUE}, for every trial a line is output indicating the traits fitted
for the particular trial.}

\item{...}{Further arguments to be passed to \code{SpATS}, \code{lme4} or
\code{asreml}.}
}
\value{
An object of class \code{STA}, a list containing, per trial
that has been analyzed, a list of:
\item{mRand}{A list of models with fitted with genotype as random effect.}
\item{mFix}{A list of models fitted with genotype as fixed effect.}
\item{TD}{An object of class \code{\link{TD}} containing the data on which
\code{mRand} and \code{mFix} are based.}
\item{traits}{A character vector indicating the traits for which the analysis
is done.}
\item{design}{A character string containing the design of the trial.
(see \code{\link{fitTD}} for the possible designs).}
\item{spatial}{A character string indicating the spatial part of the model.
\code{FALSE} if no spatial design has been used.}
\item{engine}{A character string containing the engine used for the
analysis.}
\item{predicted}{A character string indicating the variable that has been
predicted.}
}
\description{
Perform REML analysis given a specific experimental design using either
SpATS, lme4 or asreml. SpATS is used as a default method when design is
rowcol or res.rowcol, lme4 for other designs.
See details for the exact models fitted.
}
\details{
The actual model fitted depends on the design. For the supported designs, the
following models are used:
\describe{
\item{ibd}{trait = genotype + \emph{subBlock} + e}
\item{res.ibd}{trait = genotype + \strong{repId} +
\emph{repId:subBlock} + e}
\item{rcbd}{trait = genotype + \strong{repId} + e}
\item{rowcol}{trait = genotype + \emph{rowId} + \emph{colId} + e}
\item{res.rowcol}{trait = genotype + \strong{repId} +
\emph{repId:rowId} + \emph{repId:colId} + e}
}
In the above models fixed effects are indicated in \strong{bold}, random
effects in \emph{italics}. genotype is fitted as fixed or random effect
depending on the value of \code{what}.\cr
In case \code{useCheckId = TRUE}, an extra fixed effect \strong{checkId} is
included in the model.\cr
Variables in \code{covariates} are fitted as extra fixed effects.\cr\cr
When \code{SpATS} is used for modeling, an extra spatial term is included
in the model. This term is constructed using the function
\code{\link[SpATS]{PSANOVA}} from the SpATS package as\cr
\code{PSANOVA(colCoord, rowCoord, nseg = nSeg, nest.div = 2)}
where\cr \code{nSeg = (number of columns / 2, number of rows / 2)}. nseg and
nest.div can be modified using the \code{control} parameter.\cr\cr
When \code{asreml} is used for modeling and \code{spatial} is \code{TRUE}
six models are fitted with different random terms and covariance structure.
The best model is determined based on a goodness-of-fit criterion, either
AIC or BIC. This can be set using the control parameter \code{criterion},
default is AIC.
The fitted random terms depend on the structure of the data. If the design
has a regular structure, i.e. all replicates appear the same amount of times
in the design, the following combinations of random and spatial terms are
fitted
\itemize{
\item{random = NULL, spatial = exp(rowCoord):colCoord}
\item{random = NULL, spatial = rowCoord:exp(colCoord)}
\item{random = NULL, spatial = iexp(rowCoord,colCoord)}
\item{random = repId:rowId, spatial = exp(rowCoord):colCoord}
\item{random = repId:colId, spatial = rowCoord:exp(colCoord)}
\item{random = repId:rowId + repId:colId, spatial = iexp(rowCoord,colCoord)}
}
If the design is not regular the following combinations of random and spatial
terms are fitted
\itemize{
\item{random = NULL, spatial = ar1(rowId):colId}
\item{random = NULL, spatial = rowId:ar1(colId)}
\item{random = NULL, spatial = ar1(rowId):ar1(colId)}
\item{random = repId:rowId, spatial = ar1(rowId):colId}
\item{random = repId:colId, spatial = rowId:ar1(colId)}
\item{random = repId:rowId + repId:colId, spatial = ar1(rowId):ar1(colId)}
}
If there are no replicates in the model, repId is left out from the random
parts above.
}
\examples{
## Fit model using lme4.
myModel1 <- fitTD(TD = TDHeat05, design = "ibd", traits = "yield",
                      what = "fixed")

## Summarize results.
summary(myModel1)

## Create base plots of the results.
plot(myModel1)

## Create a pdf report summarizing results.
\donttest{
report(myModel1, outfile = tempfile(fileext = ".pdf"))
}

## Fit model using SpATS.
myModel2 <- fitTD(TD = TDHeat05, design = "res.rowcol", traits = "yield",
                      what = "fixed")
summary(myModel2)

## Create spatial plots of the results.
plot(myModel2, plotType = "spatial")
\donttest{
report(myModel2, outfile = tempfile(fileext = ".pdf"))
}

## Fit model using asreml.
if (requireNamespace("asreml", quietly = TRUE)) {
  myModel3 <- fitTD(TD = TDHeat05, design = "res.rowcol", traits = "yield",
                   what = "fixed", engine = "asreml")
  summary(myModel3)

  \donttest{
  report(myModel3, outfile = tempfile(fileext = ".pdf"))
  }
}
}
\references{
Maria Xose Rodriguez-Alvarez, Martin P. Boer, Fred A. van Eeuwijk, Paul H.C.
Eilers (2017). Correcting for spatial heterogeneity in plant breeding
experiments with P-splines. Spatial Statistics
\url{https://doi.org/10.1016/j.spasta.2017.10.003}

Butler, D. G., et al. (2010). Analysis of Mixed Models for S language
environments: ASReml-R reference manual. Brisbane, DPI Publications

Douglas Bates, Martin Maechler, Ben Bolker, Steve Walker (2015). Fitting
Linear Mixed-Effects Models Using lme4. Journal of Statistical Software,
67(1), 1-48. \url{https://www.jstatsoft.org/article/view/v067i01/0}.
}
