% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/graph.param.estimator.R
\name{graph.param.estimator}
\alias{graph.param.estimator}
\title{Graph parameter estimator}
\usage{
graph.param.estimator(
  Graph,
  model,
  interval = NULL,
  eps = 0.01,
  search = "grid",
  ...
)
}
\arguments{
\item{Graph}{the undirected graph (igraph object).
If \code{Graph} has the  attribute 'eigenvalues' containing
the eigenvalues of \code{Graph}, such values will be used to
compute spectral density of the graph.}

\item{model}{either a string or a function:

A string that indicates one of the following models: "ER" (Erdos-Renyi random
graph), "GRG" (geometric random graph), "KR" (k regular random graph), "WS"
(Watts-Strogatz model), and "BA" (Barabási-Albert model).

A function that returns a graph (represented by its adjacency matrix)
generated by a graph model. It must contain two arguments: the first one
corresponds to the graph size and the second to the parameter of the model.}

\item{interval}{numeric vector containing the values that will be
considered for the parameter estimation, or a list containing 'lo' and 'hi'
that indicates the model's parameter search interval <\code{lo},\code{hi}>.
The \code{graph.param.estimator} will return the element of 'parameter' that
minimizes the GIC.
If the user does not provide the argument \code{parameters}, and \code{model} is a string,
then default values are used for the predefined models ("ER", "GRG", "KR", "WS",
and "BA"). The default \code{parameter} argument corresponds to a sequence from

0 to 1 with step \code{eps} for the "ER" model (Erdos-Renyi random graph), in
which the parameter corresponds to the probability to connect a pair of
vertices;

0 to sqrt(2) with step \code{eps} for the "GRG" model (geometric random graph), in
which the parameter corresponds to the radius used to construct the geometric
graph in a unit square;

0 to 'n' with step \code{n*eps} for the "KR" model (k regular random graph), in
which the parameter of the model corresponds to the degree \code{k} of a regular
graph;

0 to 1 with step \code{eps} for the "WS" model (Watts-Strogatz model), in which
the parameter corresponds to the probability to reconnect a vertex;

and 0 to 3 with step \code{eps} for the "BA" model (Barabási-Albert model), in
which the parameter corresponds to the scaling exponent.}

\item{eps}{precision of the grid and ternary search (default is \code{0.01}).}

\item{search}{string that indicates the search algorithm to find
the parameter with the smallest GIC. If "grid" (default) parameter is
estimated using grid search, and only works when method is not "fast".
If "ternary" parameter is estimated using ternary search.}

\item{...}{Other relevant parameters for \code{\link{GIC}}.}
}
\value{
A list with class "statGraph" containing the following components:
\item{\code{method:}}{ a string indicating the used method.}
\item{\code{info:}}{ a string showing details about the method.}
\item{\code{data.name:}}{ a string with the data's name(s).}
\item{\code{param:}}{ the parameter estimate. For the "ER", "GRG", "KR", "WS", and "BA"
models, the parameter corresponds to the probability to connect a pair of
vertices, the radius used to construct the geometric graph in a unit square,
the degree \code{k} of a regular graph, the probability to reconnect a vertex, and
the scaling exponent, respectively.}
\item{\code{dist:}}{ the distance between the observed graph and the graph model with the estimated
parameter.}
}
\description{
\code{graph.param.estimator} estimates the parameter that best approximates
the model to the observed graph according to the Graph Information Criterion
(GIC).
}
\examples{
set.seed(1)
G <- igraph::sample_gnp(n=50, p=0.5)

# Using a string to indicate the graph model
result1 <- graph.param.estimator(G, "ER", eps=0.25)
result1


# Using a function to describe the graph model
# Erdos-Renyi graph
set.seed(1)
model <- function(n, p) {
  return(igraph::sample_gnp(n, p))
}
result2 <- graph.param.estimator(G, model,  seq(0.2, 0.8, 0.1))
result2



}
\references{
Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
Discriminating Different Classes of Biological Networks by Analyzing the
Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
doi:10.1371/journal.pone.0049949.

Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.

Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
*21*, 65-66.

Sheather, S. J. and Jones, M. C. (1991). A reliable data-based bandwidth
selection method for kernel density estimation.
_Journal of the Royal Statistical Society series B_, 53, 683-690.
http://www.jstor.org/stable/2345597.
}
\keyword{parameter_estimation}
