##########################################################################
#                                                                        #
#  SPRINT: Simple Parallel R INTerface                                   #
#  Copyright  2008,2009 The University of Edinburgh                     #
#                                                                        #
#  This program is free software: you can redistribute it and/or modify  #
#  it under the terms of the GNU General Public License as published by  #
#  the Free Software Foundation, either version 3 of the License, or     #
#  any later version.                                                    #
#                                                                        #
#  This program is distributed in the hope that it will be useful,       #
#  but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          #
#  GNU General Public License for more details.                          #
#                                                                        #
#  You should have received a copy of the GNU General Public License     #
#  along with this program. If not, see <http://www.gnu.or/licenses/>.   #
#                                                                        #
##########################################################################

# The R stub for the pcor function. This does some rudimentary
# argument type checking and then hands off to the C stub.

pcor <- function(
  data                         # input numerical matrix
, distance   = FALSE           # Return the distance matrix instead of the correlation coefficients
, caching_   = "mmeachflush"   # getOption("ffcaching")
, filename_  = NULL            # tempfile(pattern = pattern, tmpdir = getOption("fftempdir"))

)
  {

    # Load the "ff" package in case is not already loaded. Warn user in case the package is missing
    if( !require("ff", quietly=TRUE) ) {
        warning("Function pcor was unable to execute - failed to load package \"ff\". Please check that the package is installed and try again.")
        return(NA)
    }

    # we only work on doubles
    vmode_ = "double"

    # determine filename and finalizer
    # if user choses to work on a temporary file it will be deleted whenn all
    # references to the ff object are closed

    if (is.null(filename_)){
      # delete if temporary ff object
      finalizer_<- "delete"
    } else {
      finalizer_<- "close"
    }
     
    if (is.null(filename_)){
      # temporary ff object
      filename_<- tempfile(pattern =  "ff" , tmpdir = getwd())
    }

    # determine length of the result correlation matrix
  
    height = dim(data)[2]
    
    if (is.matrix(data) && is.numeric(data))
      length_ <- height * height
    else
      stop("PCOR only accepts numeric matrices")

    if (is.null(caching_))
      caching_<- getOption("ffcaching")
    else
      caching_<- match.arg(caching_, c("mmnoflush", "mmeachflush"))

    # Check the value of the "distance" option
    if ( (!is.logical(distance)) || (length(distance)>1) ) {
        warning(paste("Value of option \"distance\" must be a scalar logical (TRUE of FALSE). You supplied : ", distance))
        return(FALSE)
    }

    # Call C interface
    return_val <- .Call("pcor", data, filename_, distance)

    # Return values from the interface have meaning.
    #  0    -->     success
    # -1    -->     MPI is not initialized
    # -2    -->     Only the master process exists, no workers
    if ( return_val == 0 ) {
      # Open result binary file and return as ff object
      result = ff(
        dim=c(height,height)
        , filename=filename_
        , vmode=vmode_
        , caching=caching_
        , finalizer=finalizer_
        , length=length_
        )
    } else {

        if ( return_val == -1 )
            warning(paste("MPI is not initialized. Function is aborted.\n"))
        if ( return_val == -2 )
            warning(paste("No worker processes exist. Function pcor() is aborted.\n"))
        result <- FALSE
    }

    return(result)
}

