% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GuanTestGrid.R
\name{GuanTestGrid}
\alias{GuanTestGrid}
\title{Nonparametric Test of Isotropy Using the Sample Semivariogram}
\usage{
GuanTestGrid(spdata, delta = 1, lagmat = rbind(c(1, 0), c(0, 1), c(1, 1),
  c(-1, 1)), A = rbind(c(1, -1, 0, 0), c(0, 0, 1, -1)), df = 2,
  window.dims = c(2, 2), pt.est.edge = TRUE, sig.est.edge = TRUE,
  sig.est.finite = TRUE)
}
\arguments{
\item{spdata}{An \eqn{n \times 3} matrix. The first two columns provide \eqn{(x,y)} spatial coordinates. The third column provides data values at the coordinates.}

\item{delta}{A scalar indicating the distance between grid locations. Defaults to 1 (integer grid) and assumes equal spacing between locations in the x and y directions.}

\item{lagmat}{A \eqn{k \times 2} matrix of spatial lags. Each row corresponds to a lag of the form \eqn{(x.lag, y.lag)} for which the semivariogram value will be estimated. The scale of the lags provided in 'lagmat' are in units of \code{delta}. See details for more information.}

\item{A}{A \eqn{d \times k} contrast matrix. The contrasts correspond to contrasts of the estimated semivariogram at the lags given in \code{lagmat}.}

\item{df}{A scalar indicating the row rank of the matrix \code{A}. This value gives the degrees of freedom for the asymptotic \eqn{\chi^2} distribution used to compute the p-value.}

\item{window.dims}{A vector of length two corresponding to the width and height (in number of columns and rows, respectively) of the moving windows used to estimate the asymptotic variance-covariance matrix. If window width does not evenly divide the number of columns of spatial data, some data will be ommited during subsampling, i.e., function does not handle partial windows. Same applies to window height and number of rows of spatial data.}

\item{pt.est.edge}{Logical. \code{TRUE} corrects for edge effects in the point estimate (see Guan et. al. (2004), Section 4.2.1 for details).}

\item{sig.est.edge}{Logical. Defaults to \code{TRUE}, which corrects for edge effects when estimating the semivariogram in the moving windows (see Guan et. al. (2004), Section 4.2.1 for details).}

\item{sig.est.finite}{Logical. Defaults to \code{TRUE}, which provides a finite sample correction in estimating \eqn{\Sigma}, the asymptotic variance-covariance matrix. (see Guan et. al. (2004) Section 3.2.1, Equation 5). False provides the empirical variance-covariance matrix of sample semivariogram values computed via the moving windows.}
}
\value{
\item{gamma.hat}{A matrix of the spatial lags provided and the semivariogram point estimates,\eqn{\hat{\gamma}()}, at the lags used to construct the test statistic.}
\item{sigma.hat}{The estimate of asymptotic variance-covariance matrix, \eqn{\widehat{\Sigma}}, used to construct the test statistic.}
\item{n.subblocks}{The number of subblocks created by the moving window used to estimate \eqn{\Sigma}.}
\item{test.stat}{The calculated test statistic.}
\item{pvalue.finite}{The approximate, finite-sample adjusted p-value computed by using the subblocks created by the moving windows (see Guan et. al. (2004), Section 3.3 for details).}
\item{pvalue.chisq}{The p-value computed using the asymptotic \eqn{\chi^2} distribution.}
}
\description{
This function performs the nonparametric test of isotropy using the sample semivariogram from Guan et. al. (2004) for spatial data with sampling locations on a grid. See Guan et. al. (2004) for more details.
}
\details{
This function currently only supports square and rectangular sampling regions and does not currently support partial blocks. For example, suppose the sampling grid contains 20 columns and 30 rows of data. Then an ideal value of \code{window.dims} would be (2,3) since its entries evenly divide the number of columns (20) and rows (30), respectively, of data. To preserve the spatial dependence structure, the moving window should have the same shape (i.e. square or rectangle) and orientation as the entire sampling domain.

The parameter \code{delta} serves to scale the samplng locations to the integer grid. Thus the lags provided in \code{lagmat} are automatically scaled by \code{delta} by the function. For example, suppose spatial locations are observed on grid boxes of 0.5 degrees by 0.5 degrees and referenced by longitude and latitude coordinates in degrees. Then, \code{delta} should be 0.5 and a spatial lag of (0,1) corresponds to a change in coordinates of (0, 0.5), i.e., moving one sampling location north in the y-direction.
}
\examples{
library(mvtnorm)
set.seed(1)
#number of rows and columns
nr <- 12
nc <- 18
n <- nr*nc
#Set up the coordinates
coords <- expand.grid(0:(nr-1), 0:(nc-1))
coords <- cbind(coords[,2], coords[,1])
#compute the distance between sampling locations
D <- as.matrix(dist(coords))
#Set parameter values for exponential covariance function
sigma.sq <- 1
tau.sq <- 0.0
phi <- 1/4
R <- sigma.sq * exp(-phi*D)
R <- R + diag(tau.sq, nrow = n, ncol = n)
#Simulate Gaussian spatial data
z <- rmvnorm(1,rep(0,n), R, method = c("chol"))
z <-  z-mean(z)
z <- t(z)
mydata <- cbind(coords, z)
mylags <-  rbind(c(1,0), c(0, 1), c(1, 1), c(-1,1))
myA <-  rbind(c(1, -1, 0 , 0), c(0, 0, 1, -1))
tr <- GuanTestGrid(mydata, delta = 1, mylags, myA, df = 2, window.dims = c(3,2), 
pt.est.edge = TRUE, sig.est.edge = TRUE, sig.est.finite = TRUE )
tr

#Simulate data from anisotropic covariance function
aniso.angle <- pi/4
aniso.ratio <- 2
coordsA <- coords.aniso(coords, c(aniso.angle, aniso.ratio))
Da <- as.matrix(dist(coordsA))
R <- sigma.sq * exp(-phi*Da)
R <- R + diag(tau.sq, nrow = n, ncol = n)
z <- rmvnorm(1,rep(0,n), R, method = c("chol"))
z <-  z-mean(z)
z <- t(z)
mydata <- cbind(coords, z)
#Run the test on the data generated from an anisotropic covariance function
tr <- GuanTestGrid(mydata, delta = 1, mylags, myA, df = 2, window.dims = c(3,2), 
pt.est.edge = TRUE,sig.est.edge = TRUE, sig.est.finite = TRUE)
tr
}
\references{
Guan, Y., Sherman, M., & Calvin, J. A. (2004). A nonparametric test for spatial isotropy using subsampling. \emph{Journal of the American Statistical Association}, 99(467), 810-821.
}
\seealso{
\code{\link{GuanTestUnif}} \code{\link{MaityTest}}
}
\keyword{external}

