###########################################################################
# Statistics for Microarray Analysis
# plot.spatial
#
# Date : March 19, 2001
#
# History:
#    March 19, 2001: The spatial plot functions from Rarray.R.
#					
#
# Authors: Sandrine Dudoit and Yee Hwa (Jean) Yang.
##########################################################################

########################################################################/**
# \name{plot.spatial}
#
# \alias{plot.spatial}
# \alias{draw.image.func}
# \alias{spatial.func}
#
# \title{Spatial Representation of Microarray Spot Statistics}
#
#
# \description{Creates an image of shades
# of gray or colors, that represents the values of a statistic for each
# spot on the array. The statistic can be a log intensity ratio, quality
# information such as spot size or shape, or a t-statistic. This function
# can be used to explore whether there are any spatial effects in the data.
# }
#
# \usage{
# plot.spatial(x, layout, crit1=0.15, crit2=crit1, ...)
# }
#
# \arguments{
# \item{x}{a numerical vector. This vector can contain any spot 
# statistics, such
# as log intensity ratios, spot sizes or shapes, or t-statistics.} 
#
# \item{layout}{a list specifying the dimensions of the spot matrix
# and the grid matrix. This can be generated by calling 
# \code{\link{init.grid}}.}
#
# \item{crit1}{the number of values from x to be displayed on the image. 
# If crit1 < 1, the crit1*100\% spots with the largest x values are displayed.
# If crit1 >= 1, the crit1 spots with the largest x values are displayed.}    
#
# \item{crit2}{the number of values from x to be displayed on the
# image. If crit2 < 1, the crit2*100\% spots with the largest x values
# are displayed. If crit2 >= 1, the crit2 spots with the largest x
# values are displayed.}    
# 
# \item{\dots}{graphical parameters may also be supplied as arguments 
# to the function (see \code{\link{par}}).} 
# }
#
# \value{
# An image is created on the current graphics device.
# }
#
# \details{The values that didn't meet the criteria are not shown on the image.
# The image follows the layout of an actual microarray slide with the
# top left corner representing the spot (1,1,1,1).}
#
#
#
# \note{\code{\link{draw.image.func}} and \code{\link{spatial.func}}
#  are called by \code{\link{plot.spatial}} and are not typically 
# used on their own.}
#
# \author{
#  Yee Hwa Yang, \email{yeehwa@stat.berkeley.edu} \cr
#  Sandrine Dudoit, \email{sandrine@stat.berkeley.edu}}
#
# \seealso{\code{\link{draw.image.func}}, \code{\link{init.grid}}, 
# \code{\link{spatial.func}}, \code{\link{image}}.} 
#
# \examples{
# data(MouseArray)
# # mouse.setup <- init.grid()
# # mouse.data <- init.data() ## see \emph{init.data}
#
# mouse.lratio <- stat.ma(mouse.data, mouse.setup)
# plot.spatial(mouse.lratio$M[,1], mouse.setup) ## default 85% cutoff
#
# # Looking for areas where the spots are not quite circular
# plot.spatial(mouse1[,"shape"], mouse.setup, crit1=0.1)
# }   
#
# \keyword{microarray, spatial.}
#*/#########################################################################

plot.spatial <- function(x, layout, crit1=0.05, crit2=crit1,  ...)
{
  if (crit1 >= 1) crit1 <- crit1 / (length.na(x) - sum(is.na(x)))
  if (crit2 >= 1) crit2 <- crit2 / (length.na(x) - sum(is.na(x)))
  
#  if(crit < 1)
#    tmpind <- x > quantile.na(x, 1-crit)
#  if(crit >=1)
  tmpind <- (x > quantile.na(x, probs=1-crit2)) | (x < quantile.na(x, probs=crit1))

  n <- layout$ngrid.c * layout$ngrid.r * layout$nspot.c * layout$nspot.r

  if (length(as.vector(x)) == n){
    fullm <- x
    fullm[!tmpind] <- NA
  }
  if ((length(as.vector(x)) != n) & (!is.null(names(x)))){
    y <- x[tmpind]; fullm <- rep(NA, n)
    fullm[as.integer(names(y))] <- y
  }
  if ((length(as.vector(x)) != n) & (is.null(names(x)))){
    stop(" Error: Length of vector is different from total number of spots and vector has no row.name.\n")
  }
  draw.image.func(spatial.func(fullm, layout), layout, ...)
}


##########################################################################
# Internal functions called by plot.spatial
##########################################################################

spatial.func <- function(fullm, layout)
{
  gc <- layout$ngrid.c; gr <- layout$ngrid.r
  sc <- layout$nspot.c; sr <- layout$nspot.r
  grid <- split(fullm, rep(1:(gc*gr) , rep(sc*sr, gc*gr)))
  grid1 <- lapply(grid, matrix, nrow=sr, ncol=sc, byrow=TRUE)
  grid2 <- split(unlist(grid1), rep(1:gr, rep(sc*sr*gc, gr)))
  grid3 <- lapply(grid2, matrix, nrow=sr)
  full <- NULL
  for(i in 1:gr){
    full <- rbind(full, grid3[[i]])}
  full
}

draw.image.func <- function (x, layout, axes = FALSE, array.grid = TRUE, label = FALSE, ...) 
{
    gc <- layout$ngrid.c
    gr <- layout$ngrid.r
    sc <- layout$nspot.c
    sr <- layout$nspot.r
    print(summary(as.vector(x)))
    image(1:ncol(x), 1:nrow(x), t(apply(x, 2, rev)), axes = axes, 
        xlab = "", ylab = "", ...)
    if (label) {
        axis(2, ...)
        axis(3, ...)
    }
    box()
    if (array.grid) {
        abline(h = ((gr - 1):1) * (sr) + 0.5)
        abline(v = (1:(gc - 1)) * (sc) + 0.5)
    }
}

############################################################################
#                              End of file
############################################################################
