#' Plot League Dashboard
#'
#' Given the league ID, generate a dashboard with various
#' plots/tables of the current league. Note that since this
#' is a dashboard, your R session will continuously run until
#' the console is stopped. Stoppage can be achieved by pressing
#' Control + C on the keyboard.
#'
#' @return Returns a dashboard of the current league state.
#' @author Nick Bultman, \email{njbultman74@@gmail.com}, March 2024
#' @keywords league dashboard plot
#' @importFrom shiny fluidPage tabsetPanel tabPanel titlePanel hr br fluidRow column h2 h1 numericInput actionButton eventReactive runApp
#' @importFrom shinyWidgets setBackgroundColor
#' @importFrom htmltools HTML
#' @importFrom shinycssloaders withSpinner
#' @importFrom DT dataTableOutput renderDataTable
#' @importFrom plotly plotlyOutput renderPlotly
#' @export
#' @examples
#' \dontrun{plot_league_dashboard(688281863499907072)}
#'
#' @param league_id League ID generated by Sleeper (numeric).
#'
plot_league_dashboard <- function(league_id) {
  # Obtain master plotting data frame from league ID
  master_df <- get_main_data(league_id)
  # If nothing is returned for master data frame, return nothing
  # Ensure dashboard not created by having additional NULL check
  if (is.null(master_df)) {
    stop("Dashboard generation halted due to invalid league ID.")
  }
  # A message already informs user of error in get_main_data function
  # If data frame returned, store certain strings in variables
  league_name <- master_df$name[1]
  league_status <- master_df$status[1]
  league_sport <- toupper(master_df$sport[1])
  league_type <- master_df$season_type[1]
  league_id <- master_df$league_id[1]
  league_season <- master_df$season[1]
  # Create UI
  ui <- shiny::fluidPage(
    # Set background color
    shinyWidgets::setBackgroundColor("#213251"),
    # Create a series of tabs
    shiny::tabsetPanel(
      # Create first tab (overview)
      shiny::tabPanel("Overview", fluid = TRUE,
        # Display league overview name
        shiny::titlePanel(shiny::h1(htmltools::HTML(paste0("<b>",league_name, " League Overview</b>")), align = "center", style = "color:white")), # nolint
        # Display white horizontal line
        shiny::hr(style = "color:white"),
        # Line break
        shiny::br(),
        # Create fluid row to hold general league attributes
        shiny::fluidRow(
          # Display league sport
          shiny::column(width = 3,
            shiny::h2(htmltools::HTML(paste0("<b>Sport: ", league_sport, "</b>")), align = "center", style = "color:white")), # nolint
          # Display league season
          shiny::column(width = 3,
            shiny::h2(htmltools::HTML(paste0("<b>Season: ", league_season, "</b>")), align = "center", style = "color:white")), # nolint
          # Display season type
          shiny::column(width = 3,
            shiny::h2(htmltools::HTML(paste0("<b>Season Type: ", league_type, "</b>")), align = "center", style = "color:white")), # nolint
          # Display season status
          shiny::column(width = 3,
            shiny::h2(htmltools::HTML(paste0("<b>Season Status: ", league_status, "</b>")), align = "center", style = "color:white")) # nolint
        ),
        # Line break
        shiny::br(),
        # Create fluid row to display league streaks
        shiny::fluidRow(
          # Display league streaks table
          shiny::column(width = 12,
            shinycssloaders::withSpinner(DT::dataTableOutput(outputId = "league_streaks_table")) # nolint
          )
        ),
        # Line break
        shiny::br(),
        # Create fluid row to display league streaks
        shiny::fluidRow(
          # Create columns that hold titles for plots/tables to come
          shiny::column(width = 6,
            shiny::h2(htmltools::HTML("<b>Regular Season Rankings</b>"), align = "center", style = "color:white")), # nolint
          shiny::column(width = 6,
            shiny::h2(htmltools::HTML("<b>League Information</b>"), align = "center", style = "color:white")), # nolint
        ),
        # Create fluid row to hold plots and tables
        shiny::fluidRow(
          # Display regular season ranking plot
          shiny::column(width = 6,
            shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "regular_season_rankings_plot")) # nolint
          ),
          # Display league information table
          shiny::column(width = 6,
            shinycssloaders::withSpinner(DT::dataTableOutput(outputId = "league_information_table")) # nolint
          )
        )
      ),
      # Create second tab (team analysis)
      shiny::tabPanel(title = "Team Analysis",
        # Display team analysis title
        shiny::titlePanel(shiny::h1(htmltools::HTML(paste0("<b>Team Analysis</b>")), align = "center", style = "color:white")), # nolint
        # Display white horizontal line
        shiny::hr(style = "color:white"),
        # Line break
        shiny::br(),
        # Create a fluid row to hold the label for the display name input button
        shiny::fluidRow(
          shiny::column(width = 12,
                        shiny::h2(htmltools::HTML(paste0("<b>Display Name Selection</b>")), style = "color:white")) # nolint
        ),
        # Create a fluid row to hold display name input button
        shiny::fluidRow(
          shiny::column(width = 12,
                        shiny::selectInput(inputId = "displaynameselection", # nolint
                                           label = "",
                                           choices = sort(unique(master_df$display_name)), # nolint
                                           selected = sort(unique(master_df$display_name))[1]) # nolint
          )
        ),
        # Create a fluid row to hold the labels for total fantasy points plots
        shiny::fluidRow(
          shiny::column(width = 6,
                        shiny::h2(htmltools::HTML(paste0("<b>Total Fantasy Points For</b>")), align = "center", style = "color:white") # nolint
          ),
          shiny::column(width = 6,
                        shiny::h2(htmltools::HTML(paste0("<b>Total Fantasy Points Against</b>")), align = "center", style = "color:white") # nolint
          )
        ),
        # Create a fluid row to hold total fantasy points plots
        shiny::fluidRow(
          # Display total fantasy points for plot
          shiny::column(width = 6,
                        shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "user_fantasy_points_for_plot")) # nolint  
          ),
          # Display total fantasy points against plot
          shiny::column(width = 6,
                        shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "user_fantasy_points_against_plot")) # nolint  
          )
        ),
        # Create a fluid row to hold the labels for team analysis plots
        shiny::fluidRow(
          shiny::column(width = 6,
                        shiny::h2(htmltools::HTML(paste0("<b>Matchups by Week</b>")), align = "center", style = "color:white") # nolint
          ),
          shiny::column(width = 6,
                        shiny::h2(htmltools::HTML(paste0("<b>Player Points by Week</b>")), align = "center", style = "color:white") # nolint
          )
        ),
        # Create a fluid row to hold team analysis plots
        shiny::fluidRow(
          # Display fantasy points differential plot
          shiny::column(width = 6,
                        shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "user_weekly_matchups_plot")) # nolint  
          ),
          # Display user waiver budget plot
          shiny::column(width = 6,
                        shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "user_weekly_players_plot")) # nolint  
          )
        ),
        # Create a fluid row to hold the labels for team analysis plots
        shiny::fluidRow(
          shiny::column(width = 6,
                        shiny::h2(htmltools::HTML(paste0("<b>Fantasy Points Differential</b>")), align = "center", style = "color:white") # nolint
          ),
          shiny::column(width = 6,
                        shiny::h2(htmltools::HTML(paste0("<b>Waiver Budget Status</b>")), align = "center", style = "color:white") # nolint
          )
        ),
        # Create a fluid row to hold team analysis plots
        shiny::fluidRow(
          # Display fantasy points differential plot
          shiny::column(width = 6,
                        shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "user_fantasy_points_differential_plot")) # nolint  
          ),
          # Display user waiver budget plot
          shiny::column(width = 6,
                        shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "user_waiver_budget_plot")) # nolint  
          )
        )
      ),
      # Create third tab (trending players)
      shiny::tabPanel("Trending Players", fluid = TRUE,
        # Display trending players title
        shiny::titlePanel(shiny::h1(htmltools::HTML(paste0("<b>Trending Players</b>")), align = "center", style = "color:white")), # nolint
        # Display white horizontal line
        shiny::hr(style = "color:white"),
        # Line break
        shiny::br(),
        # Create a fluid row to hold the labels for the input buttons
        shiny::fluidRow(
          shiny::column(width = 3,
                        shiny::h2(htmltools::HTML(paste0("<b>Lookback Hours</b>")), style = "color:white")), # nolint
          shiny::column(width = 3,
          shiny::h2(htmltools::HTML(paste0("<b>Player Limit</b>")), style = "color:white")), # nolint
        ),
        # Create a fluid row to hold trending players inputs
        shiny::fluidRow(
          # Lookback hours input
          shiny::column(width = 3,
                        shiny::numericInput(inputId = "trendingplayerslookback_hours", # nolint
                                            label = "",
                                            value = 24,
                                            min = 1),
                        shiny::actionButton(inputId = "gentrendplayers",
                                            label = "Calculate")),
          # Player limit input
          shiny::column(width = 3,
                        shiny::numericInput(inputId = "trendingplayerslimit",
                                            label = "",
                                            value = 10,
                                            min = 1,
                                            max = 50))
        ),
        # Create a fluid row to hold trending players plot
        shiny::fluidRow(
          # Display trending players plot
          shiny::column(width = 12,
                        shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "trending_players_plot")) # nolint  
          )
        ),
        # Create a fluid row to hold trending players table
        shiny::fluidRow(
          # Display trending players table
          shiny::column(width = 12,
                        shinycssloaders::withSpinner(DT::dataTableOutput(outputId = "trending_players_table")) # nolint  
          )
        )
      ),
      # Create fourth tab (player information)
      shiny::tabPanel("Player Information", fluid = TRUE,
        # Display player information title
        shiny::titlePanel(shiny::h1(htmltools::HTML(paste0("<b>Player Information</b>")), align = "center", style = "color:white")), # nolint
        # Display white horizontal line
        shiny::hr(style = "color:white"),
        # Line break
        shiny::br(),
        # Create a fluid row to hold the labels for the plots
        shiny::fluidRow(
          shiny::column(width = 6,
                        shiny::h2(htmltools::HTML(paste0("<b>Players by High School State</b>")), align = "center", style = "color:white") # nolint
          ),
          shiny::column(width = 6,
                        shiny::h2(htmltools::HTML(paste0("<b>Players by College</b>")), align = "center", style = "color:white") # nolint
          )
        ),
        # Create a fluid row to hold player information plots
        shiny::fluidRow(
          # Players by high school state plot
          shiny::column(width = 6,
                        shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "players_by_high_school_state")) # nolint  
          ),
          # Players by college cplot
          shiny::column(width = 6,
                        shinycssloaders::withSpinner(plotly::plotlyOutput(outputId = "players_by_college")) # nolint 
          )
        ),
        # Create a fluid row to hold the label for general player information
        shiny::fluidRow(
          shiny::column(width = 12,
                        shiny::h2(htmltools::HTML(paste0("<b>General Player Information</b>")), align = "center", style = "color:white")), # nolint
        ),
        # Create a fluid row to hold general player information table
        shiny::fluidRow(
          shiny::column(width = 12,
                        shinycssloaders::withSpinner(DT::dataTableOutput(outputId = "general_players_table")) # nolint  
          )
        )
      )
    )
  )
  # Create server
  server <- function(input, output, session) {
    # Render league streaks table
    output$league_streaks_table <- DT::renderDataTable({
      plot_league_streaks_table(league_id, font_color = "#fff")
    })
    # Render regular season league rankings plot
    output$regular_season_rankings_plot <- plotly::renderPlotly({
      plot_regular_season_rankings(league_id, title = "", tick_color = "#fff")
    })
    # Render user waiver budget plot
    output$user_waiver_budget_plot <- plotly::renderPlotly({
      plot_user_waiver_budget(league_id,
                              display_name = input$displaynameselection,
                              title = "",
                              tick_color = "#fff",
                              budget_total_line_color = "#fff")
    })
    # Render user fantasy points for plot
    output$user_fantasy_points_for_plot <- plotly::renderPlotly({
      plot_user_fantasy_points_for(league_id,
                                   display_name = input$displaynameselection,
                                   title = "",
                                   tick_color = "#fff")
    })
    # Render user fantasy points against plot
    output$user_fantasy_points_against_plot <- plotly::renderPlotly({
      plot_user_fantasy_points_against(league_id,
                                       display_name = input$displaynameselection, # nolint
                                       title = "",
                                       tick_color = "#fff")
    })
    # Render user fantasy points differential plot
    output$user_fantasy_points_differential_plot <- plotly::renderPlotly({
      plot_user_fantasy_points_differential(league_id,
                                       display_name = input$displaynameselection, # nolint
                                       title = "",
                                       tick_color = "#fff")
    })
    # Render user weekly matchups plot
    output$user_weekly_matchups_plot <- plotly::renderPlotly({
      plot_user_weekly_matchups(league_id,
                                display_name = input$displaynameselection, # nolint
                                title = "",
                                tick_color = "#fff")
    })
    # Render user weekly player points plot
    output$user_weekly_players_plot <- plotly::renderPlotly({
      plot_user_weekly_players(league_id,
                                display_name = input$displaynameselection, # nolint
                                title = "",
                                tick_color = "#fff")
    })
    # Render league information table
    output$league_information_table <- DT::renderDataTable({
      plot_league_information_table(league_id, font_color = "#fff")
    })
    # Render trending plot only after user clicks on action button
    lookback_hours <- shiny::eventReactive(input$gentrendplayers, {
      input$trendingplayerslookback_hours
    })
    # Store trending player limit only after user clicks on action button
    limit <- shiny::eventReactive(input$gentrendplayers, {
      input$trendingplayerslimit
    })
    # Render trending players plot only after user clicks on action button
    output$trending_players_plot <- plotly::renderPlotly({
      plot_trending_players(lookback_hours = lookback_hours(), # nolint
                            limit = limit(),
                            title = "",
                            tick_color = "fff")
    })
    # Render trending players table
    output$trending_players_table <- DT::renderDataTable({
      plot_trending_players_table(lookback_hours = lookback_hours(),
                                  limit = limit(),
                                  font_color = "#fff")
    })
    # Render players by high school state plot
    output$players_by_high_school_state <- plotly::renderPlotly({
      plot_nfl_player_high_school_state(title = "")
    })
    # Render players by college plot
    output$players_by_college <- plotly::renderPlotly({
      plot_nfl_top_colleges(title = "",
                            tick_color = "#fff")
    })
    # Render general players table
    output$general_players_table <- DT::renderDataTable({
      plot_nfl_player_data_table(font_color = "#fff")
    })
  }
  shiny::runApp(list(ui = ui, server = server), launch.browser = TRUE)
}