% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sjTabLinReg.R
\name{sjt.lmer}
\alias{sjt.lmer}
\title{Summary of linear mixed effects models as HTML table}
\usage{
sjt.lmer(..., pred.labels = NULL, depvar.labels = NULL,
  remove.estimates = NULL, group.pred = FALSE, p.numeric = TRUE,
  emph.p = TRUE, p.zero = FALSE, p.kr = TRUE, separate.ci.col = TRUE,
  newline.ci = TRUE, show.est = TRUE, show.std = NULL, show.ci = TRUE,
  show.se = FALSE, show.header = FALSE, show.col.header = TRUE,
  show.r2 = TRUE, show.icc = TRUE, show.re.var = TRUE,
  show.fstat = FALSE, show.aic = FALSE, show.aicc = FALSE,
  show.dev = FALSE, string.pred = "Predictors",
  string.dv = "Dependent Variables", string.interc = "(Intercept)",
  string.obs = "Observations", string.est = "B", string.std = "std. Beta",
  string.ci = "CI", string.se = "std. Error", string.p = "p",
  ci.hyphen = "&nbsp;&ndash;&nbsp;", minus.sign = "&#45;", digits.est = 2,
  digits.std = 2, digits.p = 3, digits.ci = 2, digits.se = 2,
  digits.summary = 3, cell.spacing = 0.2, cell.gpr.indent = 0.6,
  sep.column = TRUE, CSS = NULL, encoding = NULL, file = NULL,
  use.viewer = TRUE, no.output = FALSE, remove.spaces = TRUE)
}
\arguments{
\item{...}{one or more fitted linear (mixed) models.}

\item{pred.labels}{character vector with labels of predictor variables.
If not \code{NULL}, \code{pred.labels} will be used in the first
table column with the predictors' names. If \code{NULL}, variable
labels are set based on label attributes (see \code{\link[sjmisc]{get_label}}).
If \code{pred.labels = ""}, column names (vector names) are used
as predictor labels. See 'Examples'.}

\item{depvar.labels}{character vector with labels of dependent 
variables of all fitted models. See 'Examples'.}

\item{remove.estimates}{numeric vector with indices (order equals to row index of \code{coef(fit)}) 
or character vector with coefficient names that indicate which estimates should be removed
from the table output. The first estimate is the intercept, followed by the model predictors.
\emph{The intercept cannot be removed from the table output!} \code{remove.estimates = c(2:4)} 
would remove the 2nd to the 4th estimate (1st to 3rd predictor after intercept) from the output. 
\code{remove.estimates = "est_name"} would remove the estimate \emph{est_name}. Default 
is \code{NULL}, i.e. all estimates are printed.}

\item{group.pred}{logical, if \code{TRUE} (default), automatically groups table rows with 
factor levels of same factor, i.e. predictors of type \code{\link{factor}} will
be grouped, if the factor has more than two levels. Grouping means that a separate headline
row is inserted to the table just before the predictor values.}

\item{p.numeric}{logical, if \code{TRUE}, the p-values are printed 
as numbers. If \code{FALSE} (default), asterisks are used.}

\item{emph.p}{logical, if \code{TRUE} (default), significant p-values are shown bold faced.}

\item{p.zero}{logical, if \code{TRUE}, p-values have a leading 0 before the
period (e.g. \emph{0.002}), else p-values start with a period and
without a zero (e.g. \emph{.002}).}

\item{p.kr}{logical, if \code{TRUE}, p-value estimation is based on conditional 
F-tests with Kenward-Roger approximation for the df. Caution: Computation
may take very long time for large samples!}

\item{separate.ci.col}{if \code{TRUE}, the CI values are shown in a separate table column.
Default is \code{FALSE}.}

\item{newline.ci}{logical, if \code{TRUE} and \code{separate.ci.col = FALSE}, inserts a line break
between estimate and CI values. If \code{FALSE}, CI values are printed in the same
line as estimate values.}

\item{show.est}{logical, if \code{TRUE} (default), the estimates are printed.}

\item{show.std}{indicates whether standardized beta-coefficients should 
also printed, and if yes, which type of standardization is done.
See 'Details'.}

\item{show.ci}{logical, if \code{TRUE} (default), the confidence intervall is also printed to the table. Use
\code{FALSE} to omit the CI in the table.}

\item{show.se}{logical, if \code{TRUE}, the standard errors are also printed.
Default is \code{FALSE}.}

\item{show.header}{logical, if \code{TRUE}, the header strings \code{string.pred}
and \code{string.dv} are shown. By default, they're hidden.}

\item{show.col.header}{logical, if \code{TRUE} (default), the table data columns have a headline with 
abbreviations for estimates, std. beta-values, confidence interval and p-values.}

\item{show.r2}{logical, if \code{TRUE} (default), the R2 and adjusted R2 values for each model are printed
in the model summary. For linear mixed models, the R2 and Omega-squared values are printed
(see \code{\link[sjstats]{r2}} for details).}

\item{show.icc}{logical, if \code{TRUE}, the intra-class-correlation for each 
model is printed in the model summary. Only applies to mixed models.}

\item{show.re.var}{logical, if \code{TRUE}, the variance parameters for the random
effects for each model are printed in the model summary. Only applies to mixed models.
For details output, see 'Note' in \code{\link[sjstats]{icc}}.}

\item{show.fstat}{If \code{TRUE}, the F-statistics for each model is printed
in the model summary. Default is \code{FALSE}. This argument does not apply to
\code{\link{sjt.lmer}}.}

\item{show.aic}{logical, if \code{TRUE}, the AIC value for each model is printed
in the model summary. Default is \code{FALSE}.}

\item{show.aicc}{logical, if \code{TRUE}, the second-order AIC value for each model 
is printed in the model summary. Default is \code{FALSE}.}

\item{show.dev}{logical, if \code{TRUE}, the deviance for each model 
is printed in the model summary.}

\item{string.pred}{character vector,used as headline for the predictor column.
Default is \code{"Predictors"}.}

\item{string.dv}{character vector, used as headline for the 
dependent variable columns. Default is \code{"Dependent Variables"}.}

\item{string.interc}{character vector, used as headline for the Intercept row.
Default is \code{"Intercept"}.}

\item{string.obs}{character vector, used in the summary row for the count of observation
(cases). Default is \code{"Observations"}.}

\item{string.est}{character vector, used for the column heading of estimates.}

\item{string.std}{character vector, used for the column heading of standardized beta coefficients. Default is \code{"std. Beta"}.}

\item{string.ci}{character vector, used for the column heading of confidence interval values. Default is \code{"CI"}.}

\item{string.se}{character vector, used for the column heading of standard error values. Default is \code{"std. Error"}.}

\item{string.p}{character vector, used for the column heading of p values. Default is \code{"p"}.}

\item{ci.hyphen}{character vector, indicating the hyphen for confidence interval range.
May be an HTML entity. See 'Examples'.}

\item{minus.sign}{string, indicating the minus sign for negative numbers.
May be an HTML entity. See 'Examples'.}

\item{digits.est}{amount of decimals for estimates}

\item{digits.std}{amount of decimals for standardized beta}

\item{digits.p}{amount of decimals for p-values}

\item{digits.ci}{amount of decimals for confidence intervals}

\item{digits.se}{amount of decimals for standard error}

\item{digits.summary}{amount of decimals for values in model summary}

\item{cell.spacing}{numeric, inner padding of table cells. By default, this value is 0.2 (unit is cm), which is
suitable for viewing the table. Decrease this value (0.05 to 0.1) if you want to import the table
into Office documents. This is a convenient argument for the \code{CSS} argument for changing
cell spacing, which would be: \code{CSS = list(css.thead = "padding:0.2cm;", css.tdata = "padding:0.2cm;")}.}

\item{cell.gpr.indent}{indent for table rows with grouped factor predictors. Only applies
if \code{group.pred = TRUE}.}

\item{sep.column}{logical, if \code{TRUE}, an empty table column is added after 
each model column, to add margins between model columns. By default, this
column will be added to the output; however, when copying tables to 
office applications, it might be helpful not to add this separator column
when modifying the table layout.}

\item{CSS}{\code{\link{list}}-object with user-defined style-sheet-definitions, according to the 
\href{http://www.w3.org/Style/CSS/}{official CSS syntax}. See 'Details'.}

\item{encoding}{string, indicating the charset encoding used for variable and 
value labels. Default is \code{NULL}, so encoding will be auto-detected 
depending on your platform (e.g., \code{"UTF-8"} for Unix and \code{"Windows-1252"} for
Windows OS). Change encoding if specific chars are not properly displayed (e.g. German umlauts).}

\item{file}{destination file, if the output should be saved as file.
If \code{NULL} (default), the output will be saved as temporary file and 
openend either in the IDE's viewer pane or the default web browser.}

\item{use.viewer}{If \code{TRUE}, the HTML table is shown in the IDE's viewer pane. If
\code{FALSE} or no viewer available, the HTML table is opened in a web browser.}

\item{no.output}{logical, if \code{TRUE}, the html-output is neither opened in a browser nor shown in
the viewer pane and not even saved to file. This option is useful when the html output
should be used in \code{knitr} documents. The html output can be accessed via the return
value.}

\item{remove.spaces}{logical, if \code{TRUE}, leading spaces are removed from all lines in the final string
that contains the html-data. Use this, if you want to remove parantheses for html-tags. The html-source
may look less pretty, but it may help when exporting html-tables to office tools.}
}
\value{
Invisibly returns
         \itemize{
           \item the web page style sheet (\code{page.style}),
           \item the web page content (\code{page.content}),
           \item the complete html-output (\code{output.complete}) and
           \item the html-table with inline-css for use with knitr (\code{knitr})
           }
           for further use.
}
\description{
Summarizes (multiple) fitted linear mixed effects models 
               (estimates, std. beta values etc.)  as HTML table, 
               or saves them as file. The fitted models may have different 
               predictors, e.g. when comparing different stepwise fitted models.
}
\details{
Concerning the \code{show.std} argument, \code{show.std = "std"}
           will print normal standardized estimates. For \code{show.std = "std2"},
           however, standardization of estimates follows 
           \href{http://www.stat.columbia.edu/~gelman/research/published/standardizing7.pdf}{Gelman's (2008)}
           suggestion, rescaling the estimates by dividing them by two standard 
           deviations instead of just one. Resulting coefficients are then 
           directly comparable for untransformed binary predictors. This type 
           of standardization uses the \code{\link[arm]{standardize}}-function
           from the \pkg{arm}-package.
           For backward compatibility reasons, \code{show.std} also may be 
           a logical value; if \code{TRUE}, normal standardized estimates are 
           printed (same effect as \code{show.std = "std"}). Use 
           \code{show.std = NULL} (default) or \code{show.std = FALSE},
           if standardized estimats should not be printed.
           \cr \cr
           Computation of p-values (if necessary and if \code{p.kr = TRUE}) are based 
           on conditional F-tests with Kenward-Roger approximation for the df, using 
           the \pkg{pbkrtest}-package. If \pkg{pbkrtest} is not available or
           \code{p.kr = FALSE}, computation of p-values is based 
           on normal-distribution assumption, treating the t-statistics as Wald
           z-statistics. See 'Details' in \code{\link[sjstats]{get_model_pval}}.
           \cr \cr
           The confidence intervals stem from \pkg{broom}'s 
           \code{\link[broom]{tidy}}-function. For linear mixed models, the computation
           method is "Wald" (\code{lme4::confint.merMod(fit, method = "Wald")}).
           \cr \cr
           Furthermore, see 'Details' in \code{\link{sjt.frq}}.
}
\note{
The variance components of the random parts (see \code{show.re.var}) are
        denoted like:
        \itemize{
         \item within-group variance: sigma-squared
         \item between-group-variance: tau-zero-zero
         \item random-slope-intercept-correlation: rho-zero-one
         }
}
\examples{
\dontrun{
library(lme4)
library(sjmisc)
data(efc)

# prepare group variable
efc$grp = as.factor(efc$e15relat)
levels(x = efc$grp) <- get_labels(efc$e15relat)
efc$care.level <- sjmisc::rec(efc$n4pstu, 
                              recodes = "0=0;1=1;2=2;3:4=3", 
                              as.fac = TRUE)
levels(x = efc$care.level) <- c("none", "I", "II", "III")

# data frame for fitted model
mydf <- data.frame(neg_c_7 = efc$neg_c_7,
                   sex = efc$c161sex,
                   c12hour = efc$c12hour,
                   barthel = efc$barthtot,
                   education = to_factor(efc$c172code),
                   grp = efc$grp,
                   carelevel = efc$care.level)
                   
# fit three sample models
fit1 <- lmer(neg_c_7 ~ sex + c12hour + barthel + (1|grp), data = mydf)
fit2 <- lmer(neg_c_7 ~ sex + c12hour + education + barthel + (1|grp), data = mydf)
fit3 <- lmer(neg_c_7 ~ sex + c12hour + education + barthel + 
              (1|grp) + (1|carelevel), data = mydf)

# print summary table... automatic grouping does not work here,
# barthel-index is printed as category of education (values are
# correct, however, indentation is wrong)
sjt.lmer(fit1, fit2, ci.hyphen = " to ", group.pred = TRUE)

# either change order of models
sjt.lmer(fit2, fit1, group.pred = TRUE)
# or turn off automatic grouping of categorical predictors
sjt.lmer(fit1, fit2, group.pred = FALSE)

# print table, using vector names as labels
sjt.lmer(fit1, fit2, fit3, pred.labels = "")

# show other statistics
sjt.lmer(fit1, fit2, show.aic = TRUE, show.ci = FALSE, 
         show.se = TRUE, p.numeric = FALSE)
           
sjt.lmer(fit1, fit2, fit3, show.aic = TRUE, 
         separate.ci.col = FALSE, newline.ci = FALSE)

# user defined predictor labels
sjt.lmer(fit1, fit2, fit3, pred.labels = c("Elder's gender (female)",
         "Hours of care per week", "Barthel Index", "Educational level (mid)",
         "Educational level (high)"))}
                  
}
\seealso{
\href{http://strengejacke.de/sjPlot/sjt.lmer/}{sjPlot manual: sjt.lmer} 
           and \code{\link{sjt.lm}} for further examples.
}

