#' @title Linear fit function.
#'
#' @param dataInput a data frame composed of two columns. One is for time other is for intensity. Should be normalized data generated by normalizeData.
#' @param tryCounter the number of times the data is fit via maximum likelihood.
#' @param startList  the initial set of parameters that the fitting algorithm tries, these parameters are the slope and the y intercept.
#' @param lowerBounds the lower bounds for the randomly generated start parameters, these parameters are the slope and the y intercept.
#' @param upperBounds the upper bounds for the randomly generated start parameters, these parameters are the slope and the y intercept.
#' @param min_Factor minimum step size used by the fitting algorithm.
#' @param n_iterations maximum number of iterations used by the fitting algorithm.
#'
#' @description Fits a linear model to a given data by using likelihood maximization (LM) and gives the parameters (slope and intersection) describing the line as output. It also provides information about the goodness of fit such as AIC, BIC values, residual sum of squares, and log likelihood.
#' @return Returns fitted parameters for lineFit. The slope, intersection, and goodness of fit metrics.
#' @export
#'
#' @examples
#'time=seq(3,24,0.5)
#'
#'#intensity with Noise
#'noise_parameter=.2
#'intensity_noise=stats::runif(n = length(time),min = 0,max = 1)*noise_parameter
#'intensity=lineFitFormula(time, slope=4, intersection=-2)
#'intensity=intensity+intensity_noise
#'
#'dataInput=data.frame(intensity=intensity,time=time)
#'normalizedInput = normalizeData(dataInput)
#'parameterVector<-lineFitFunction(normalizedInput,tryCounter=2)
#'
#'#Check the results
#'if(parameterVector$isThisaFit){
#'  intensityTheoretical=lineFitFormula(time,
#'                                      slope=parameterVector$slope_Estimate,
#'                                      intersection=parameterVector$intersection_Estimate)
#'
#'  comparisonData=cbind(dataInput,intensityTheoretical)
#'
#'  print(parameterVector$residual_Sum_of_Squares)
#'
#'  require(ggplot2)
#'  ggplot(comparisonData)+
#'    geom_point(aes(x=time, y=intensity))+
#'    geom_line(aes(x=time,y=intensityTheoretical))+
#'    expand_limits(x = 0, y = 0)}
#'
#'if(!parameterVector$isThisaFit){print(parameterVector)}

lineFitFunction<-function(dataInput,
                          tryCounter,
                          startList=list(slope=0,intersection=1),
                          lowerBounds=c(-100,-1000),
                          upperBounds=c(100,1000),
                          min_Factor=1/2^20,
                          n_iterations=500)
{

  isalist=(is.list(dataInput) & !is.data.frame(dataInput))
  if(isalist){dataFrameInput=dataInput$timeIntensityData}
  isadataframe=(is.data.frame(dataInput))
  if(isadataframe){dataFrameInput=dataInput}

  if(tryCounter==1){counterDependentStartList=startList}
  if(tryCounter!=1){
    randomVector=stats::runif(length(startList), 0, 1)
    names(randomVector)<-c("slope","intersection")
    counterDependentStartVector=randomVector*(upperBounds-lowerBounds)+lowerBounds
    counterDependentStartList=as.list(counterDependentStartVector)}


  theFitResult <- try(minpack.lm::nlsLM(intensity ~ sicegar::lineFitFormula(time, slope, intersection),
                                        dataFrameInput,
                                        start=counterDependentStartList,
                                        control = list(maxiter = n_iterations, minFactor = min_Factor),
                                        lower = lowerBounds,
                                        upper = upperBounds,
                                        trace=F),silent = TRUE)


  if(class(theFitResult)!="try-error")
  {
    parameterMatrix=summary(theFitResult)$parameters
    colnames(parameterMatrix)<-c("Estimate","Std_Error","t_value","Pr_t")


    parameterVector=c(t(parameterMatrix))
    names(parameterVector)<- c("slope_N_Estimate","slope_Std_Error","slope_t_value","slope_Pr_t",
                               "intersection_N_Estimate","intersection_Std_Error","intersection_t_value","intersection_Pr_t")

    parameterVector<-c(parameterVector,
                       residual_Sum_of_Squares=sum((as.vector(stats::resid(theFitResult)))^2)[1],
                       log_likelihood=as.vector(stats::logLik(theFitResult))[1],
                       AIC_value=as.vector(stats::AIC(theFitResult))[1],
                       BIC_value=as.vector(stats::BIC(theFitResult))[1])

    parameterList=as.list(parameterVector)
    parameterList$isThisaFit=TRUE
    parameterList$startVector=counterDependentStartList
    if(isalist){parameterList$dataScalingParameters=as.list(dataInput$dataScalingParameters)}
    parameterList$model="linaer"

    parameterDf=as.data.frame(parameterList)
    #Renormalize Parameters
    parameterDf=linearRenormalizeParameters(parameterDf,isalist)
  }

  if(class(theFitResult)=="try-error")
  {
    parameterVector=rep(NA, 24)
    names(parameterVector)<- c("slope_N_Estimate","slope_Std_Error","slope_t_value","slope_Pr_t",
                               "intersection_N_Estimate","intersection_Std_Error","intersection_t_value","intersection_Pr_t")
    parameterVector<-c(parameterVector,
                       residual_Sum_of_Squares=Inf,
                       log_likelihood=NA,
                       AIC_value=NA,
                       BIC_value=NA)

    parameterList=as.list(parameterVector)
    parameterList$isThisaFit=FALSE
    parameterList$startVector=counterDependentStartList
    if(isalist){parameterList$dataScalingParameters=as.list(dataInput$dataScalingParameters)}
    parameterList$model="linear"

    parameterDf=as.data.frame(parameterList)
    #Renormalize Parameters
    parameterDf=linearRenormalizeParameters(parameterDf,isalist)
  }


  return(parameterDf)

}
#**************************************


#**************************************
#' @title Line fit formula.
#
#' @param x the "time" (time) column of the dataframe.
#' @param slope the slope of the line.
#' @param intersection the intensity intersection point of the line when time is zero in the time-intensity graph.
#'
#' @description Calculates intesities for given time points (x) by using line fit model and parameters (slope and intersection).
#' @return Returns the predicted intensities for given time points in the line fit model for given slope and intersection values.
#
#' @examples
#'time=seq(3,24,0.5)
#'
#'#simulate intensity data and add noise
#'noise_parameter=.2
#'intensity_noise=stats::runif(n = length(time),min = 0,max = 1)*noise_parameter
#'intensity=lineFitFormula(time, slope=4, intersection=-2)
#'intensity=intensity+intensity_noise
#'
#'dataInput=data.frame(intensity=intensity,time=time)
#'normalizedInput = normalizeData(dataInput)
#'parameterVector<-lineFitFunction(normalizedInput,tryCounter=2)
#'
#'#Check the results
#'if(parameterVector$isThisaFit){
#'  intensityTheoretical=lineFitFormula(time,
#'                                      slope=parameterVector$slope_Estimate,
#'                                      intersection=parameterVector$intersection_Estimate)
#'
#'  comparisonData=cbind(dataInput,intensityTheoretical)
#'
#'  print(parameterVector$residual_Sum_of_Squares)
#'
#'  require(ggplot2)
#'  ggplot(comparisonData)+
#'    geom_point(aes(x=time, y=intensity))+
#'    geom_line(aes(x=time,y=intensityTheoretical))+
#'    expand_limits(x = 0, y = 0)}
#'
#'if(!parameterVector$isThisaFit){print(parameterVector)}
#'
#'
#' @export
lineFitFormula<-function(x, slope, intersection){
  y=slope*x+intersection;
  return(y)}
#**************************************


#**************************************
# @title linearRenormalizeParameters (This is an internal function)
# @param parametersDf it is the parameter data frame generated by line fit function
#        includes the parameters named
#        *slope_N_Estimate (normalzied Slope Estimate)
#        *intersection_N_Estimate (normalized Intersection Estimate)
#        *dataScalingParameters.intensityRatio the range of initial unnormalized intensity. Provieded if the data is normalized
#        *parameterDF$dataScalingParameters.intensityMin the minimum of unnormalized intensity. Provieded if the data is normalized
#        *parameterDF$dataScalingParameters.timeRatio tha maximum time that the experiment reach. Provieded if the data is normalized
# @param isalist defines if the input is provided as a list (i.e normalized) or as a data frame (i.e not normalized)
# @details If the fit was done in normalized data frame then the found fit parameters will belong to normalized data.
#          This function generates unnormalized counterparts of those parameters
linearRenormalizeParameters<-function(parameterDF,isalist)
{
  if(isalist){
    parameterDF$intersection_Estimate=parameterDF$intersection_N_Estimate*parameterDF$dataScalingParameters.intensityRatio+parameterDF$dataScalingParameters.intensityMin
    parameterDF$slope_Estimate=parameterDF$slope_N_Estimate*parameterDF$dataScalingParameters.intensityRatio/parameterDF$dataScalingParameters.timeRatio
  }
  if(!isalist){
    parameterDF$intersection_Estimate=parameterDF$intersection_N_Estimate
    parameterDF$slope_Estimate=parameterDF$slope_N_Estimate
  }
  return(parameterDF)
}
#**************************************
