#' Map 2 stratified rasters
#'
#' @description Map stratified rasters to a combined stratification.
#'
#' @family stratify functions
#'
#' @inheritParams strat_breaks
#' @inheritParams strat_poly
#' @param sraster spatRaster. Primary stratification raster.
#' @param sraster2 spatRaster. Secondary stratification raster.
#' @param stack Logical. Default = \code{FALSE}. If \code{TRUE}, output raster will be
#' 3 layers: \code{strata, strata2, stratamapped}.
#' @param details Logical. If \code{FALSE} (default) output is a mapped stratified spatRaster object.
#' If \code{TRUE} return a list where \code{$outRaster} is the mapped stratified raster, and
#' \code{$lookUp} is the lookup table for the stratification.
#'
#' @section Mapping:
#' The mapping algorithm will take the stratification from \code{sraster} and combine it with
#' overlying strata values in \code{sraster2}. This will result in a \code{stratamapped} attribute
#' where the values from both inputs are combined.
#'
#' i.e.
#'
#' If \code{strata = 1} and \code{strata2 = 1} then \code{stratamapped = 11}.
#'
#' If \code{strata = 2} and \code{strata2 = 14} then \code{stratamapped = 214}.
#'
#' @examples
#' #--- load input metrics raster ---#
#' raster <- system.file("extdata", "sraster.tif", package = "sgsR")
#' sraster <- terra::rast(raster)
#'
#' #--- read polygon coverage ---#
#' poly <- system.file("extdata", "inventory_polygons.shp", package = "sgsR")
#' fri <- sf::st_read(poly)
#'
#' #--- stratify polygon coverage ---#
#' #--- specify polygon attribute to stratify ---#
#'
#' attribute <- "NUTRIENTS"
#'
#' #--- specify features within attribute & how they should be grouped ---#
#' #--- as a single vector ---#
#'
#' features <- c("poor", "rich", "medium")
#'
#' srasterfri <- strat_poly(
#'   poly = fri,
#'   attribute = attribute,
#'   features = features,
#'   raster = sraster
#' )
#'
#' #--- map srasters ---#
#' strat_map(
#'   sraster = srasterfri,
#'   sraster2 = sraster
#' )
#'
#' strat_map(
#'   sraster = srasterfri,
#'   sraster2 = sraster,
#'   stack = TRUE,
#'   details = TRUE
#' )
#' @importFrom methods is
#'
#' @return A spatRaster object.
#'
#' @author Tristan R.H. Goodbody, Robert Hijmans
#'
#' @export


strat_map <- function(sraster,
                      sraster2,
                      stack = FALSE,
                      filename = NULL,
                      overwrite = FALSE,
                      plot = FALSE,
                      details = FALSE
                      ) {

  #--- global variables ---#
  strata <- strata2 <- value <- NULL

  #--- error handling ---#

  if (!inherits(sraster, "SpatRaster")) {
    stop("'sraster' must be type SpatRaster.", call. = FALSE)
  }

  if (!inherits(sraster2, "SpatRaster")) {
    stop("'sraster2' must be type SpatRaster.", call. = FALSE)
  }

  if (!is.logical(stack)) {
    stop("'stack' must be type logical.", call. = FALSE)
  }

  if (!is.logical(overwrite)) {
    stop("'overwrite' must be type logical.", call. = FALSE)
  }

  if (!is.logical(plot)) {
    stop("'plot' must be type logical.", call. = FALSE)
  }

  if (!is.logical(details)) {
    stop("'details' must be type logical.", call. = FALSE)
  }

  #--- error handling for raster inputs ---#

  if (terra::nlyr(sraster) > 1) {
    stop("'sraster' must only contain 1 layer. Please subset the layer you would like to use for mapping.", call. = FALSE)
  }

  if (terra::nlyr(sraster2) > 1) {
    stop("'sraster2' must only contain 1 layer. Please subset the layer you would like to use for mapping.", call. = FALSE)
  }

  if (!grepl("strata", names(sraster))) {
    stop("A layer name containing 'strata' does not exist within 'sraster'.", call. = FALSE)
  }

  if (!grepl("strata", names(sraster2))) {
    stop("A layer name containing 'strata' does not exist within 'sraster2'.", call. = FALSE)
  }

  #--- check that extents and resolutions of sraster and sraster2 match ---#

  if (isFALSE(terra::compareGeom(sraster, sraster2, stopOnError = FALSE))) {
    stop("Extents of 'sraster' and 'sraster2' do not match.", call. = FALSE)
  }
  
  if (isFALSE(terra::compareGeom(sraster, sraster2, stopOnError = FALSE, ext = FALSE, res = TRUE))) {
    stop("Spatial resolutions of 'sraster' and 'sraster2' do not match.", call. = FALSE)
  }

  #--- map stratification rasters ---#

  joined <- c(sraster, sraster2)
  names(joined) <- c("strata", "strata2")

  featuresJoin <- terra::values(joined, dataframe = TRUE)

  oclass <- featuresJoin %>%
    dplyr::group_by(strata, strata2) %>%
    #--- ensure NA's are transfered ---#
    dplyr::mutate(stratamapped = ifelse(is.na(strata) | is.na(strata2), NA, paste0(strata,strata2)))
  
  if(is.character(oclass$strata) || is.character(oclass$strata2)){
    
    oclass$stratamapped <- as.character(oclass$stratamapped)
    
  } else if(is.factor(oclass$strata) || is.factor(oclass$strata2)){
    
    oclass$stratamapped <- as.character(oclass$stratamapped)
    
  } else {
    
    oclass$stratamapped <- as.integer(oclass$stratamapped)
    
  }

  #--- create lookUp table ---#

  lookUp <- dplyr::distinct(oclass) %>%
    stats::na.omit() %>%
    as.data.frame()
  
  #--- set newly stratified values ---#

  rout <- terra::setValues(sraster, oclass$stratamapped)
  names(rout) <- "strata"

  if (isTRUE(stack)) {
    message("Stacking sraster, sraster2, and their combination (stratamapped).")

    #--- stack 3 rasters if requested ---#

    routstack <- c(sraster, sraster2, rout)
    names(routstack) <- c("strata", "strata2", "stratamapped")
  }

  #--- if not stacking rename for output ---#

  if (exists("routstack")) {
    rout <- routstack
  }


  #--- plot if requested

  if (isTRUE(plot)) {
    terra::plot(rout)
  }

  #--- write file to disc ---#

  if (!is.null(filename)) {
    
    if (!is.character(filename)) {
      stop("'filename' must be type character.", call. = FALSE)
    }

    #--- write file to disc depending on whether 'stack' was specified ---#

    if (isTRUE(stack)) {
      terra::writeRaster(x = routstack, filename = filename, overwrite = overwrite)
      message("Output stack written to disc.")
    } else {
      terra::writeRaster(x = rout, filename = filename, overwrite = overwrite)
      message("Output raster written to disc.")
    }
  }

  #--- output details if desired ---#

  if (isTRUE(details)) {

    #--- output metrics details along with stratification raster ---#

    output <- list(raster = rout, lookUp = lookUp)

    #--- output samples dataframe ---#

    return(output)
  } else {

    #--- just output raster ---#

    return(rout)
  }
}
