% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est_change.R
\name{est_change}
\alias{est_change}
\title{Standardized Case Influence on Parameter Estimates}
\usage{
est_change(rerun_out, parameters = NULL)
}
\arguments{
\item{rerun_out}{The output from \code{\link[=lavaan_rerun]{lavaan_rerun()}}.}

\item{parameters}{A character vector to specify the selected
parameters. Each parameter is named as in \code{lavaan} syntax, e.g.,
\code{x ~ y} or \code{x ~~ y}, as appeared in the columns \code{lhs}, \code{op}, and \code{rhs}
in the output of \code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}}.
Supports specifying an operator to select all parameters with this
operators: \code{~}, \verb{~~}, \verb{=~}, and \code{~1}. This vector can contain
both parameter names and operators. More details can be found
in the help of \code{\link[=pars_id]{pars_id()}}.
If omitted or \code{NULL}, the
default, changes on all free parameters will be computed.}
}
\value{
An \code{est_change}-class object, which is
matrix with the number of columns equals to the number of
requested parameters plus one, the last column being the
generalized Cook's
distance. The number of rows equal to the number
of cases. The row names are the case identification values used in
\code{\link[=lavaan_rerun]{lavaan_rerun()}}. The elements are the standardized difference.
Please see Pek and MacCallum (2011), Equation 7.
A print method is available for user-friendly output.
}
\description{
Gets a \code{\link[=lavaan_rerun]{lavaan_rerun()}} output and computes the
standardized changes in selected parameters for each case
if included.
}
\details{
For each case, \code{\link[=est_change]{est_change()}} computes the differences in
the estimates of selected parameters with and without this case:

(Estimate with all case) - (Estimate without this case).

The
differences are standardized by dividing the raw differences by
their standard errors (Pek & MacCallum, 2011).
This is a measure of the standardized influence of a case on
the parameter
estimates if it is included.

If the value of a case is positive, including the case increases an estimate.

If the value of a case is negative, including the case decreases an estimate.

If the analysis is not admissible or does not converge when a case
is deleted, \code{NA}s will be turned for this case on the differences.

Unlike \code{\link[=est_change_raw]{est_change_raw()}}, \code{\link[=est_change]{est_change()}} does not support
computing the standardized changes of standardized estimates.

It will also compute generalized Cook's distance (\emph{gCD}), proposed by
Pek and MacCallum (2011) for structural equation modeling.
Only the parameters selected (all free parameters, by default)
will be used in computing \emph{gCD}.

Since version 0.1.4.8, if (a) a model has one or more equality
constraints, and
(b) some selected parameters are linearly dependent or constrained
to be equal due to the constraint(s), \emph{gCD} will be computed
by removing parameters such that the remaining parameters are
not linearly dependent nor constrained to be equal.
(Support for equality constraints and
linearly dependent parameters available in 0.1.4.8 and later version).

Supports both single-group and multiple-group models.
(Support for multiple-group models available in 0.1.4.8 and later version).
}
\examples{
library(lavaan)

# A path model

dat <- pa_dat
mod <-
"
m1 ~ a1 * iv1 + a2 * iv2
dv ~ b * m1
a1b := a1 * b
a2b := a2 * b
"
# Fit the model
fit <- lavaan::sem(mod, dat)
summary(fit)
# Fit the model several times. Each time with one case removed.
# For illustration, do this only for four selected cases
fit_rerun <- lavaan_rerun(fit, parallel = FALSE,
                          to_rerun = c(2, 4, 7, 9))
# Compute the standardized changes in parameter estimates
# if a case is included vs. if this case is excluded.
# That is, case influence on parameter estimates, standardized.
out <- est_change(fit_rerun)
# Case influence:
out
# Note that these are the differences divided by the standard errors
# The rightmost column, `gcd`, contains the
# generalized Cook's distances (Pek & MacCallum, 2011).
out[, "gcd", drop = FALSE]

# Compute the changes for the paths from iv1 and iv2 to m1
out2 <- est_change(fit_rerun, c("m1 ~ iv1", "m1 ~ iv2"))
# Case influence:
out2
# Note that only the changes in the selected parameters are included.
# The generalized Cook's distance is computed only from the selected
# parameter estimates.

# A CFA model

dat <- cfa_dat
mod <-
"
f1 =~  x1 + x2 + x3
f2 =~  x4 + x5 + x6
f1 ~~ f2
"
# Fit the model
fit <- lavaan::cfa(mod, dat)

# Examine four selected cases
fit_rerun <- lavaan_rerun(fit, parallel = FALSE,
                          to_rerun = c(2, 3, 5, 7))
# Compute the standardized changes in parameter estimates
# if a case is included vs. if a case is excluded.
# That is, case influence on parameter estimates, standardized.
# For free loadings only
out <- est_change(fit_rerun, parameters = "=~")
out

# A latent variable model

dat <- sem_dat
mod <-
"
f1 =~  x1 + x2 + x3
f2 =~  x4 + x5 + x6
f3 =~  x7 + x8 + x9
f2 ~   a * f1
f3 ~   b * f2
ab := a * b
"
# Fit the model
fit <- lavaan::sem(mod, dat)

# Examine four selected cases
fit_rerun <- lavaan_rerun(fit, parallel = FALSE,
                          to_rerun = c(2, 3, 5, 7))
# Compute the changes in parameter estimates if a case is included
# vs. if a case is excluded.
# That is, standardized case influence on parameter estimates.
# For structural paths only
out <- est_change(fit_rerun, parameters = "~")
out

}
\references{
Pek, J., & MacCallum, R. (2011). Sensitivity analysis
in structural equation models: Cases and their influence.
\emph{Multivariate Behavioral Research, 46}(2), 202-228.
doi:10.1080/00273171.2011.561068
}
\author{
Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}.
}
