% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lda.R
\name{textmodel_lda}
\alias{textmodel_lda}
\title{Unsupervised Latent Dirichlet allocation}
\usage{
textmodel_lda(
  x,
  k = 10,
  max_iter = 2000,
  auto_iter = FALSE,
  alpha = 0.5,
  beta = 0.1,
  gamma = 0,
  model = NULL,
  batch_size = 1,
  verbose = quanteda_options("verbose")
)
}
\arguments{
\item{x}{the dfm on which the model will be fit.}

\item{k}{the number of topics.}

\item{max_iter}{the maximum number of iteration in Gibbs sampling.}

\item{auto_iter}{if \code{TRUE}, stops Gibbs sampling on convergence before
reaching \code{max_iter}. See details.}

\item{alpha}{the values to smooth topic-document distribution.}

\item{beta}{the values to smooth topic-word distribution.}

\item{gamma}{a parameter to determine change of topics between sentences or
paragraphs. When \code{gamma > 0}, Gibbs sampling of topics for the current
document is affected by the previous document's topics.}

\item{model}{a fitted LDA model; if provided, \code{textmodel_lda()} inherits
parameters from an existing model. See details.}

\item{batch_size}{split the corpus into the smaller batches (specified in
proportion) for distributed computing; it is disabled when a batch include
all the documents \code{batch_size = 1.0}. See details.}

\item{verbose}{logical; if \code{TRUE} print diagnostic information during
fitting.}
}
\value{
\code{textmodel_seededlda()} and \code{textmodel_lda()} returns a list of model
parameters. \code{theta} is the distribution of topics over documents; \code{phi} is
the distribution of words over topics. \code{alpha} and \code{beta} are the small
constant added to the frequency of words to estimate \code{theta} and \code{phi},
respectively, in Gibbs sampling. Other elements in the list subject to
change.
}
\description{
Implements unsupervised Latent Dirichlet allocation (LDA). Users can run
Seeded LDA by setting \code{gamma > 0}.
}
\details{
If \code{auto_iter = TRUE}, the iteration stops even before \code{max_iter}
when \code{delta <= 0}. \code{delta} is computed to measure the changes in the number
of words whose topics are updated by the Gibbs sampler in every 100
iteration as shown in the verbose message.

If \code{batch_size < 1.0}, the corpus is partitioned into sub-corpora of
\code{ndoc(x) * batch_size} documents for Gibbs sampling in sub-processes with
synchronization of parameters in every 10 iteration. Parallel processing is
more efficient when \code{batch_size} is small (e.g. 0.01). The algorithm is the
Approximate Distributed LDA proposed by Newman et al. (2009). User can
changed the number of sub-processes used for the parallel computing via
\code{options(seededlda_threads)}.

\code{set.seed()} should be called immediately before \code{textmodel_lda()} or
\code{textmodel_seededlda()} to control random topic assignment. If the random
number seed is the same, the serial algorithm produces identical results;
the parallel algorithm produces non-identical results because it
classifies documents in different orders using multiple processors.

To predict topics of new documents (i.e. out-of-sample), first, create a
new LDA model from a existing LDA model passed to \code{model} in
\code{textmodel_lda()}; second, apply \code{\link[=topics]{topics()}} to the new model. The \code{model}
argument takes objects created either by \code{textmodel_lda()} or
\code{textmodel_seededlda()}.
}
\examples{
\donttest{
require(seededlda)
require(quanteda)

corp <- head(data_corpus_moviereviews, 500)
toks <- tokens(corp, remove_punct = TRUE, remove_symbols = TRUE, remove_number = TRUE)
dfmt <- dfm(toks) \%>\%
    dfm_remove(stopwords("en"), min_nchar = 2) \%>\%
    dfm_trim(max_docfreq = 0.1, docfreq_type = "prop")

lda <- textmodel_lda(dfmt, k = 6, max_iter = 500) # 6 topics
terms(lda)
topics(lda)
}
}
\references{
Newman, D., Asuncion, A., Smyth, P., & Welling, M. (2009). Distributed
Algorithms for Topic Models. The Journal of Machine Learning Research, 10,
1801–1828.
}
\seealso{
\link[topicmodels:lda]{LDA} \link[keyATM:weightedLDA]{weightedLDA}
}
\keyword{textmodel}
