\name{rsad}
\alias{rsad}

\title{Poisson and negative binomial sampling of a species abundance distribution
}

\description{
  A given number of realizations of a probability distribution (species
  abundances in a community) is sampled by a Poisson or Negative Binomial process. 
}

\usage{
rsad(S, frac, sad, Pois.samp = TRUE, k , zeroes = FALSE, ssize=1, \dots)
}

\arguments{
  \item{S}{
    positive integer; number of species in the community, which is the number of random deviates
    generated by the probability distribution given by argument \code{sad}
  }
  \item{frac}{
    single numeric \code{0 < frac <= 1}; fraction of the community sampled
  }
  \item{sad}{
    character; root name of community sad distribution - e.g., \kbd{lnorm}
    for the lognormal distribution \code{rlnorm}; \kbd{geom} for the geometric distribution
	\code{rgeom}.
	Note that at the present moment only the following distributions used in fitsad have 
	random generators implemented:
	\kbd{gamma}, \kbd{geom}, \kbd{lnorm}, \kbd{rnbinom}, \kbd{rpoilog}, \kbd{rweibull},
	however you can use any base name for which a random sample generating function is available.
  }  
  \item{Pois.samp}{
    logical; if \kbd{TRUE} the sampling process is Poisson (independent sampling of
    individuals); if \kbd{FALSE} negative binomial
    sampling is used to simulate aggregation of individuals in sampling units.
  }
  \item{k}{
    positive; size parameter for the sampling binomial negative.
  }
  \item{zeroes}{
    logical; should zero values be included in the returned vector?
  }
  \item{ssize}{
    positive integer; sample size: number of draws taken from the community.
  }
  \item{\dots}{
    named arguments to be passed to the probability function defined by the
    argument sad.
  }
}

\details{
  This function simulates a random sample taken from a community with \code{S}
  species. The expected species abundances in the sampled community follow a
  probability distribution given by the argument \code{sad}. A fraction \code{frac} of
  the community is sampled, thus the expected abundance in the sample of each
  species is \code{frac*n}, where n is the species' expected abundance in the
  community.

  Two sampling processes can be simulated: Poisson,
  where individuals are sampled independently, and negative binomial,
  where individuals are aggregated over sampling units. 

  In general terms, this function takes a Poisson or negative
  binomial sampling with replacement of a vector of \code{S} realizations of a random variable,
  with the sampling intensity given by \code{frac}. The resulting values are
  realizations of a Poisson (or a Negative Binomial) random variable where the
  parameter that corresponds to the mean (=expected value of the variable) follows a probability
  distribution given by the argument \code{sad}.
}

\value{
  if \kbd{ssize=1} a vector of (zero truncated) abundances in the sample;
  if \kbd{ssize>1} a data frame with sample identification, species identification, and (zero truncated) abundances.
}

\references{

  Pielou, E.C. 1977. \emph{Mathematical Ecology}. New York: John Wiley
  and Sons.
  
  Green,J. and Plotkin, J.B. 2007 A statistical theory for sampling species abundances. \emph{Ecology Letters 10}:1037--1045
}

\author{
  Paulo I. Prado \email{prado@ib.usp.br}.
}


\seealso{
  \code{\link{dpoix}}, \code{\link{dpoig}} and \code{\link{dpoilog}} for
  examples of compound Poisson probability distributions like those
  simulated by \code{rsad}.
}

\examples{

##A Poisson sample from a community with a lognormal sad
samp2 <- rsad(S = 100, frac=0.1, sad="lnorm", meanlog=5, sdlog=2)
## Preston plot
plot(octav(samp2))
## Once this is a Poisson sample of a lognormal community, the abundances
## in the sample should follow a Poisson-lognormal distribution.
## Adds line of theoretical Poisson-lognormal with
## mu=meanlog+log(frac) and sigma=sdlog)
## Predicted by the theoretical Poisson-lognormal truncated at zero
samp2.pred <- octavpred(samp2, sad="poilog", coef= list(mu=5+log(0.1), sig=2), trunc=0)
## Adding the line in the Preston plot
lines(samp2.pred)
}
