% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rxrandom.R
\name{cvPost}
\alias{cvPost}
\title{Sample a covariance Matrix from the Posterior Inverse Wishart
distribution.}
\usage{
cvPost(
  nu,
  omega,
  n = 1L,
  omegaIsChol = FALSE,
  returnChol = FALSE,
  type = c("invWishart", "lkj", "separation"),
  diagXformType = c("log", "identity", "variance", "nlmixrSqrt", "nlmixrLog",
    "nlmixrIdentity")
)
}
\arguments{
\item{nu}{Degrees of Freedom (Number of Observations) for
covariance matrix simulation.}

\item{omega}{Either the estimate of covariance matrix or the
estimated standard deviations in matrix form each row forming
the standard deviation simulated values}

\item{n}{Number of Matrices to sample.  By default this is 1.
This is only useful when \code{omega} is a matrix.  Otherwise
it is determined by the number of rows in the input
\code{omega} matrix of standard deviations}

\item{omegaIsChol}{is an indicator of if the omega matrix is in
the Cholesky decomposition. This is only used when \code{type="invWishart"}}

\item{returnChol}{Return the Cholesky decomposition of the
covariance matrix sample. This is only used when \code{type="invWishart"}}

\item{type}{The type of covariance posterior that is being
simulated.  This can be:
\itemize{
\item \code{invWishart} The posterior is an inverse wishart; This allows
for correlations between parameters to be modeled.  All the
uncertainty in the parameter is captured in the degrees of freedom
parameter.
\item \code{lkj} The posterior separates the standard deviation
estimates (modeled outside and provided in the \code{omega}
argument) and the correlation estimates. The correlation estimate
is simulated with the \code{\link[=rLKJ1]{rLKJ1()}}.  This simulation uses
the relationship \code{eta=(nu-1)/2}.  This is relationship based
on the proof of the relationship between the restricted
LKJ-distribution and inverse wishart distribution (XXXXXX).  Once
the correlation posterior is calculated, the estimated standard
deviations are then combined with the simulated correlation matrix
to create the covariance matrix.
\item \code{separation} Like the \code{lkj} option, this separates out
the estimation of the correlation and standard deviation.  Instead
of using the \code{LKJ} distribution to simulate the correlation,
it simulates the inverse wishart of the identity matrix and
converts the result to a correlation matrix.  This correlation
matrix is then used with the standard deviation to calculate the
simulated covariance matrix.
}}

\item{diagXformType}{Diagonal transformation type.  These could be:
\itemize{
\item \code{log} The standard deviations are log transformed, so the
actual standard deviations are exp(omega)
\item \code{identity} The standard deviations are not transformed. The
standard deviations are not transformed;  They should be positive.
\item \code{variance} The variances are specified in the \code{omega}
matrix; They are transformed into standard deviations.
\item \code{nlmixrSqrt} These standard deviations come from an nlmixr
omega matrix where diag(chol(inv(omega))) = x^2
\item \code{nlmixrLog} These standard deviations come from a nlmixr
omega matrix omega matrix where diag(chol(solve(omega))) = exp(x)
\item \code{nlmixrIdentity} These standard deviations come from a nlmixr
omega matrix omega matrix where diag(chol(solve(omega))) = x
}

The nlmixr transformations only make sense when there is no
off-diagonal correlations modeled.}
}
\value{
a matrix (n=1) or a list of matrices  (n > 1)
}
\description{
Note this Inverse wishart rescaled to match the original scale of
the covariance matrix.
}
\details{
If your covariance matrix is a 1x1 matrix, this uses an scaled
inverse chi-squared which is equivalent to the Inverse Wishart
distribution in the uni-directional case.

In general, the separation strategy is preferred for diagonal
matrices.  If the dimension of the matrix is below 10, \code{lkj}
is numerically faster than \code{separation} method.  However, the
\code{lkj} method has densities too close to zero (XXXX) when the
dimension is above 10.  In that case, though computationally more
expensive \code{separation} method performs better.

For matrices with modeled covariances, the easiest method to use
is the inverse Wishart which allows the simulation of correlation
matrices (XXXX).  This method is more well suited for well behaved
matrices, that is the variance components are not too low or too
high.  When modeling nonlinear mixed effects modeling matrices
with too high or low variances are considered sub-optimal in
describing a system.  With these rules in mind, it is reasonable
to use the inverse Wishart.
}
\examples{

## Sample a single covariance.
draw1 <- cvPost(3, matrix(c(1, .3, .3, 1), 2, 2))

## Sample 3 covariances
set.seed(42)
draw3 <- cvPost(3, matrix(c(1, .3, .3, 1), 2, 2), n = 3)

## Sample 3 covariances, but return the cholesky decomposition
set.seed(42)
draw3c <- cvPost(3, matrix(c(1, .3, .3, 1), 2, 2), n = 3, returnChol = TRUE)

## Sample 3 covariances with lognormal standard deviations via LKJ
## correlation sample
cvPost(3, sapply(1:3, function(...) {
  rnorm(10)
}), type = "lkj")

## or return cholesky decomposition
cvPost(3, sapply(1:3, function(...) {
  rnorm(10)
}),
type = "lkj",
returnChol = TRUE
)

## Sample 3 covariances with lognormal standard deviations via separation
## strategy using inverse Wishart correlation sample
cvPost(3, sapply(1:3, function(...) {
  rnorm(10)
}), type = "separation")

## or returning the cholesky decomposition
cvPost(3, sapply(1:3, function(...) {
  rnorm(10)
}),
type = "separation",
returnChol = TRUE
)
}
\references{
Alvarez I, Niemi J and Simpson M. (2014) \emph{Bayesian Inference for a
Covariance Matrix}. Conference on Applied Statistics in Agriculture.
\url{https://newprairiepress.org/cgi/viewcontent.cgi?article=1004&context=agstatconference}

Wang1 Z, Wu Y, and Chu H. (2018) \emph{On Equivalence of the LKJ
distribution and the restricted Wishart distribution}. arXiv:1809.04746
}
\author{
Matthew L.Fidler & Wenping Wang
}
