% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/call_rtmpt.R
\name{fit_rtmpt}
\alias{fit_rtmpt}
\title{Posterior sample, diagnostics and some optional stuff}
\usage{
fit_rtmpt(
  model,
  data,
  n.chains = 4,
  n.iter = 5000,
  n.burnin = 200,
  n.thin = 1,
  Rhat_max = 1.05,
  Irep = 1000,
  prior_params = NULL,
  indices = FALSE,
  save_log_lik = FALSE,
  old_label = FALSE
)
}
\arguments{
\item{model}{A list of the class \code{rtmpt_model}.}

\item{data}{Optimally, a list of class \code{rtmpt_data}. Also possible is a \code{data.frame} or a 
path to the text file. Both, \code{data.frame} and the text file must contain the column names "subj", 
"group", "tree", "cat", and "rt" preferably but not necessarily in this order. The values of the latter must 
be in milliseconds. It is always advised to use \code{\link{to_rtmpt_data}} first, which gives back an \code{rtmpt_data} list
with informations about the changes in the data, that were needed.}

\item{n.chains}{Number of chains to use. Default is 4. Must be larger than 1 and smaller or equal to 16.}

\item{n.iter}{Number of samples per chain. Default is 5000.}

\item{n.burnin}{Number of warm-up samples. Default is 200.}

\item{n.thin}{Thinning factor. Default is 1.}

\item{Rhat_max}{Maximal Potential scale reduction factor: A lower threshold that needs to be reached before the actual sampling starts. Default is 1.05}

\item{Irep}{Every \code{Irep} samples an interim state with the current maximal potential scale reduction
factor is shown. Default is 1000. The following statements must hold true for \code{Irep}:
\itemize{
  \item \code{n.burnin} is smaller than or equal to \code{Irep},
  \item \code{Irep} is a multiple of \code{n.thin} and
  \item \code{n.iter} is a multiple of \code{Irep / n.thin}.
}}

\item{prior_params}{Named list with prior parameters. All parameters have default values, that lead to uninformative priors. Vectors are not allowed.
Allowed parameters are:
\itemize{
  \item \code{mean_of_exp_mu_beta}: This is the a priori expected exponential rate (\code{E(exp(beta)) = E(lambda)}) and 
        \code{1/mean_of_exp_mu_beta} is the a priori expected process time (\code{1/E(exp(beta)) = E(tau)}). The default
        mean is set to \code{10}, such that the expected a priori process time is \code{0.1} seconds.
  \item \code{var_of_exp_mu_beta}: The a priori group-specific variance of the exponential rates. Since
        \code{exp(mu_beta)} is Gamma distributed, the rate of the distribution is just mean divided by variance and
        the shape is the mean times the rate. The default is set to \code{100}.
  \item \code{mean_of_mu_gamma}: This is the a priori expected \emph{mean parameter} of the encoding and response execution times,
        which follow a normal distribution truncated from below at zero, so \code{E(mu_gamma) < E(gamma)}. The default is \code{0}.
  \item \code{var_of_mu_gamma}: The a priori group-specific variance of the \emph{mean parameter}. Its default is \code{10}.
  \item \code{mean_of_omega_sqr}: This is the a priori expected residual variance (\code{E(omega^2)}). Its distribution
        differs from the one used in the paper. Here it is a Gamma distribution instead of an improper one. The default
        is \code{0.005}.
  \item \code{var_of_omega_sqr}: The a priori variance of the residual variance (\code{Var(omega^2)}). The default is
        \code{0.01}. The default of the mean and variance is equivalent to a shape and rate of \code{0.0025} and 
        \code{0.5}, respectivly.
  \item \code{df_of_sigma_sqr}: A priori degrees of freedom for the individual variance of the response executions. The
        individual variance has a scaled inverse chi-squared prior with \code{df_of_sigma_sqr} degrees of freedom and
        \code{omega^2} as scale. \code{2} is the default and it should be an integer.
  \item \code{sf_of_scale_matrix_SIGMA}: The original scaling matrix (S) of the (scaled) inverse Wishart distribution for the process 
        related parameters is an identity matrix \code{S=I}. \code{sf_of_scale_matrix_SIGMA} is a scaling factor, that scales this 
        matrix (\code{S=sf_of_scale_matrix_SIGMA*I}). Its default is \code{1}.
  \item \code{sf_of_scale_matrix_GAMMA}: The original scaling matrix (S) of the (scaled) inverse Wishart distribution for the encoding and
        motor execution parameters is an identity matrix \code{S=I}. \code{sf_of_scale_matrix_GAMMA} is a scaling factor, that scales 
        this matrix (\code{S=sf_of_scale_matrix_GAMMA*I}). Its default is \code{1}.
  \item \code{prec_epsilon}: This is epsilon in the paper. It is the precision of mu_alpha and all xi (scaling parameter
        in the scaled inverse Wishart distribution). Its default is also \code{1}.
  \item \code{add_df_to_invWish}: If \code{P} is the number of parameters or rather the size of the scale matrix used in the (scaled)
        inverse Wishart distribution then \code{add_df_to_invWish} is the number of degrees of freedom that can be added to it. So
        \code{DF = P + add_df_to_invWish}. The default for \code{add_df_to_invWish} is \code{1}, such that the correlations are uniformly 
        distributed within \code{[-1, 1]}.
}}

\item{indices}{Model selection indices. If set to \code{TRUE} the log-likelihood for each iteration and trial will be stored temporarily
and with that the WAIC and LOO will be calculated via the \code{loo} package. If you want to have this log-likelihood matrix stored in the
output of this function, you can set \code{save_log_lik} to \code{TRUE}. The default for \code{indices} is \code{FALSE}.}

\item{save_log_lik}{If set to \code{TRUE} and \code{indices = TRUE} the log-likelihood matrix for each iteration and trial will
be saved in the output as a matrix. Its default is \code{FALSE}.}

\item{old_label}{If set to \code{TRUE} the old labels of "subj" and "group" of the data will be used in the elements of the output list. Default is \code{FALSE}.}
}
\value{
A list of the class \code{rtmpt_fit} containing 
  \itemize{
    \item \code{samples}: the posterior samples as an \code{mcmc.list} object,
    \item \code{diags}: some diagnostics like deviance information criterion, posterior predictive checks for the frequencies and latencies, 
                        potential scale reduction factors, and also the 99\% and 95\% HDIs and medians for the group-level parameters,
    \item \code{specs}: some model specifications like the model, arguments of the model call, and information about the data transformation,
    \item \code{indices} (optional): if enabled, WAIC and LOO,
    \item \code{LogLik} (optional): if enabled, the log-likelihood matrix used for WAIC and LOO. 
    \item \code{summary} includes posterior mean and median of the main parameters.
  }
}
\description{
Given model and data, this function calls an altered version of the C++ program by Klauer and Kellen (2018) to sample from
  the posterior distribution via a Metropolis-Gibbs sampler and storing it in an mcmc.list called \code{samples}. 
  Posterior predictive checks developed by Klauer (2010), deviance information criterion (DIC; Spiegelhalter et al., 2002),
  99\% and 95\% highest density intervals (HDI) together with the median will be provided for the main parameters in a list 
  called \code{diags}. Optionally, the \code{indices} widely applicable information criterion (WAIC; Watanabe, 2010; Vehtari et al., 2017) and 
  leave-one-out cross-validation (LOO; Vehtari et al., 2017) can be saved. Additionally the log-likelihood (\code{LogLik}) can also be stored. 
  Some specifications of the function call are also saved in \code{specs}.
}
\examples{
####################################################################################
# Detect-Guess variant of the Two-High Threshold model.
# The encoding and motor execution times are assumed to be equal for each response.
####################################################################################

mdl_2HTM <- "
# targets
do+(1-do)*g
(1-do)*(1-g)

# lures
(1-dn)*g
dn+(1-dn)*(1-g)

# do: detect old; dn: detect new; g: guess
"

model <- to_rtmpt_model(mdl_file = mdl_2HTM)

data_file <- system.file("extdata/data.txt", package="rtmpt")
data <- read.table(file = data_file, header = TRUE)
data_list <- to_rtmpt_data(raw_data = data, model = model)
\donttest{
# This might take some time
rtmpt_out <- fit_rtmpt(model = model, data = data_list, Rhat_max = 1.1)
rtmpt_out
}
# Type ?SimData for another working example.
}
\references{
Klauer, K. C. (2010). Hierarchical multinomial processing tree models: A latent-trait approach. \emph{Psychometrika, 75(1)}, 70-98. doi:\doi{10.1007/s11336-004-1188-3}

Klauer, K. C., & Kellen, D. (2018). RT-MPTs: Process models for response-time distributions based on multinomial processing trees with 
  applications to recognition memory. \emph{Journal of Mathematical Psychology, 82}, 111-130.

Spiegelhalter, D. J., Best, N. G., Carlin, B. P., & Van Der Linde, A. (2002). Bayesian measures of model complexity and fit. 
  \emph{Journal of the royal statistical society: Series b (statistical methodology), 64(4)}, 583-639.
  
Vehtari, A., Gelman, A., & Gabry, J. (2017). Practical Bayesian model evaluation using leave-one-out cross-validation and WAIC. 
  \emph{Statistics and Computing, 27(5)}, 1413-1432.

Watanabe, S. (2010). Asymptotic equivalence of Bayes cross validation and widely applicable information criterion in singular learning theory. 
  \emph{Journal of Machine Learning Research, 11(Dec)}, 3571-3594.
}
\author{
Raphael Hartmann
}
