% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/streamEdit.R
\name{streamEdit}
\alias{streamEdit}
\title{High-level wrapper for stream editing functions}
\usage{
streamEdit(commandList, stream = NULL, inFile = NULL, outFile = NULL,
  warn = FALSE)
}
\arguments{
\item{commandList}{A list that designates the insertion, replacement, substitution, or commenting commands
that will be performed on \code{stream} (or the stream obtained from \code{inFile}).
The list must have names corresponding to unique abbreviations of
"insert", "replace", "substitute", and/or "comment".  Each element in \code{commandList} must also be
a list with names and values that correspond to the arguments of \code{\link{sed_insert}}, \code{\link{sed_replace}},
\code{\link{sed_substitute}}, and/or \code{\link{sed_comment}} respectively.  See Examples.}

\item{stream}{A character vector, each element typically (but not necessarily) containing the text
from a single line in a file, which can be generated via \code{\link{readLines}}.}

\item{inFile}{A character string designating a file that is to be read (using \code{\link{readLines}}
and will become a \code{stream},
where each line of the file is mapped to a single element in the character vector \code{stream}.}

\item{outFile}{A character string designating a file to which the resulting, edited stream will be
written using \code{\link{writeLines}}. If \code{outFile = NULL}, no file is written.}

\item{warn}{A logical that, when \code{TRUE}, issues a worning if the insertion, replacement, subsitution, or
commenting cannot be performed. See Details.}
}
\value{
Invisibly returns the edited stream, and writes the stream to a file if \code{outFile} is supplied.
}
\description{
High-level wrapper for stream editing functions (\code{\link{sed_insert}}, \code{\link{sed_replace}},
\code{\link{sed_substitute}}, and \code{\link{sed_comment}}) to execute any number of sequential
insertion, replacement, substitution, or commenting steps.
}
\details{
One (and only one) of \code{inFile} or \code{stream} must be specified.

If \code{inFile} and \code{outFile} are the same, a backup copy of \code{inFile} is made by appending
"~" to the end of the filename, e.g., if the original file were \file{aFile.txt}, the backup would be
\file{aFile.txt~}.

The value of \code{warn} in \code{streamEdit} is passed to the worker functions
(\code{\link{sed_insert}}, \code{\link{sed_replace}}, \code{\link{sed_substitute}}, and \code{\link{sed_comment}})
unless the \code{warn} argument is specified for a command in \code{commandList}, in which case, for that
particular command, the locally supplied value of \code{warn} takes precedence.
}
\examples{
################################################################################
# Let's create an example stream we can edit
################################################################################
cat("Here's a line\\n",
   "Line after which we'll insert a string\\n",
   "A line we'll delete\\n",
   "A line we'll replace with something else\\n",
   "A line where we'll make a substitution\\n",
   "A line we'll comment\\n",
   "The last line\\n",
   sep = "", file = "tmpTest_streamEdit.txt")

# Read the file into a 'stream'
s <- readLines("tmpTest_streamEdit.txt")

################################################################################
# Excecute a series of commands 'manually', using the individual worker functions
################################################################################

s <- sed_insert(s, after = 3, insertion = "Here's an insertion")
s <- sed_replace(s, at = "delete", replacement = NULL)
s <- sed_replace(s, at = "we'll replace", replacement = "The replacement", fixed = TRUE)
s <- sed_substitute(s, pattern = "make a substitution", replacement = "have a party")
s <- sed_comment(s, at = "comment", type = "html")
s

################################################################################
# Now execute these same commands using a single call to streamEdit(), along
# with reading the input file and writing the output file
################################################################################

# Build the list of commands
comList <- list(

 # i for 'insert', arguments for sed_insert()
 i = list(after = 3, insertion = "Here's an insertion"),

 # r for 'replace', arguments for sed_replace()
 r = list(at = "delete", replacement = NULL),

 # r for 'replace', arguments for sed_replace()
 r = list(at = "we'll replace", replacement = "The replacement", fixed = TRUE),

 # s for 'substitute', arguments for sed_substitute()
 s = list(pattern = "make a substitution", replacement = "have a party"),

 # c for 'comment', arguments for sed_comment()
 c = list(at = "comment", type = "html")

)

s1 <- streamEdit(comList, inFile = "tmpTest_streamEdit.txt",
                outFile = "tmpTest_streamEdit1.txt")
s1

# Compare the results
identical(s, s1)

# Remove the files
unlink(c("tmpTest_streamEdit.txt", "tmpTest_streamEdit1.txt"))
}
\seealso{
\code{\link{sed_insert}}, \code{\link{sed_replace}}, \code{\link{sed_substitute}}, \code{\link{sed_comment}}
}
\author{
Landon Sego
}
\keyword{misc}
