% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/slide.R
\name{slide-resampling}
\alias{slide-resampling}
\alias{sliding_window}
\alias{sliding_index}
\alias{sliding_period}
\title{Time-based Resampling}
\usage{
sliding_window(
  data,
  ...,
  lookback = 0L,
  assess_start = 1L,
  assess_stop = 1L,
  complete = TRUE,
  step = 1L,
  skip = 0L
)

sliding_index(
  data,
  index,
  ...,
  lookback = 0L,
  assess_start = 1L,
  assess_stop = 1L,
  complete = TRUE,
  step = 1L,
  skip = 0L
)

sliding_period(
  data,
  index,
  period,
  ...,
  lookback = 0L,
  assess_start = 1L,
  assess_stop = 1L,
  complete = TRUE,
  step = 1L,
  skip = 0L,
  every = 1L,
  origin = NULL
)
}
\arguments{
\item{data}{A data frame.}

\item{...}{These dots are for future extensions and must be empty.}

\item{lookback}{The number of elements to look back from the current element
when computing the resampling indices of the analysis set. The current
row is always included in the analysis set.
\itemize{
\item For \code{sliding_window()}, a single integer defining the number of rows to
look back from the current row.
\item For \code{sliding_index()}, a single object that will be subtracted from the
\code{index} as \code{index - lookback} to define the boundary of where to start
searching for rows to include in the current resample. This is often
an integer value corresponding to the number of days to look back,
or a lubridate Period object.
\item For \code{sliding_period()}, a single integer defining the number of groups
to look back from the current group, where the groups were defined from
breaking up the \code{index} according to the \code{period}.
}

In all cases, \code{Inf} is also allowed to force an expanding window.}

\item{assess_start, assess_stop}{This combination of arguments determines
how far into the future to look when constructing the assessment set.
Together they construct a range of
\verb{[index + assess_start, index + assess_stop]} to search for rows to include
in the assessment set.

Generally, \code{assess_start} will always be \code{1} to indicate that the first
value to potentially include in the assessment set should start one element
after the current row, but it can be increased to a larger value to
create "gaps" between the analysis and assessment set if you are worried
about high levels of correlation in short term forecasting.
\itemize{
\item For \code{sliding_window()}, these are both single integers defining the
number of rows to look forward from the current row.
\item For \code{sliding_index()}, these are single objects that will be added
to the \code{index} to compute the range to search for rows to include
in the assessment set. This is often an integer value corresponding to
the number of days to look forward, or a lubridate Period object.
\item For \code{sliding_period()}, these are both single integers defining the
number of groups to look forward from the current group, where the groups
were defined from breaking up the \code{index} according to the \code{period}.
}}

\item{complete}{A single logical. When using \code{lookback} to compute the
analysis sets, should only complete windows be considered? If set to
\code{FALSE}, partial windows will be used until it is possible to create
a complete window (based on \code{lookback}). This is a way to use an
expanding window up to a certain point, and then switch to a sliding
window.}

\item{step}{A single positive integer. After computing the resampling
indices, \code{step} is used to thin out the results by selecting every
\code{step}-th result by subsetting the indices with
\code{seq(1L, n_indices, by = step)}. \code{step} is applied after \code{skip}.
Note that \code{step} is independent of any time \code{index} used.}

\item{skip}{A single positive integer, or zero. After computing the
resampling indices, the first \code{skip} results will be dropped by subsetting
the indices with \code{seq(skip + 1L, n_indices)}. This can be especially
useful when combined with \code{lookback = Inf}, which creates an expanding
window starting from the first row. By skipping forward, you can drop
the first few windows that have very few data points. \code{skip} is
applied before \code{step}. Note that \code{skip} is independent of any time
\code{index} used.}

\item{index}{The index to compute resampling indices relative to, specified
as a bare column name. This must be an existing column in \code{data}.
\itemize{
\item For \code{sliding_index()}, this is commonly a date vector, but is not
required.
\item For \code{sliding_period()}, it is required that this is a Date or POSIXct
vector.
}

The \code{index} must be an \emph{increasing} vector, but duplicate values are
allowed. Additionally, the index cannot contain any missing values.}

\item{period}{The period to group the \code{index} by. This is specified as a
single string, such as \code{"year"} or \code{"month"}. See the \code{.period} argument
of \code{\link[slider:slide_index]{slider::slide_index()}} for the full list of options and further
explanation.}

\item{every}{A single positive integer. The number of periods to group
together.

For example, if the \code{period} was set to \code{"year"} with an \code{every}
value of 2, then the years 1970 and 1971 would be placed in the same
group.}

\item{origin}{The reference date time value. The default when left
as \code{NULL} is the epoch time of \verb{1970-01-01 00:00:00},
\emph{in the time zone of the index}.

This is generally used to define the anchor time to count from,
which is relevant when the \code{every} value is \verb{> 1}.}
}
\description{
These resampling functions are focused on various forms of \emph{time series}
resampling.
\itemize{
\item \code{sliding_window()} uses the row number when computing the resampling
indices. It is independent of any time index, but is useful with
completely regular series.
\item \code{sliding_index()} computes resampling indices relative to the \code{index}
column. This is often a Date or POSIXct column, but doesn't have to be.
This is useful when resampling irregular series, or for using irregular
lookback periods such as \code{lookback = lubridate::years(1)} with daily
data (where the number of days in a year may vary).
\item \code{sliding_period()} first breaks up the \code{index} into less granular groups
based on \code{period}, and then uses that to construct the resampling indices.
This is extremely useful for constructing rolling monthly or yearly
windows from daily data.
}
}
\examples{
library(vctrs)
library(tibble)
library(modeldata)
data("Chicago")

index <- new_date(c(1, 3, 4, 7, 8, 9, 13, 15, 16, 17))
df <- tibble(x = 1:10, index = index)
df

# Look back two rows beyond the current row, for a total of three rows
# in each analysis set. Each assessment set is composed of the two rows after
# the current row.
sliding_window(df, lookback = 2, assess_stop = 2)

# Same as before, but step forward by 3 rows between each resampling slice,
# rather than just by 1.
rset <- sliding_window(df, lookback = 2, assess_stop = 2, step = 3)
rset

analysis(rset$splits[[1]])
analysis(rset$splits[[2]])

# Now slide relative to the `index` column in `df`. This time we look back
# 2 days from the current row's `index` value, and 2 days forward from
# it to construct the assessment set. Note that this series is irregular,
# so it produces different results than `sliding_window()`. Additionally,
# note that it is entirely possible for the assessment set to contain no
# data if you have a highly irregular series and "look forward" into a
# date range where no data points actually exist!
sliding_index(df, index, lookback = 2, assess_stop = 2)

# With `sliding_period()`, we can break up our date index into less granular
# chunks, and slide over them instead of the index directly. Here we'll use
# the Chicago data, which contains daily data spanning 16 years, and we'll
# break it up into rolling yearly chunks. Three years worth of data will
# be used for the analysis set, and one years worth of data will be held out
# for performance assessment.
sliding_period(
  Chicago,
  date,
  "year",
  lookback = 2,
  assess_stop = 1
)

# Because `lookback = 2`, three years are required to form a "complete"
# window of data. To allow partial windows, set `complete = FALSE`.
# Here that first constructs two expanding windows until a complete three
# year window can be formed, at which point we switch to a sliding window.
sliding_period(
  Chicago,
  date,
  "year",
  lookback = 2,
  assess_stop = 1,
  complete = FALSE
)

# Alternatively, you could break the resamples up by month. Here we'll
# use an expanding monthly window by setting `lookback = Inf`, and each
# assessment set will contain two months of data. To ensure that we have
# enough data to fit our models, we'll `skip` the first 4 expanding windows.
# Finally, to thin out the results, we'll `step` forward by 2 between
# each resample.
sliding_period(
  Chicago,
  date,
  "month",
  lookback = Inf,
  assess_stop = 2,
  skip = 4,
  step = 2
)
}
\seealso{
\code{\link[=rolling_origin]{rolling_origin()}}

\code{\link[slider:slide]{slider::slide()}}, \code{\link[slider:slide_index]{slider::slide_index()}}, and \code{\link[slider:slide_period]{slider::slide_period()}},
which power these resamplers.
}
