## ---- include = FALSE, eval = FALSE-------------------------------------------
#  knitr::opts_chunk$set(
#    collapse = TRUE,
#    comment = "#>"
#  )

## ---- message = FALSE, include = FALSE----------------------------------------------------------------------------------------------------------------------------------------------------------------
options(width = 200)

## ----setup, eval = FALSE, message = FALSE-------------------------------------------------------------------------------------------------------------------------------------------------------------
#  install.packages("rrMixture")
#  library(rrMixture)

## ---- message = FALSE, echo = FALSE-------------------------------------------------------------------------------------------------------------------------------------------------------------------
library(rrMixture)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Load and pre-process a data set
library(bayesm)
data(tuna)
tunaY <- log(tuna[, c("MOVE1", "MOVE2", "MOVE3", "MOVE4", "MOVE5", "MOVE6", "MOVE7")])
tunaX <- tuna[, c("NSALE1", "NSALE2", "NSALE3", "NSALE4", "NSALE5", "NSALE6", "NSALE7",
              "LPRICE1", "LPRICE2", "LPRICE3", "LPRICE4", "LPRICE5", "LPRICE6", "LPRICE7")]
tunaX <- cbind(intercept = 1, tunaX)

## ---- cache = TRUE------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Parameter estimation with `rrmix()` using the rank penalized method
tuna.mod <- rrmix(K = 2, X = tunaX, Y = tunaY, est = "RP", lambda = 3,
                  seed = 100, n.init = 100)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Estimated parameters
tuna.mod$para

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Estimated ranks of coefficient matrices
tuna.mod$est.rank

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Summarize the estimation results
summary(tuna.mod)

## ---- fig.width = 7, fig.height = 5-------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Visualize the estimation results
plot(tuna.mod)

## ---- cache = TRUE, results = 'hide'------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Parameter tuning with `tune.rrmix()` using the rank penalized method
tuna.tune1 <- tune.rrmix(K = 2, X = tunaX, Y = tunaY, est = "RP",
                         lambda = exp(seq(0, log(20), length = 20)),
                         seed = 100, n.init = 100)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Summarize the results
summary(tuna.tune1)

## ---- fig.width = 7, fig.height = 5-------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Visualize the results
plot(tuna.tune1, transform.x = log, xlab = "log(lambda)")

## ---- cache = TRUE, results = 'hide'------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Parameter tuning with `tune.rrmix()` using the rank penalized method
tuna.tune2 <- tune.rrmix(K.max = 3, X = tunaX, Y = tunaY, est = "RP",
                         lambda = exp(seq(0, log(20), length = 20)),
                         seed = 100, n.init = 100)

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Summarize the results
summary(tuna.tune2)

## ---- fig.width = 7, fig.height = 5-------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Visualize the results
plot(tuna.tune2, transform.y = log, ylab = "log(lambda)")

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# The final model
best.K <- summary(tuna.tune2)$best.K
best.lambda <- summary(tuna.tune2)$best.lambda
best.mod <- rrmix(K = best.K, X = tunaX, Y = tunaY, est = "RP", lambda = best.lambda,
                  seed = 100, n.init = 100)
summary(best.mod)
best.mod$para

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Estimated ranks of coefficient matrices
best.mod$est.rank

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# Membership information of mixture components
best.mod$ind
best.mod$n.est

## ---- message = FALSE---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# rrpack package
require(rrpack)
rfit <- rrr(Y = as.matrix(tunaY), X = as.matrix(tunaX),
            modstr = list(lambda = 3, gamma = 2), penaltySVD = "ann")
# estimated coefficient matrix
coef(rfit)
# estimated rank of the coefficient matrix
rfit$rank

## -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
# rrMixture package
rfit2 <- rrmix(K = 1, Y = tunaY, X = tunaX, lambda = 3, gamma = 2, est = "ANNP")
# estimated coefficient matrix
rfit2$para[[1]]$B
# estimated rank of the coefficient matrix
rfit2$est.rank

