internal_f <- function(p, f) {
  stopifnot(is.character(p), length(p) == 1)
  stopifnot(is.character(f), length(f) == 1)

  get(f, envir = asNamespace(p))
}

"%||%" <- function(a, b) {
  if (length(a) > 0) a else b
}

subs <- matrix(ncol = 2, byrow = T, c(
  # Common special function names
  '[<-', 'subset',
  '[', 'sub',
  '<-', 'set',

  # Infix verbs
  '!', 'not',
  '&', 'and',
  '|', 'or',
  '*', 'times',
  '+', 'plus',
  '^', 'pow',

  # Others
  '"', 'quote',
  '#', 'hash',
  '$', 'cash',
  '%', 'grapes',
  "'", 'single-quote',
  '(', 'open-paren',
  ')', 'close-paren',
  ':', 'colon',
  ';', 'semi-colon',
  '<', 'less-than',
  '==', 'equals',
  '=', 'equals',
  '>', 'greater-than',
  '?', 'help',
  '@', 'at',
  ']', 'close-brace',
  '\\', 'backslash',
  '/', 'slash',
  '`', 'tick',
  '{', 'open-curly',
  '}', 'close',
  '~', 'twiddle'
))
subs[, 2] <- paste0("-", subs[, 2], "-")

nice_name <- function(x) {
  x <- stringi::stri_replace_all_fixed(x, subs[, 1], subs[, 2],
    vectorize_all = FALSE)

  # Clean up any remaining
  x <- str_replace_all(x, "[^A-Za-z0-9_.-]+", "-")
  x <- str_replace_all(x, "-+", "-")
  x <- str_replace_all(x, "^-|-$", "")
  x <- str_replace_all(x, "^\\.", "dot-")
  x
}

write_if_different <- function(path, contents, check = TRUE) {
  if (!file.exists(dirname(path))) {
    dir.create(dirname(path), showWarnings = FALSE)
  }

  if (check && !made_by_roxygen(path)) {
    warning("The existing '", basename(path),
      "' file was not generated by roxygen2, and will not be overwritten.",
      call. = FALSE, immediate. = TRUE)
    return(FALSE)
  }

  if (same_contents(path, contents)) return(FALSE)

  name <- basename(path)
  if (!str_detect(name, "^[a-zA-Z][a-zA-Z0-9_.-]*$")) {
    cat("Skipping invalid path: ", name, "\n")
    FALSE
  } else {
    cat(sprintf('Writing %s\n', name))
    write_lines(contents, path)
    TRUE
  }
}

same_contents <- function(path, contents) {
  if (!file.exists(path)) return(FALSE)

  contents <- paste0(paste0(contents, collapse = "\n"), "\n")
  if (.Platform$OS.type == "windows") {
    contents <- gsub("\n", "\r\n", contents, fixed = TRUE)
  }

  text_hash <- digest::digest(contents, serialize = FALSE)
  file_hash <- digest::digest(file = path)

  identical(text_hash, file_hash)
}

r_files <- function(path) {
  sort_c(dir(file.path(path, "R"), "\\.[Rr]$", full.names = TRUE))
}

ignore_files <- function(rfiles, path) {
  rbuildignore <- file.path(path, ".Rbuildignore")
  if (!file.exists(rbuildignore))
    return(rfiles)

  # Strip leading directory and slashes
  rfiles_relative <- sub(normalizePath(path, winslash = "/"), "", normalizePath(rfiles, winslash = "/"), fixed = TRUE)
  rfiles_relative <- sub("^[/]*", "", rfiles_relative)

  # Remove any files that match any perl-compatible regexp
  patterns <- read_lines(rbuildignore)
  patterns <- patterns[patterns != ""]
  if (length(patterns) == 0L) {
    return(rfiles)
  }
  matches <- lapply(patterns, grepl, rfiles_relative, perl = TRUE)
  matches <- Reduce("|", matches)
  rfiles[!matches]
}

compact <- function(x) {
  null <- vapply(x, is.null, logical(1))
  x[!null]
}

block_eval <- function(tag, block, env, tag_name) {
  tryCatch({
    expr <- parse(text = tag)
    out <- eval(expr, envir = env)

    if (!is.character(out)) {
      block_warning(block, tag_name, " did not evaluate to a string")
    } else if (anyNA(out)) {
      block_warning(block, tag_name, " result contained NA")
    } else {
      out
    }
  }, error = function(e) {
    block_warning(block, tag_name, " failed with error:\n", e$message)
  })
}


# Parse DESCRIPTION into convenient format
read.description <- function(file) {
  dcf <- desc::desc(file = file)

  fields <- dcf$fields()
  purrr::map(purrr::set_names(fields), ~ dcf$get_field(.x))
}


wrap_string <- function(x, width = 80L) UseMethod("wrap_string")
wrap_string.NULL <- function(x, width = 80L) return(x)
wrap_string.default <- function(x, width = 80L) {
  y <- wrapString(x, width = as.integer(width))
  y <- gsub("\u{A0}", " ", y, useBytes = TRUE)
  Encoding(y) <- "UTF-8"
  class(y) <- class(x)
  y
}

invert <- function(x) {
  if (length(x) == 0) return()
  stacked <- utils::stack(x)
  tapply(as.character(stacked$ind), stacked$values, list)
}

has_colons <- function(x) {
   grepl("::", x, fixed = TRUE)
}

# Collapse the values associated with duplicated keys
collapse <- function(key, value, fun, ...) {
  stopifnot(is.character(key))
  stopifnot(length(key) == length(value))

  dedup <- tapply(value, key, fun, ..., simplify = FALSE)
  # tapply orders alphabetically, so reorder to match original order
  dedup <- dedup[unique(key)]

  list(
    key = names(dedup),
    value = unname(dedup)
  )
}

cat_line <- function(...) {
  cat(..., "\n", sep = "")
}
