% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/correlation.R
\name{cor_spearman}
\alias{cor_spearman}
\title{Spearman's correlation coefficient for bivariate functional data}
\usage{
cor_spearman(mfD, ordering = "MEI", ...)
}
\arguments{
\item{mfD}{a bivariate functional dataset whose Spearman's correlation
coefficient must be computed, in form of bivariate \code{mfData} object
(\code{mfD$L=2}).}

\item{ordering}{the ordering relation to use on functional observations,
either \code{"MEI"} for MEI or \code{"MHI"} for MHI (default is \code{"MEI"}).}

\item{...}{additional parameters to be passed to rank function}
}
\value{
The function returns the Spearman's correlation coefficient for
the bivariate dataset provided with \code{mfData}.
}
\description{
This function computes the Spearman's tau correlation coefficient for a
bivariate functional dataset, with either a Modified Epigraph Index (MEI) or
Modified Hypograph Index (MHI) based ranking of univariate elments of data
components.
}
\details{
Given a bivariate functional dataset, with first components \eqn{X_1(t),
X_2(t), \ldots, X_N(t)} and second components \eqn{Y_1(t), Y_2(t), \ldots,
Y_N(t)}, the function exploits either the MEI or MHI to rank observations and
hence compute the value of the correlation coefficient.

See the references for more details.
}
\examples{

#### TOTALLY INDEPENDENT COMPONENTS

N = 2e2
P = 1e3

grid = seq( 0, 1, length.out = P )

# Creating an exponential covariance function to simulate guassian data
Cov = exp_cov_function( grid, alpha = 0.3, beta = 0.4 )

# Simulating (independent) gaussian functional data with given center and
# covariance function
Data_1 = generate_gauss_fdata( N, centerline = sin( 2 * pi * grid ), Cov = Cov )
Data_2 = generate_gauss_fdata( N, centerline = sin( 2 * pi * grid ), Cov = Cov )

# Using the simulated data as (independent) components of a bivariate functional
# dataset
mfD = mfData( grid, list( Data_1, Data_2 ) )

# Correlation approx. zero (components were created independently)
cor_spearman( mfD, ordering = 'MEI' )

# Correlation approx. zero (components were created independently)
cor_spearman( mfD, ordering = 'MHI' )

#### TOTALLY DEPENDENT COMPONENTS

# Nonlinear transform of first component
Data_3 = t( apply( Data_1, 1, exp ) )

# Creating bivariate dataset starting from nonlinearly-dependent components
mfD = mfData( grid, list( Data_1, Data_3 ) )

# Correlation very high (components are nonlinearly dependent)
cor_spearman( mfD, ordering = 'MEI' )

# Correlation very high (components are nonlinearly dependent)
cor_spearman( mfD, ordering = 'MHI' )

}
\references{
Valencia, D., Romo, J. and Lillo, R. (2015). Spearman coefficient for
functions, \emph{Universidad Carlos III de Madrid technical report},
\code{http://EconPapers.repec.org/RePEc:cte:wsrepe:ws133329}.
}
\seealso{
\code{\link{mfData}}, \code{\link{MEI}}, \code{\link{MHI}}
}

