#' Convert to an HTML document
#'
#' Format for converting from R Markdown to an HTML document.
#'
#' @param toc \code{TRUE} to include a table of contents in the output
#' @param toc_depth Depth of headers to include in table of contents
#' @param number_sections \code{TRUE} to number section headings
#' @param fig_width Default width (in inches) for figures
#' @param fig_height Default width (in inches) for figures
#' @param fig_retina Scaling to perform for retina displays (defaults to 2 when
#'   \code{fig_caption} is \code{FALSE}, which currently works for all widely
#'   used retina displays). Set to \code{NULL} to prevent retina scaling. Note
#'   that this will always be \code{NULL} when \code{keep_md} is specified (this
#'   is because \code{fig_retina} relies on outputting HTML directly into the
#'   markdown document).
#' @param fig_caption \code{TRUE} to render figures with captions
#' @param dev Graphics device to use for figure output (defaults to png)
#' @param smart Produce typographically correct output, converting straight
#'   quotes to curly quotes, --- to em-dashes, -- to en-dashes, and ... to
#'   ellipses.
#' @param self_contained Produce a standalone HTML file with no external
#'   dependencies, using data: URIs to incorporate the contents of linked
#'   scripts, stylesheets, images, and videos. Note that even for self
#'   contained documents MathJax is still loaded externally (this is
#'   necessary because of it's size).
#' @param theme Visual theme ("default", "cerulean", "journal", "flatly",
#'   "readable", "spacelab", "united", or "cosmo"). Pass \code{NULL} for
#'   no theme (in this case you can use the \code{css} parameter to add your own
#'   styles).
#' @param highlight Syntax highlighting style. Supported styles include
#'   "default", "tango", "pygments", "kate", "monochrome", "espresso",
#'   "zenburn", "haddock", and "textmate". Pass \code{NULL} to prevent syntax
#'   highlighting.
#' @param mathjax Include mathjax. The "default" option uses an https URL from
#'   the official MathJax CDN. The "local" option uses a local version of
#'   MathJax (which is copied into the output directory). You can pass an
#'   alternate URL or pass \code{NULL} to exclude MathJax entirely.
#' @param template Pandoc template to use for rendering. Pass "default"
#'   to use the rmarkdown package default template; pass \code{NULL}
#'   to use pandoc's built-in template; pass a path to use a custom template
#'   that you've created. Note that if you don't use the "default" template
#'   then some features of \code{html_document} won't be available (see the
#'   Templates section below for more details).
#' @param css One or more css files to include
#' @param includes Named list of additional content to include within the
#'   document (typically created using the \code{\link{includes}} function).
#' @param keep_md Keep the markdown file generated by knitting.
#' @param lib_dir Directory to copy dependent HTML libraries (e.g. jquery,
#'   bootstrap, etc.) into. By default this will be the name of the document
#'   with \code{_files} appended to it.
#' @param md_extensions Markdown extensions to be added or removed from the
#'   default definition or R Markdown. See the \code{\link{rmarkdown_format}}
#'   for additional details.
#' @param pandoc_args Additional command line options to pass to pandoc
#' @param ... Additional function arguments to pass to the base R Markdown HTML
#'   output formatter
#'
#' @return R Markdown output format to pass to \code{\link{render}}
#'
#' @details
#'
#' See the \href{http://rmarkdown.rstudio.com/html_document_format.html}{online
#' documentation} for additional details on using the \code{html_document} format.
#'
#' R Markdown documents can have optional metadata that is used to generate a
#' document header that includes the title, author, and date. For more details
#' see the documentation on R Markdown \link[=rmd_metadata]{metadata}.
#'
#' R Markdown documents also support citations. You can find more information on
#' the markdown syntax for citations in the
#' \href{http://rmarkdown.rstudio.com/authoring_bibliographies_and_citations.html}{Bibliographies
#' and Citations} article in the online documentation.
#'
#'
#' @section Templates:
#'
#' You can provide a custom HTML template to be used for rendering. The syntax
#' for templates is described in the 
#' \href{http://pandoc.org/README.html}{pandoc
#' documentation}. You can also use the basic pandoc template by passing
#' \code{template = NULL}.
#'
#' Note however that if you choose not to use the "default" HTML template
#' then several aspects of HTML document rendering will behave differently:
#'
#' \itemize{
#'   \item{The \code{theme} parameter does not work (you can still
#'      provide styles using the \code{css} parameter).
#'   }
#'   \item{For the \code{highlight} parameter, the default highlighting
#'      style will resolve to "pygments" and the "textmate" highlighting
#'      style is not available
#'   }
#'   \item{MathJax will not work if \code{self_contained} is \code{TRUE}
#'      (these two options can't be used together in normal pandoc templates).
#'   }
#' }
#'
#' Due to the above restrictions, you might consider using the \code{includes}
#' parameter as an alternative to providing a fully custom template.
#'
#' @examples
#' \dontrun{
#'
#' library(rmarkdown)
#'
#' render("input.Rmd", html_document())
#'
#' render("input.Rmd", html_document(toc = TRUE))
#' }
#'
#' @export
html_document <- function(toc = FALSE,
                          toc_depth = 3,
                          number_sections = FALSE,
                          fig_width = 7,
                          fig_height = 5,
                          fig_retina = if (!fig_caption) 2,
                          fig_caption = FALSE,
                          dev = 'png',
                          smart = TRUE,
                          self_contained = TRUE,
                          theme = "default",
                          highlight = "default",
                          mathjax = "default",
                          template = "default",
                          css = NULL,
                          includes = NULL,
                          keep_md = FALSE,
                          lib_dir = NULL,
                          md_extensions = NULL,
                          pandoc_args = NULL,
                          ...) {

  # build pandoc args
  args <- c("--standalone")

  # use section divs
  args <- c(args, "--section-divs")

  # table of contents
  args <- c(args, pandoc_toc_args(toc, toc_depth))

  # template path and assets
  if (identical(template, "default"))
    args <- c(args, "--template",
              pandoc_path_arg(rmarkdown_system_file("rmd/h/default.html")))
  else if (!is.null(template))
    args <- c(args, "--template", pandoc_path_arg(template))

  # numbered sections
  if (number_sections)
    args <- c(args, "--number-sections")

  # additional css
  for (css_file in css)
    args <- c(args, "--css", pandoc_path_arg(css_file))

  # pre-processor for arguments that may depend on the name of the
  # the input file (e.g. ones that need to copy supporting files)
  pre_processor <- function(metadata, input_file, runtime, knit_meta, files_dir,
                            output_dir) {

    # use files_dir as lib_dir if not explicitly specified
    if (is.null(lib_dir))
      lib_dir <- files_dir

    # extra args
    args <- c()

    # highlight
    args <- c(args, pandoc_html_highlight_args(highlight,
                                               template,
                                               self_contained,
                                               lib_dir,
                                               output_dir))

    # content includes (we do this here so that user include-in-header content
    # goes after dependency generated content). make the paths absolute if
    # making a Shiny document so we can resolve them even if rendering
    # elsewhere.
    args <- c(args, includes_to_pandoc_args(includes,
                      filter = if (identical(runtime, "shiny"))
                        normalize_path
                      else
                        identity))

    # return additional args
    args
  }

  # return format
  output_format(
    knitr = knitr_options_html(fig_width, fig_height, fig_retina, keep_md, dev),
    pandoc = pandoc_options(to = "html",
                            from = from_rmarkdown(fig_caption, md_extensions),
                            args = args),
    keep_md = keep_md,
    clean_supporting = self_contained,
    pre_processor = pre_processor,
    base_format = html_document_base(smart = smart, theme = theme,
                                     self_contained = self_contained,
                                     lib_dir = lib_dir, mathjax = mathjax,
                                     template = template,
                                     pandoc_args = pandoc_args, ...)
  )
}


#' Knitr options for an HTML output format
#'
#' Define knitr options for an R Markdown output format that creates
#' HTML output.
#'
#' @inheritParams html_document
#'
#' @return An list that can be passed as the \code{knitr} argument of the
#'   \code{\link{output_format}} function.
#'
#' @seealso \link{knitr_options}, \link{output_format}
#'
#' @export
knitr_options_html <- function(fig_width, fig_height, fig_retina, keep_md, dev = 'png') {

  opts_chunk <- list(dev = dev,
                     dpi = 96,
                     fig.width = fig_width,
                     fig.height = fig_height,
                     fig.retina = fig_retina)

  if (keep_md)
    opts_chunk$fig.retina <- NULL

  knitr_options(opts_chunk = opts_chunk)
}

themes <- function() {
  c("default",
    "cerulean",
    "journal",
    "flatly",
    "readable",
    "spacelab",
    "united",
    "cosmo")
}

html_highlighters <- function() {
  c(highlighters(), "textmate")
}

default_mathjax <- function() {
  paste("https://cdn.mathjax.org/mathjax/latest/",
        mathjax_config(), sep="")
}

mathjax_config <- function() {
  "MathJax.js?config=TeX-AMS-MML_HTMLorMML"
}
