% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cnd-restarts.R
\name{rst_muffle}
\alias{rst_muffle}
\title{Jump to a muffling restart}
\usage{
rst_muffle(c)
}
\arguments{
\item{c}{A condition to muffle.}
}
\description{
Muffle restarts are established at the same location as where a
condition is signalled. They are useful for two non-exclusive
purposes: muffling signalling functions and muffling conditions. In
the first case, \code{rst_muffle()} prevents any further side effects of
a signalling function (a warning or message from being displayed,
an aborting jump to top level, etc). In the second case, the
muffling jump prevents a condition from being passed on to other
handlers. In both cases, execution resumes normally from the point
where the condition was signalled.
}
\examples{
side_effect <- function() cat("side effect!\\n")
handler <- inplace(function(c) side_effect())

# A muffling handler is an inplace handler that jumps to a muffle
# restart:
muffling_handler <- inplace(function(c) {
  side_effect()
  rst_muffle(c)
})

# You can also create a muffling handler simply by setting
# muffle = TRUE:
muffling_handler <- inplace(function(c) side_effect(), muffle = TRUE)

# You can then muffle the signalling function:
fn <- function(signal, msg) {
  signal(msg)
  "normal return value"
}
with_handlers(fn(message, "some message"), message = handler)
with_handlers(fn(message, "some message"), message = muffling_handler)
with_handlers(fn(warning, "some warning"), warning = muffling_handler)

# Note that exiting handlers are thrown to the establishing point
# before being executed. At that point, the restart (established
# within the signalling function) does not exist anymore:
\dontrun{
with_handlers(fn(warning, "some warning"),
  warning = exiting(function(c) rst_muffle(c)))
}


# Another use case for muffle restarts is to muffle conditions
# themselves. That is, to prevent other condition handlers from
# being called:
undesirable_handler <- inplace(function(c) cat("please don't call me\\n"))

with_handlers(foo = undesirable_handler,
  with_handlers(foo = muffling_handler, {
    cnd_signal("foo", mufflable = TRUE)
    "return value"
  }))

# See the `mufflable` argument of cnd_signal() for more on this point
}
\seealso{
The \code{muffle} argument of \code{\link[=inplace]{inplace()}}, and the \code{mufflable}
argument of \code{\link[=cnd_signal]{cnd_signal()}}.
}
