% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_risk_diff.R
\name{calc_risk_diff}
\alias{calc_risk_diff}
\title{Calculate Risk Differences with Robust Model Fitting}
\usage{
calc_risk_diff(
  data,
  outcome,
  exposure,
  adjust_vars = NULL,
  strata = NULL,
  link = "auto",
  alpha = 0.05,
  verbose = FALSE
)
}
\arguments{
\item{data}{A data frame containing all necessary variables}

\item{outcome}{Character string naming the binary outcome variable (must be 0/1 or logical)}

\item{exposure}{Character string naming the exposure variable of interest}

\item{adjust_vars}{Character vector of variables to adjust for (default: NULL)}

\item{strata}{Character vector of stratification variables (default: NULL)}

\item{link}{Character string specifying link function: "auto", "identity", "log", or "logit" (default: "auto")}

\item{alpha}{Significance level for confidence intervals (default: 0.05)}

\item{verbose}{Logical indicating whether to print diagnostic messages (default: FALSE)}
}
\value{
A tibble of class "riskdiff_result" containing the following columns:
\describe{
\item{exposure_var}{Character. Name of exposure variable analyzed}
\item{rd}{Numeric. Risk difference estimate (proportion scale, e.g. 0.05 = 5 percentage points)}
\item{ci_lower}{Numeric. Lower bound of confidence interval}
\item{ci_upper}{Numeric. Upper bound of confidence interval}
\item{p_value}{Numeric. P-value for test of null hypothesis (risk difference = 0)}
\item{model_type}{Character. Link function successfully used ("identity", "log", "logit", or error type)}
\item{n_obs}{Integer. Number of observations used in analysis}
\item{...}{Additional columns for stratification variables if specified}
}

The returned object has attributes including the original function call and
alpha level used. Risk differences are on the probability scale where 0.05
represents a 5 percentage point difference.
}
\description{
Calculates risk differences (or prevalence differences for cross-sectional data)
using generalized linear models with identity, log, or logit links. Includes
support for stratification, adjustment variables, and robust model fitting.

The function addresses common convergence issues with identity link binomial
GLMs by implementing a fallback strategy across multiple link functions,
similar to approaches described in Donoghoe & Marschner (2018) for relative
risk regression.
}
\details{
\subsection{Statistical Approach}{

The function attempts to fit models in the following order:
\enumerate{
\item \strong{Identity link}: Directly estimates risk differences using a binomial GLM with identity link
\item \strong{Log link}: Estimates relative risks, then transforms to risk differences via prediction
\item \strong{Logit link}: Estimates odds ratios, then transforms to risk differences via prediction
}
}

\subsection{Convergence Strategy}{

Identity link models for binary outcomes often fail to converge due to
parameter constraints (fitted probabilities must be \\le{} 1). This package
implements robust starting values and automatic fallback to alternative
link functions when convergence fails, ensuring reliable estimation.
}

\subsection{Risk Difference Interpretation}{

Risk differences represent absolute changes in probability. A risk difference
of 0.05 means the exposed group has a 5 percentage point higher risk than
the unexposed group. This is often more interpretable than relative measures
(risk ratios, odds ratios) for public health decision-making.
}
}
\examples{
# Simple risk difference
data(birthweight)
rd_simple <- calc_risk_diff(
  data = birthweight,
  outcome = "low_birthweight",
  exposure = "smoking"
)
print(rd_simple)

# Age-adjusted risk difference
rd_adjusted <- calc_risk_diff(
  data = birthweight,
  outcome = "low_birthweight",
  exposure = "smoking",
  adjust_vars = "maternal_age"
)
print(rd_adjusted)

# Stratified analysis
rd_stratified <- calc_risk_diff(
  data = birthweight,
  outcome = "low_birthweight",
  exposure = "smoking",
  strata = "race"
)
print(rd_stratified)

}
\references{
Donoghoe MW, Marschner IC (2018). "logbin: An R Package for Relative Risk
Regression Using the Log-Binomial Model." Journal of Statistical Software,
86(9), 1-22. doi:10.18637/jss.v086.i09

Rothman KJ, Greenland S, Lash TL (2008). Modern Epidemiology, 3rd edition.
Lippincott Williams & Wilkins.
}
