% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rif.R
\name{get_rif}
\alias{get_rif}
\title{Estimate Recentered Influence Functions}
\usage{
get_rif(
  dep_var,
  weights = NULL,
  statistic,
  probs = NULL,
  custom_rif_function = NULL,
  ...
)
}
\arguments{
\item{dep_var}{dependent variable of distributional function. Discrete or continuous numeric vector.}

\item{weights}{numeric vector of non-negative observation weights, hence of same length as \code{dep_var}.
The default (\code{NULL)} is equivalent to \code{weights = rep(1, length(dep_var))}.}

\item{statistic}{string containing the distributional statistic for which to compute the RIF. Can be one of
"mean", "variance", "quantiles", "gini", "interquantile_range", "interquantile_ratio", or "custom". If "custom"
is selected a \code{custom_rif_function} needs to be provided.}

\item{probs}{a vector of length 1 or more with quantile positions to calculate the RIF.
Each quantile is indicated with value between 0 and 1. Only required if \code{statistic = "quantiles"}.}

\item{custom_rif_function}{the RIF function to compute the RIF of the custom distributional statistic.
Default is NULL. Only needs to be provided if \code{statistic = "custom"}.
Every custom_rif_function needs the parameters \code{dep_var}, \code{weights} and \code{probs}.
If they are not needed they must be set to NULL in the function definition (e.g. \code{probs = NULL}).
A custom function must return a data frame containing at least a "rif" and "weights" column.
See \code{examples} for further details.}

\item{...}{additional parameters passed to the \code{custom_rif_function}.
Apart from \code{dep_var}, \code{weights} and \code{probs} they must have a different name than the the ones in
\code{rif}. For instance, if you want to pass a parameter \code{statistic} to the
\code{custom_rif_function}, name it \code{custom_statistic}.}
}
\value{
a data frame with the RIF value for each observation and in the case of several quantiles
        a column for each quantile.
}
\description{
This function estimates the recentered influence function (RIF) of a chosen distributional statistic
(e.g. quantiles, variance or gini).
}
\examples{

dep_var <- c(1, 3, 9, 16, 3, 7, 4, 9)
probs <- seq(1:9) / 10
weights <- c(2, 1, 3, 4, 4, 1, 6, 3)
rif <- get_rif(
  dep_var = dep_var,
  weights = weights,
  statistic = "quantiles",
  probs = probs
)

# custom function
custom_variance_function <- function(dep_var, weights, probs = NULL) {
  weighted_mean <- weighted.mean(x = dep_var, w = weights)
  rif <- (dep_var - weighted_mean)^2
  rif <- data.frame(rif, weights)
  names(rif) <- c("rif_variance", "weights")
  return(rif)
}

set.seed(123)
dep_var <- rlnorm(100)
weights <- rep(1, 100)

# custom function top 10\% percent income share
# (see Essam-Nassah & Lambert, 2012, and Rios-Avila, 2020)
custom_top_income_share_function <- function(dep_var, weights, probs = 0.1) {
  probs <- 1 - probs
  weighted_mean <- weighted.mean(x = dep_var, w = weights)
  weighted_quantile <- Hmisc::wtd.quantile(x = dep_var, weights = weights, probs = probs)
  lorenz_ordinate <-
    sum(dep_var[which(dep_var <= weighted_quantile)] *
      weights[which(dep_var <= weighted_quantile)]) / sum(dep_var * weights)
  if_lorenz_ordinate <- -(dep_var / weighted_mean) * lorenz_ordinate +
    ifelse(dep_var < weighted_quantile,
      dep_var - (1 - probs) * weighted_quantile,
      probs * weighted_quantile
    ) / weighted_mean
  rif_top_income_share <- (1 - lorenz_ordinate) - if_lorenz_ordinate
  rif <- data.frame(rif_top_income_share, weights)
  names(rif) <- c("rif_top_income_share", "weights")
  return(rif_top_income_share)
}

rif_custom <- get_rif(
  dep_var = dep_var,
  weights = weights,
  statistic = "custom",
  custom_rif_function = custom_variance_function
)

}
\references{
Firpo, Sergio P., Nicole M. Fortin, and Thomas Lemieux. 2009. “Unconditional Quantile
Regressions.” \emph{Econometrica} 77(3): 953–73.

Cowell, Frank A., and Emmanuel Flachaire. 2015. “Statistical Methods for Distributional Analysis.”
In Anthony B. Atkinson and François Bourguignon (eds.), \emph{Handbook of Income Distribution}. Amsterdam: Elsevier.
}
