\name{xyplot.z}
\alias{xyplot.z}
\title{ Display a Third Variable in a X-Y Plot using Proportional Symbols }
\description{
Displays a third variable where the data are represented by open circles whose diameters are proportional to the value of the data at their x-y locations.  The rate of change of symbol diameter with value and the absolute size of the symbols are defined by the user.  The x-y plot axes may be optionally displayed with logarithmic (base 10) scaling.  Optionally a legend may be displayed on the plot.
}
\usage{
xyplot.z(xx, yy, zz, p = 0.5, sfact = 2.5, zmin = NA, zmax = NA, log = NULL, 
	xlim = NULL, ylim = NULL, xlab = deparse(substitute(xx)), 
	ylab = deparse(substitute(yy)), zlab = deparse(substitute(zz)), 
	main = "", symcolr = 1, ifparams = FALSE, iflgnd = FALSE,
	title = deparse(substitute(zz)), ...)
}
\arguments{
  \item{xx}{ name of the x-axis variable. }
  \item{yy}{ name of the y-axis variable. }
  \item{zz}{ name of the third variable to be plotted. }
  \item{p}{ a parameter that controls the rate of change of symbol diameter with changing value.  A default of \code{p = 0.5} is provided.  See Details below. }
  \item{sfact}{ controls the absolute size of the plotted symbols, by default \code{sfact = 2.5}.  Increasing sfact results in larger symbols. }
  \item{zmin}{ a value below which all symbols will be plotted at the same minumum size.  By default \code{zmin = NA} which results in the minimum value of the variable defining the minimum symbol size.  See Details below. }
  \item{zmax}{ a value above which all symbols will be plotted at the same maximum size.  By default \code{zmax = NA} which results in the maximum value of the variable defining the maximum symbol size.  See Details below. }
  \item{log}{ if it is required to display the data with logarithmic axis scaling, set \code{log = "x"} for a logarithmically scaled x-axis, \code{log = "y"} for a logarithmically scaled y-axis, and \code{log = "xy"} for both axes logarithmically scaled. }
  \item{xlim}{ user defined limits for the x-axis, see Details below. }
  \item{ylim}{ user defined limits for the y-axis, see Details below. }
  \item{xlab}{  by default the character string for \code{xx} is used for the x-axis title.  An alternate title can be displayed with \code{xlab = "text string"}, see Examples. }
  \item{ylab}{ by default the character string for \code{yy} is used for the y-axis title.  An alternate title can be displayed with \code{ylab = "text string"}, see Examples. }
  \item{zlab}{ by default, \code{zlab = deparse(substitute(zz))}, a plot title is generated by appending the input variable name text string to \code{"Proportional Symbol Plot for "}. Alternative titles may be generated, see Details below. }
  \item{main}{ an alternative plot title, see Details below.}
  \item{symcolr}{ the colour of the symbols, the default is black, \code{symcolr = 1}.  This may be changed if required, see \code{\link{display.lty}} for the default colour palette.  For example, \code{symcolr = 2} will cause the symbols to be plotted in red. }
  \item{ifparams}{if \code{ifparams = TRUE} on completion of plotting and after the legend has been plotted, if requested, the cursor is activated, locate that at the top left corner of the desired text position and \sQuote{left button} on the pointing device.  This text comprises three lines: the values of \code{p} to three significant figures and \code{sfact}; the maximum value of \code{z} to 3 significant figures and \code{zmax}; and the minimum value of \code{z} to 3 significant figures and \code{zmin}.  The default is no text display. }
  \item{iflgnd}{ the default is no legend.  If a legend is required set \code{iflgnd = TRUE}, following the plotting of the data the cursor will be activated, locate that at the top left corner of the desired legend position and \sQuote{left button} on the pointing device.  See Notes below.}
  \item{title}{ a short title for the legend, e.g., \code{title = "Zn (mg/kg"}.  The default is the variable name. }
  \item{\dots}{ further arguments to be passed to methods.  For example, if smaller plotting characters are required for the legend, specify, for example, \code{cex = 0.8};  and if some other colour than black is required for the legend, specify, for example, \code{col = 3}, to obtain blue.  See \code{\link{display.lty}} for the default colour palette.    If it is required to make the plot title smaller, add \code{cex.main = 0.9} to reduce the font size by 10\%. }
}
\details{
The symbol diameter is computed as a function of the value \code{z} to be plotted:\cr
\code{diameter = dmin + (dmax - dmin) * {(z - zmin)/(zmax - zmin)}^p}\cr
where dmin and dmax are defined as 0.1 and 1 units, so the symbol diameters range over an order of magnitude (and symbol areas over two); \code{zmin} and \code{zmax} are the observed range of the data, or the range over which the user wants the diameters to be computed; and \code{p} is a power defined by the user.  The value of \code{(z - zmin)/(zmax - zmin)} is the value of \code{z} normalized, 0 - 1, to the range over which the symbol diameters are to be computed.  After being raised to the power \code{p}, which will result in a number in the range 0 to 1, this value is multiplied by the permissable range of diameters and added to the minimum diameter.  This results in a diameter between 0.1 and 1 units that is proportional to the value of \code{z}.

A p value of \code{1} results in a linear rate of change.  Values of \code{p} less than unity lead to a rapid intial rate of change with increasing value of \code{z} which is often suitable for displaying positively skewed data sets, see the example below.  In contrast, values of \code{p} greater than unity result in an initial slow rate of change with increasing value of \code{z} which is often suitable for displaying negatively skewed data sets.  Experimentation is usually necessary to obtain a satisfactory visual effect.  See \code{\link{syms.pfunc}} for a graphic demonstrating the effect of varying the \code{p} parameter.

The user may choose to transform the variable to be plotted prior to determining symbol size etc., e.g. \code{log10(zz)}, to generate a logarithmic rate of symbol size change. See Example below.

If \code{zmin} or \code{zmax} are defined this has the effect of setting a minimum or maximum value of \code{z}, respectively, beyond which changes in the value of z do not result in changes in symbol diameter.  This can be useful in limiting the effect of one, or a few, extreme outlier(s) while still plotting them, they simply plot at the minimum or maximum symbol size and are not involved in the calculation of the range of \code{z} over which the symbol diameters vary.  \bold{Note:} If the variable \code{z} includes a transform, e.g., \code{log10(z)}, the values of \code{zmin} and/or \code{zmax} must be in those transform units. 

If \code{zlab} and \code{main} are undefined a default a plot title is generated by appending the input variable name text string to \code{"Proportional Symbol Plot for "}.  If no plot title is required set \code{zlab = " "}, and if some user defined plot title is required it should be defined in \code{main}, e.g. \code{main = "Plot Title Text"}.

If the default selection for \code{xlim} is inappropriate it can be set, e.g., \code{xlim = c(0, 200)} or \code{c(2, 200)}, the latter being appropriate for a logarithmically scaled plot, i.e. \code{log  = "x"}.  If the defined limits lie within the observed data range a truncated plot will be displayed.  The same procedure applies to setting \code{ylim}.   
}
\note{
Any less than detection limit values represented by negative values, or zeros or other numeric codes representing blanks in the data, must be removed prior to executing this function, see \code{\link{ltdl.fix.df}}.

The legend consists of five proportional symbols and their corresponding \code{z} values: \code{zmin}; the three quartiles; and \code{zmax}.  If \code{zmin} and \code{zmax} have been user defined it is over their range that the symbol sizes are computed and displayed.  When defining \code{zmin} and/or \code{zmax} it is useful to set \code{ifparams = TRUE} as a reminder, whilst developing the required display.

Any data vectors containing \code{NA}s are removed prior to displaying the plot.
}
\author{ Robert G. Garrett }
\seealso{ \code{\link{syms}}, \code{\link{syms.pfunc}}, \code{\link{ltdl.fix.df}}, \code{\link{remove.na}}, \code{\link{display.lty}} }  
\examples{ 
## Make test data available
data(sind)
attach(sind)

## Display a default symbol plot, p = 0.5 and sfact = 2.5
xyplot.z(Fe, Mn, Zn)

## Plot with logarithmically scaled symbols and more appropriately
## labelled axes
xyplot.z(Fe, Mn, log10(Zn), p = 1, log = "y",
	xlab = "Fe (pct) in stream sediment",
	ylab = "Mn (mg/kg) in stream sediment") 

## Plot with differently scaled symbols in red and more appropriate
## scaling and labelling with a user specified title
xyplot.z(Fe, Mn, Zn, p = 0.3, sfact = 2.0, log = "y",
	xlab = "Fe (pct) in stream sediment",
	ylab = "Mn (mg/kg) in stream sediment",
	main = "Howarth & Sinding-Larsen Test Data\nZn (mg/kg)", symcolr = 2) 

## Plot as above but where outliers above a value of 1000 displayed
## with the same symbol
xyplot.z(Fe, Mn, Zn, p = 0.3, sfact = 2.0, zmax = 1000, log = "y",
	xlab = "Fe (pct) in stream sediment",
	ylab = "Mn (mg/kg) in stream sediment", 
	main = "Howarth & Sinding-Larsen Test Data\nZn (mg/kg)", symcolr = 2) 

## Detach test data
detach(sind) 
}
\keyword{ hplot }


