#' Overview of User's System Profile
#'
#' @name profile
#'
#' @aliases position
#'
#' @description This is an auth based function. User must have valid api keys generated by GDAX which must be passed as mandatory arguments. This function takes no additional arguments and returns system identifiable information for the user and the status of the profile. If \code{active}, the profile can be used for trading. If \code{pending}, the profile is currently being created. If \code{locked}, the profile is undergoing a rebalance. If \code{default}, you were not able repay funding after a margin call or expired funding and now have a default. The complete \code{profile} response gets a lot of information which can be more easily fetched by \code{\link{accounts}}. This functions trims down the complete response only to the information that cannot be sought by other methods.
#'
#' @param api.key Mandatory character value. This is the API key as generated by GDAX. Typically a 32 character value.
#' @param secret Mandatory character value. This is the API secret as generated by GDAX. Typically a 88 character value.
#' @param passphrase Mandatory character value. This is the passphrase as generated by GDAX. Typically a 11 character value.
#'
#' @return  A dataframe with current status, system generated user id and profile id.
#'
#' @examples
#' \dontrun{
#' profile(api.key = your_key, secret = your_api_secret, passphrase = your_api_pass)
#' }
#'
#' @export

profile <- function(api.key,
                    secret,
                    passphrase) {
  #get url extension----
  req.url = "/position"

  #define method----
  method <- "GET"

  #fetch response----
  response <- auth(
    method = method,
    req.url = req.url,
    api.key = api.key,
    secret = secret,
    passphrase = passphrase
  )

  #transform----
  # drop accounts as they can be fetched by accounts.
  response <- response[-which(names(response) == "accounts")]
  response <- as.data.frame(response)

  #return----
  return(response)
}
