% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictive.R
\name{predict.evpost}
\alias{predict.evpost}
\title{Predictive inference for the largest value observed in N years.}
\usage{
\method{predict}{evpost}(object, type = c("i", "p", "d", "q", "r"),
  x = NULL, x_num = 100, n_years = 100, npy = NULL, level = 95,
  hpd = FALSE, lower_tail = TRUE, log = FALSE, big_q = 1000, ...)
}
\arguments{
\item{object}{An object of class "evpost", a result of a call to
\code{\link{rpost}} or \code{\link{rpost_rcpp}} with \code{model = "gev"},
\code{model = "os"}, \code{model = "pp"} or \code{model == "bingp"}.
Calling these functions after a call to \code{rpost} or \code{rpost_rcpp}
with \code{model == "gp"} will produce an error, because inferences about
the probability of threshold exceedance are required, in addition to the
distribution of threshold excesses. The model is stored in
\code{object$model}.}

\item{type}{A character vector.  Indicates which type of inference is
  required:
\itemize{
  \item "i" for predictive intervals,
  \item "p" for the predictive distribution function,
  \item "d" for the predictive density function,
  \item "q" for the predictive quantile function,
  \item "r" for random generation from the predictive distribution.
}}

\item{x}{A numeric vector or a matrix with \code{n_years} columns.
  The meaning of \code{x} depends on \code{type}.
  \itemize{
    \item{\code{type = "p"} or \code{type = "d"}:} \code{x} contains
      quantiles at which to evaluate the distribution or density function.

      If \code{object$model == "bingp"} then no element of \code{x} can be
      less than the threshold \code{object$thresh}.

      If \code{x} is not supplied then \code{n_year}-specific defaults are
      set: vectors of length \code{x_num} from the 0.1\% quantile to the
      99\% quantile, subject all values being greater than the threshold.

    \item{\code{type = "q"}:} \code{x} contains probabilities in (0,1)
      at which to evaluate the quantile function.  Any values outside
      (0, 1) will be removed without warning.

      If \code{object$model == "bingp"} then no element of \code{p} can
      correspond to a predictive quantile that is below the threshold,
      \code{object$thresh}.  That is, no element of \code{p} can be less
      than the value of \code{predict.evpost(object,}
      \code{type = "q", x = object$thresh)}.

      If \code{x} is not supplied then a default value of
      \code{c(0.025, 0.25, 0.5, 0.75, 0.975)} is used.
    \item{\code{type = "i"} or \code{type = "r"}:} \code{x} is not relevant.
  }}

\item{x_num}{A numeric scalar.  If \code{type = "p"} or \code{type = "d"}
and \code{x} is not supplied then \code{x_num} gives the number of values
in \code{x} for each value in \code{n_years}.}

\item{n_years}{A numeric vector. Values of N.}

\item{npy}{A numeric scalar. The mean number of observations per year
  of data, after excluding any missing values, i.e. the number of
  non-missing observations divided by total number of years of non-missing
  data.

If \code{rpost} or \code{rpost_rcpp} was called with
\code{model == "bingp"} then \code{npy} must either have been supplied
in that call or be supplied here.

Otherwise, a default value will be assumed if \code{npy} is not supplied,
based on the value of \code{model} in the call to \code{rpost} or
\code{rpost_rcpp}:
\itemize{
  \item{\code{model = "gev"}:} \code{npy} = 1, i.e. the data were
    annual maxima so the block size is one year.
  \item{\code{model = "os"}:} \code{npy} = 1, i.e. the data were
    annual order statistics so the block size is one year.
  \item{\code{model = "pp"}:}
    \code{npy} = \code{length(x$data)} / \code{object$noy},
    i.e. the value of \code{noy} used in the call to \code{\link{rpost}}
    or \code{\link{rpost_rcpp}} is equated to a block size of one year.
}
If \code{npy} is supplied twice then the value supplied here will be
used and a warning given.}

\item{level}{A numeric vector of values in (0, 100).
Only relevant when \code{type = "i"}.
Levels of predictive intervals for the largest value observed in
N years, i.e. level\% predictive intervals are returned.}

\item{hpd}{A logical scalar.
  Only relevant when \code{type = "i"}.

  If \code{hpd = FALSE} then the interval is
  equi-tailed, with its limits produced by
  \code{predict.evpost(}\code{object, type ="q", x = p)},
  where \code{p = c((1-level/100)/2,} \code{(1+level/100)/2)}.

  If \code{hpd = TRUE} then, in addition to the equi-tailed interval,
  the shortest possible level\% interval is calculated.
  If the predictive distribution is unimodal then this
  is a highest predictive density (HPD) interval.}

\item{lower_tail}{A logical scalar.
Only relevant when \code{type = "p"} or \code{type = "q"}.
If TRUE (default), (output or input) probabilities are
\eqn{P[X \leq x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}

\item{log}{A logical scalar.  Only relevant when \code{type = "d"}.
If TRUE the log-density is returned.}

\item{big_q}{A numeric scalar.  Only relevant when \code{type = "q"}.
An initial upper bound for the desired quantiles to be passed to
\code{\link[stats]{uniroot}} (its argument \code{upper}) in the
search for the predictive quantiles.  If this is not sufficiently large
then it is increased until it does provide an upper bound.}

\item{...}{Additional optional arguments. At present no optional
arguments are used.}
}
\value{
An object of class "evpred", a list containing a subset of the
  following components:
    \item{type}{The argument \code{type} supplied to \code{predict.evpost}.
    Which of the following components are present depends \code{type}.}
    \item{x}{A matrix containing the argument \code{x} supplied to
      \code{predict.evpost}, or set within \code{predict.evpost} if \code{x}
      was not supplied, replicated to have \code{n_years} columns
      if necessary.
      Only present if \code{type} is \code{"p", "d"} or \code{"q"}.}
    \item{y}{The content of \code{y} depends on \code{type}:
    \itemize{
      \item{\code{type = "p", "d", "q"}:}  A matrix with the same
      dimensions as \code{x}.  Contains distribution function values
      (\code{type = "p"}), predictive density (\code{type = "d"})
      or quantiles (\code{type = "q"}).
      \item{\code{type = "r"}:} A numeric matrix with \code{length(n_years)}
      columns and number of rows equal to the size of the posterior sample.
      \item{\code{type = "i"}:} \code{y} is not present.
      }}
      \item{long}{A \code{length(n_years)*length(level)} by 4 numeric
        matrix containing the equi-tailed limits with columns:
        lower limit, upper limit, n_years, level.
        Only present if \code{type = "i"}.  If an interval extends below
        the threshold then \code{NA} is returned.}
      \item{short}{A matrix with the same structure as \code{long}
        containing the HPD limits.  Only present if \code{type = "i"}.
        Columns 1 and 2 contain \code{NA}s if \code{hpd = FALSE}
        or if the corresponding equi-tailed interval extends below
        the threshold.}
  The arguments \code{n_years, level, hpd, lower_tail, log} supplied
  to \code{predict.evpost} are also included, as is the argument \code{npy}
  supplied to, or set within, \code{predict.evpost} and
  the arguments \code{data} and \code{model} from the original call to
  \code{\link{rpost}} or \code{\link{rpost_rcpp}}.
}
\description{
\code{predict} method for class "evpost".  Performs predictive inference
about the largest value to be observed over a future time period of
N years.  Predictive inferences accounts for uncertainty in model
parameters and for uncertainty owing to the variability of future
observations.
}
\details{
Inferences about future extreme observations are integrated over
  the posterior distribution of the model parameters, thereby accounting
  for uncertainty in model parameters and uncertainty owing to the
  variability of future observations.  In practice the integrals involved
  are estimated using an empirical mean over the posterior sample.
  See, for example,
  \href{https://doi.org/10.1007/978-1-4471-3675-0_9}{Coles (2001),
  chapter 9},
  \href{https://doi.org/10.1201/b19721}{Stephenson (2016)}
  or
  \href{https://doi.org/10.1111/rssc.12159}{Northrop et al. (2017)}
  for details.
  See also the vignette
  \href{https://CRAN.R-project.org/package=revdbayes}{Posterior Predictive Extreme Value Inference}

  \strong{GEV / OS / PP}.
  If \code{model = "gev"}, \code{model = "os"} or \code{model = "pp"}
  in the call to \code{\link{rpost}} or \code{\link{rpost_rcpp}}
  we first calculate the number of blocks \eqn{b} in \code{n_years} years.
  To calculate the density function or distribution function of the maximum
  over \code{n_years} we call \code{\link{dgev}} or \code{\link{pgev}}
  with \code{m} = \eqn{b}.

  \itemize{
    \item{\code{type = "p"}.} We calculate using \code{\link{pgev}}
    the GEV distribution function at \code{q} for each of the posterior
    samples of the location, scale and shape parameters.  Then we take
    the mean of these values.

    \item{\code{type = "d"}.} We calculate using \code{\link{dgev}}
    the GEV density function at \code{x} for each of the posterior samples
    of the location, scale and shape parameters.  Then we take the
    mean of these values.

    \item{\code{type = "q"}.} We solve numerically
    \code{predict.evpost(object, type = "p", x = q)} = \code{p[i]}
    numerically for \code{q} for each element \code{p[i]} of \code{p}.

    \item{\code{type = "i"}.} If \code{hpd = FALSE} then the interval is
    equi-tailed, equal to \code{predict.evpost()} \code{object, type ="q", x = p)},
    where \code{p = c((1-level/100)/2,} \code{(1+level/100)/2)}.
    If \code{hpd = TRUE} then, in addition, we perform a
    numerical minimisation of the length of level\% intervals, after
    approximating the predictive quantile function using monotonic
    cubic splines, to reduce computing time.

    \item{\code{type = "r"}.} For each simulated value of the GEV parameters
    at the \code{n_years} level of aggregation we simulate one value from
    this GEV distribution using \code{\link{rgev}}.  Thus, each sample
    from the predictive distribution is of a size equal to the size of
    the posterior sample.
  }

  \strong{Binomial-GP}.  If \code{model = "bingp"} in the call to
  \code{\link{rpost}} or \code{\link{rpost_rcpp}} then we calculate the
  mean number of observations in \code{n_years} years, i.e.
  \code{npy * n_years}.

  Following \href{https://doi.org/10.1111/rssc.12159}{Northrop et al. (2017)}
  let \eqn{M_N} be the largest value observed in \eqn{N} years,
  \eqn{m} = \code{npy * n_years} and \eqn{u} the threshold
  \code{object$thresh} used in the call to \code{rpost}
  or \code{rpost_rcpp}.
  For fixed values of \eqn{\theta = (p, \sigma, \xi)} the distribution
  function of \eqn{M_N} is given by \eqn{F(z, \theta)^m}, for
  \eqn{z \geq u}{z >= u}, where
  \deqn{F(z, \theta) = 1 - p [1 + \xi (x - u) / \sigma] ^ {-1/\xi}.}{%
        F(z, \theta) = 1 - p * [1 + \xi (x - u) / \sigma] ^ (-1/\xi).}
  The distribution function of \eqn{M_N} cannot be evaluated for
  \eqn{z < u} because no model has been supposed for observations below
  the threshold.

\itemize{
  \item{\code{type = "p"}.} We calculate
    \eqn{F(z, \theta)^m} at \code{q} for each of the posterior samples
    \eqn{\theta}.  Then we take the mean of these values.
  \item{\code{type = "d"}.}  We calculate the density of of \eqn{M_n}, i.e.
    the derivative of \eqn{F(z, \theta)^m} with respect to \eqn{z} at
    \code{x} for each of the posterior samples \eqn{\theta}.  Then we take
    the mean of these values.
  \item{\code{type = "q"} and \code{type = "i"}} We perform calculations
    that are analogous to the GEV case above.  If \code{n_years} is very
    small and/or level is very close to 100 then a predictive interval
    may extend below the threshold.  In such cases \code{NA}s are returned
    (see \strong{Value} below).
  \item{\code{type = "r"}.}  For each simulated value of the bin-GP
    parameter we simulate from the distribution of \eqn{M_N} using the
    inversion method applied to the distribution function of \eqn{M_N} given
    above.  Occasionally a value below the threshold would need to be
    simulated.  If these instances a missing value code \code{NA} is
    returned. Thus, each sample from the predictive distribution is of a
    size equal to the size of the posterior sample, perhaps with a small
    number os \code{NA}s.
  }
}
\examples{
### GEV
data(portpirie)
mat <- diag(c(10000, 10000, 100))
pn <- set_prior(prior = "norm", model = "gev", mean = c(0,0,0), cov = mat)
gevp  <- rpost_rcpp(n = 1000, model = "gev", prior = pn, data = portpirie)

# Interval estimation
predict(gevp)$long
predict(gevp, hpd = TRUE)$short
# Density function
x <- 4:7
predict(gevp, type = "d", x = x)$y
plot(predict(gevp, type = "d", n_years = c(100, 1000)))
# Distribution function
predict(gevp, type = "p", x = x)$y
plot(predict(gevp, type = "p", n_years = c(100, 1000)))
# Quantiles
predict(gevp, type = "q", n_years = c(100, 1000))$y
# Random generation
plot(predict(gevp, type = "r"))

### Binomial-GP
data(gom)
u <- quantile(gom, probs = 0.65)
fp <- set_prior(prior = "flat", model = "gp", min_xi = -1)
bp <- set_bin_prior(prior = "jeffreys")
npy_gom <- length(gom)/105
bgpg <- rpost_rcpp(n = 1000, model = "bingp", prior = fp, thresh = u,
                   data = gom, bin_prior = bp)

# Setting npy in call to predict.evpost()
predict(bgpg, npy = npy_gom)$long

# Setting npy in call to rpost() or rpost_rcpp()
bgpg <- rpost_rcpp(n = 1000, model = "bingp", prior = fp, thresh = u,
                   data = gom, bin_prior = bp, npy = npy_gom)

# Interval estimation
predict(bgpg)$long
predict(bgpg, hpd = TRUE)$short
# Density function
plot(predict(bgpg, type = "d", n_years = c(100, 1000)))
# Distribution function
plot(predict(bgpg, type = "p", n_years = c(100, 1000)))
# Quantiles
predict(bgpg, type = "q", n_years = c(100, 1000))$y
# Random generation
plot(predict(bgpg, type = "r"))
}
\references{
Coles, S. G. (2001) \emph{An Introduction to Statistical
  Modeling of Extreme Values}, Springer-Verlag, London.
  Chapter 9: \url{https://doi.org/10.1007/978-1-4471-3675-0_9}

Northrop, P. J., Attalides, N. and Jonathan, P. (2017)
  Cross-validatory extreme value threshold selection and uncertainty
  with application to ocean storm severity.
  \emph{Journal of the Royal Statistical Society Series C: Applied
  Statistics}, \strong{66}(1), 93-120.
  \url{https://doi.org/10.1111/rssc.12159}

Stephenson, A. (2016). Bayesian Inference for Extreme Value
  Modelling. In \emph{Extreme Value Modeling and Risk Analysis: Methods and
  Applications}, edited by D. K. Dey and J. Yan, 257-80. London:
  Chapman and Hall. \url{https://doi.org/10.1201/b19721}
}
\seealso{
\code{\link{plot.evpred}} for the S3 \code{plot} method for
  objects of class \code{evpred}.

\code{\link{rpost}} or \code{\link{rpost_rcpp}} for sampling
  from an extreme value posterior distribution.
}
