% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rposterior.R
\name{rpost}
\alias{rpost}
\title{Random sampling from extreme value posterior distributions}
\usage{
rpost(n, model = c("gp", "gev", "pp", "os"), data, prior, thresh = NULL,
  noy = NULL, use_noy = TRUE, ros = NULL, init_ests = NULL, mult = 2,
  use_phi_map = FALSE, ...)
}
\arguments{
\item{n}{A numeric scalar. The size of posterior sample required.}

\item{model}{A character string.  Specifies the extreme value model.}

\item{data}{Sample data, of a format appropriate for the model.
\itemize{
  \item {"gp"} {A numeric vector of threshold excesses or raw data.}
  \item {"gev"} {A numeric vector of block maxima.}
  \item {"pp"} {A numeric vector of raw data.}
  \item {"os"} {A numeric matrix or data frame. Each row should contain
    the largest order statistics for a block of data.  These needs not
    be ordered because they are sorted inside \code{rpost}. If a block
    contains fewer than \code{dim(as.matrix(data)[2]} order statistics
    then the corresponding row should be padded by \code{NA}s. If
    \code{ros} is supplied then only the largest \code{ros} values in
    each row are used.}
}}

\item{prior}{A function to evaluate the prior, created by
\code{\link{set_prior}}.}

\item{thresh}{A numeric scalar.  Extreme value threshold applied to data.
Only relevant when \code{model = "gp"} or \code{model = "pp"}.  Must
be supplied when \code{model = "pp"}.  If \code{model = "gp"} and
\code{thresh} is not supplied then \code{thresh = 0} is used and
\code{data} should contain threshold excesses.}

\item{noy}{A numeric scalar  The number of observations per block, where a
block is often a year.  Only relevant, and must be supplied, if
\code{model = "pp"}.}

\item{use_noy}{A logical scalar.  Only relevant if model is "pp".  By
default (\code{use_noy = FALSE}) sampling is based on a likelihood in
which the number of blocks (years) is set equal to the number of threshold
excesses, to reduce posterior dependence between the parameters.  The
sampled values are transformed back to the required parameterisation
before returning them to the user.  If \code{use_noy = TRUE} then the
user's value of \code{noy} is used in the likelihood.}

\item{ros}{A numeric scalar.  Only relevant when \code{model = "os"}. The
largest \code{ros} values in each row of the matrix \code{data} are used
in the analysis.}

\item{init_ests}{A numeric vector.  Initial parameter estimates for search
for the mode of the posterior distribution.}

\item{mult}{A numeric scalar.  The grid of values used to choose the Box-Cox
transformation parameter lambda is based on the maximum aposteriori (MAP)
estimate +/- mult x estimated posterior standard deviation.}

\item{use_phi_map}{A logical scalar. If trans = "BC" then \code{use_phi_map}
determines whether the grid of values for phi used to set lambda is
centred on the maximum a posterior (MAP) estimate of phi
(\code{use_phi_map = TRUE}), or on the initial estimate of phi
(\code{use_phi_map = FALSE}).}

\item{...}{Further argments to be passed to \code{\link[rust]{ru}}.  Most
importantly \code{trans} and \code{rotate} (see \strong{Details}), and
perhaps \code{r}, \code{ep}, \code{a_algor}, \code{b_algor},
\code{a_method}, \code{b_method}, \code{a_control}, \code{b_control}.
May also be used to pass the arguments \code{n_grid} and/or \code{ep_bc}
to \code{\link[rust]{find_lambda}}.}
}
\value{
An object of class "evpost", which has the same structure as
  an object of class "ru" returned from \code{\link[rust]{ru}}.
}
\description{
Uses the \code{\link[rust]{ru}} function in the \code{\link[rust]{rust}}
package to simulate from the posterior distribution of an extreme value
model.
}
\details{
\emph{Generalised Pareto (GP)}: \code{model = "gp"}.  A model for threshold
  excesses.  Required arguments: \code{n}, \code{data} and \code{prior}.
  If \code{thresh} is supplied then only the values in \code{data} that
  exceed \code{thresh} are used and the GP distribution is fitted to the
  amounts by which those values exceed \code{thresh}.
  If \code{thresh} is not supplied then the GP distribution is fitted to
  all values in \code{data}, in effect \code{thresh = 0}.

\emph{Generalised extreme value (GEV) model}: \code{model = "gev"}.  A
  model for block maxima.  Required arguments: \code{n}, \code{data},
  \code{prior}.

\emph{Point process (PP) model}: \code{model = "pp"}. A model for
  occurrences of threshold exceedances and threshold excesses.  Required
  arguments: \code{n}, \code{data}, \code{prior}, \code{thresh} and
  \code{noy}.

\emph{r-largest order statistics (OS) model}: \code{model = "os"}.
  A model for the largest order statistics within blocks of data.
  Required arguments: \code{n}, \code{data}, \code{prior}.  All the values
  in \code{data} are used unless \code{ros} is supplied.

\emph{Parameter transformation}.  The scalar logical arguments (to the
  function \code{\link{ru}}) \code{trans} and \code{rotate} determine,
  respectively, whether or not Box-Cox transformation is used to reduce
  asymmetry in the posterior distribution and rotation of parameter
  axes is used to reduce posterior parameter dependence.  The default
  is \code{trans = "none"} and \code{rotate = TRUE}.

See the revdbayes vignette for further details and examples.
}
\examples{
# GP model
data(gom)
u <- quantile(gom, probs = 0.65)
fp <- set_prior(prior = "flat", model = "gp", min_xi = -1)
gpg <- rpost(n = 1000, model = "gp", prior = fp, thresh = u, data = gom)
plot(gpg)

# GEV model
data(portpirie)
mat <- diag(c(10000, 10000, 100))
pn <- set_prior(prior = "norm", model = "gev", mean = c(0,0,0), cov = mat)
gevp  <- rpost(n = 1000, model = "gev", prior = pn, data = portpirie)
plot(gevp)

# PP model
data(rainfall)
rthresh <- 40
pf <- set_prior(prior = "flat", model = "gev", min_xi = -1)
ppr <- rpost(n = 1000, model = "pp", prior = pf, data = rainfall,
  thresh = rthresh, noy = 54)
plot(ppr)

# OS model
data(venice)
mat <- diag(c(10000, 10000, 100))
pv <- set_prior(prior = "norm", model = "gev", mean = c(0,0,0), cov = mat)
osv <- rpost(n = 1000, model = "os", prior = pv, data = venice)
plot(osv)
}
\seealso{
\code{\link{set_prior}} for setting a prior distribution.

\code{\link[rust]{ru}} in the \code{\link[rust]{rust}}
  package for details of the arguments that can be passed to \code{ru} and
  the form of the object returned by \code{rpost}.

\code{\link[rust]{find_lambda}} in the
  \code{\link[rust]{rust}} package is used inside \code{rpost} to set the
  Box-Cox transformation parameter lambda when the \code{trans = "BC"}
  argument is given.

\code{\link[evdbayes]{posterior}} for sampling from an extreme
  value posterior using the evdbayes package.
}

