#' Load plain-text data from a URL (either http or https)
#'
#' \code{source_data} loads plain-text formatted data stored at a URL (both http
#' and https) into R.
#' @param url The plain-text formatted data's URL.
#' @param sha1 Character string of the file's SHA-1 hash, generated by
#' \code{source_data}. Note if you are using data stored using Git, this is not
#' the file's commit SHA-1 hash.
#' @param cache logical. Whether or not to cache the data so that it is not
#' downloaded every time the function is called.
#' @param clearCache logical. Whether or not to clear the downloaded data from
#' the cache.
#' @param sep The separator method for the data. For example, to load
#' comma-separated values data (CSV) use \code{sep = ","} (the default). To load
#'  tab-separated values data (TSV) use \code{sep = "\t"}.
#' @param quote  the set of quoting characters. To disable quoting altogether,
#' use \code{quote = ""}. See \code{\link{scan}} for the behaviour on quotes
#' embedded in quotes.
#' @param header Logical, whether or not the first line of the file is the
#' header (i.e. variable names). The default is \code{header = TRUE}.
#' @param stringsAsFactors logical. Should character vectors be converted to
#' factors? Note that this is overridden by \code{as.is} and \code{colClasses},
#' both of which allow finer control.
#' @param ... additional arguments passed to \code{\link{read.table}}.
#' 
#' @return a data frame
#' 
#' @details Loads plain-text data (e.g. CSV, TSV) data from a URL. Works with
#' both HTTP and HTTPS sites. Note: the URL you give for the \code{url} argument
#' must be for the RAW version of the file. The function should work to download
#' plain-text data from any secure URL (https), though I have not verified this.
#'
#' From the \code{source_url} documentation: "If a SHA-1 hash is specified with
#' the \code{sha1} argument, then this function will check the SHA-1 hash of the
#' downloaded file to make sure it matches the expected value, and throw an error
#' if it does not match. If the SHA-1 hash is not specified, it will print a
#' message displaying the hash of the downloaded file. The purpose of this is to
#'  improve security when running remotely-hosted code; if you have a hash of
#' the file, you can be sure that it has not changed."
#' @examples
#' \dontrun{
#' # Download electoral disproportionality data stored on GitHub
#' # Note: Using shortened URL created by bitly
#' DisData <- source_data("http://bit.ly/156oQ7a")
#'
#' # Check to see if SHA-1 hash matches downloaded file
#' DisDataHash <- source_data("http://bit.ly/Ss6zDO",
#'    sha1 = "dc8110d6dff32f682bd2f2fdbacb89e37b94f95d")
#' }
#' @source Based on source_url from the Hadley Wickham's devtools package.
#' @seealso \link{httr} and \code{\link{read.table}}
#' @importFrom digest digest
#' @importFrom httr GET stop_for_status text_content content
#' @importFrom R.cache saveCache loadCache findCache
#' @export

source_data <-function(url, sha1 = NULL, cache = FALSE, clearCache = FALSE,
                       sep = ",", quote = "\"'", header = TRUE,
                       stringsAsFactors = default.stringsAsFactors(), ...)
{
    stopifnot(is.character(url), length(url) == 1)

    temp_file <- tempfile()
    on.exit(unlink(temp_file))

    key <- as.list(url)
    if (isTRUE(clearCache)){
        Found <- findCache(key = key)
        if (is.null(Found)){
            message('Data not in cache. Nothing to remove.')
        }
        else if (!is.null(Found)){
            message('Clearing data from cache.\n')
            file.remove(Found)
        }
    }

    if (isTRUE(cache)){
        data <- loadCache(key)
        if (!is.null(data)){
            message('Loading cached data.\n')
            return(data);
        }
        data <- download_data_intern(url = url, sha1 = sha1,
                                    temp_file = temp_file)
        data <- read.table(data, sep = sep, quote = quote, header = header,
                       stringsAsFactors = stringsAsFactors, ...)
        saveCache(data, key = key)
        data;
    }
    else if (!isTRUE(cache)){
        data <- download_data_intern(url = url, sha1 = sha1,
                                    temp_file = temp_file)
        data <- read.table(data, sep = sep, quote = quote, header = header,
                            stringsAsFactors = stringsAsFactors, ...)
        return(data)
    }
}
