% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fieldCastingFunctions.R
\name{fieldCastingFunctions}
\alias{fieldCastingFunctions}
\alias{recastRecords}
\alias{castForImport}
\alias{guessCast}
\alias{guessDate}
\alias{mChoiceCast}
\title{Functions for Casting Fields After Export (Post Processing)}
\usage{
recastRecords(data, rcon, fields, cast = list(), suffix = "")

castForImport(
  data,
  rcon,
  fields = NULL,
  na = list(),
  validation = list(),
  cast = list()
)

guessCast(
  data,
  rcon,
  na = isNAorBlank,
  validation,
  cast,
  quiet = FALSE,
  threshold = 0.8
)

guessDate(
  data,
  rcon,
  na = isNAorBlank,
  validation = valRx("^[0-9]{1,4}-(0?[1-9]|1[012])-(0?[1-9]|[12][0-9]|3[01])$"),
  cast = function(x, ...) as.POSIXct(x, format = "\%Y-\%m-\%d"),
  quiet = FALSE,
  threshold = 0.8
)

mChoiceCast(data, rcon, style = "labelled", drop_fields = TRUE)
}
\arguments{
\item{data}{\code{data.frame} with the data fields to be recoded.}

\item{rcon}{A \code{redcapConnection} object.}

\item{fields}{\code{character/logical/integerish}. A vector for identifying
which fields to recode. When \code{logical}, the length must match
the number of columns in \code{data} (i.e., recycling not permitted).
A message is printed if any of the indicated fields are not a
multiple choice field; no action will be taken on such fields.
For this function, yes/no and true/false fields are considered
multiple choice fields. Fields of class \code{mChoice} are quietly skipped.}

\item{cast}{A named \code{list} of user specified class casting functions. The
same named keys are supported as the na argument. The function will be
provided the variables (x, field_name, coding). The function must return a
vector of logical matching the input length. See \code{\link[=fieldValidationAndCasting]{fieldValidationAndCasting()}}}

\item{suffix}{\code{character(1)}. An optional suffix to provide if
the recoded variables should be returned as new columns. For example,
if recoding a field \code{forklift_brand} and \code{suffix = "_labeled"},
the result will have one column with the coded values
(\code{forklift_brand}) and one column with the labeled values
(\code{forklift_brand_labeled}).}

\item{na}{A named \code{list} of user specified functions to determine if the
data is NA. This is useful when data is loaded that has coding for NA, e.g.
-5 is NA. Keys must correspond to a truncated REDCap field type, i.e.
{date_, datetime_, datetime_seconds_, time_mm_ss, time_hh_mm_ss, time, float,
number, calc, int, integer, select, radio, dropdown, yesno, truefalse,
checkbox, form_complete, sql}. The function will be provided the variables
(x, field_name, coding). The function must return a vector of logicals
matching the input. It defaults to \code{\link[=isNAorBlank]{isNAorBlank()}} for all
entries.}

\item{validation}{A named \code{list} of user specified validation functions. The
same named keys are supported as the na argument. The function will be
provided the variables (x, field_name, coding). The function must return a
vector of logical matching the input length. Helper functions to construct
these are \code{\link[=valRx]{valRx()}} and \code{\link[=valChoice]{valChoice()}}. Only fields that
are not identified as NA will be passed to validation functions.}

\item{quiet}{Print no messages if triggered, Default=FALSE.}

\item{threshold}{numeric(1). The threshold of non-NA data to trigger casting.}

\item{style}{character. One of "labelled" or "coded". Default is "labelled"}

\item{drop_fields}{\code{character} or \code{NULL}. A vector of field names to remove from
the data.}
}
\description{
The functions provided here allow for recasting fields
after records have been exported. They generally have a similar
interface to the casting strategy of \code{\link[=exportRecordsTyped]{exportRecordsTyped()}},
though they may not each offer all the same options.
}
\details{
\code{recastRecords} is a post-processing function motivated
initially by the need to switch between codes and labels in multiple
choice fields. Field types for which no casting function is specified will
be returned with no changes. It will not attempt to validate the content
of fields; fields that cannot be successfully cast will be quietly
returned as missing values.

\code{castForImport} is written with defaults that will return data
in a format ready to be imported to a project via \code{importRecords}.
All fields are returned as character vectors. If any values fail to
validation check, are report is returned as an attribute named \code{invalid}.
This attribute may be retrieved using \code{\link[=reviewInvalidRecords]{reviewInvalidRecords()}}.
These are then set to \code{NA}, which will be imported as blanks through
the API.

\code{guessCast} is a helper function to make a guess at casting uncast
columns. It will do a type cast if a validation is met above
a threshold ratio of non-NA records. It modifies the existing
\code{invalid} attribute to reflect the cast.
This attribute may be retrieved using \code{\link[=reviewInvalidRecords]{reviewInvalidRecords()}}.
\code{guessDate} is a special cast of \code{guessCast} that has defaults set for casting
a date field.

\code{mChoiceCast} is a helper function that adds the \code{Hmisc::mChoice}
multiple choice class. It adds a column for a multiple choice checkbox
that is cast to the \code{Hmisc::mChoice} class. Requires \code{Hmisc}
to be loaded.
}
\section{Zero-Coded Check Fields}{

A zero-coded check field is a field of the REDCap type \code{checkbox} that has
a coding definition of \verb{0, [label]}. When exported, the field names for
these fields is \verb{[field_name]___0}. As in other checkbox fields, the
raw data output returns binary values where 0 represent and unchecked
box and 1 represents a checked box. For zero-coded checkboxes, then, a
value of 1 indicates that 0 was selected.

This coding rarely presents a problem when casting from raw values
(as is done in \code{exportRecordsTyped}). However, casting from coded or
labeled values can be problematic, as it In this case, it becomes
indeterminate from context if the intent of \code{0} is 'false' or the coded
value '0' ('true') ...

The situations in which casting may fail to produce the desired results are\tabular{lll}{
   Code \tab Label \tab Result \cr
   0 \tab anything other than "0" \tab Likely to fail when casting from coded values \cr
   0 \tab 0 \tab Likely to fail when casting from coded or labeled values \cr
}


Because of the potential for miscast data, casting functions will issue
a warning anytime a zero-coded check field is encountered. A separate
warning is issued when a field is cast from coded or labeled values.

When casting from coded or labeled values, it is strongly recommended that
the function \code{\link[=castCheckForImport]{castCheckForImport()}} be used. This function permits the
user to state explicitly which values should be recognized as checked,
avoiding the ambiguity resulting from the coding.
}

\examples{
\dontrun{
# Using recastRecords after export
Recs <- 
  exportRecordsTyped(rcon) |>
  recastRecords(rcon, 
                fields = "dropdown_test",
                cast = list(dropdown = castCode))
                
                
# Using castForImport
castForImport(Records, 
              rcon)
              
              
# Using castForImport to recast zero-coded checkbox values
castForImport(Records, 
              rcon, 
              cast = list(checkbox = castCheckForImport(c("0", "Unchecked"))))


# Using guessCast
exportRecordsTyped(rcon, 
                           cast = raw_cast) |> 
  guessCast(rcon, 
            validation=valRx("^[0-9]{1,4}-(0?[1-9]|1[012])-(0?[1-9]|[12][0-9]|3[01])$"), 
            cast=as.Date,
            threshold=0.6)
            
            
# Using mChoiceCast
exportRecordsTyped(rcon) |> 
  mChoiceCast(rcon)

}


}
\seealso{
\subsection{Exporting records}{

\code{\link[=exportRecordsTyped]{exportRecordsTyped()}}, \cr
\code{\link[=exportReportsTyped]{exportReportsTyped()}}, \cr
\code{\link[=fieldValidationAndCasting]{fieldValidationAndCasting()}}, \cr
\code{\link[=reviewInvalidRecords]{reviewInvalidRecords()}}
}

\subsection{Other Post Processing Functions}{

\code{\link[=splitForms]{splitForms()}}, \cr
\code{\link[=widerRepeated]{widerRepeated()}}
}
}
