#' @name exportFiles
#' 
#' @title Exports a File attached to a Record
#' @description A single file from a single record is retrieved.  The behavior 
#' of this function is consistent with the behavior of the API, which only 
#' allows one file to be downloaded at a time
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param record The record ID in which the desired file is stored. Must be length 1.
#' @param field The field name in which the file is stored. Must be length 1. 
#' @param event The event name for the file.  Must be length 1.  
#'   This applies only to longitudinal projects.  If the event is not
#'   supplied for a longitudinal project, the API will return an error message
#' @param dir A directory/folder to which the file will be saved. 
#'   By default, the working directory is used
#' @param filePrefix Logical.  Determines if a prefix is appended to the file 
#'   name.  The prefix takes the form [record_id]-[event_name]-[file_name].  
#'   The file name is always the same name of the file as it exists in REDCap
#' @param repeat_instance The repeat instance number of the repeating
#'   event or the repeating instrument. When available in your instance
#'   of REDCap, and passed as NULL, the API will assume a value of 1.
#' @param ... Arguments to be passed to other methods
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @details The function may only export a single file.  
#' See the examples for suggestions on exporting multiple files.
#' 
#' Note that the name of the file can not be changed.  Whatever name exists in 
#' REDCap is the name that will be used, although the record ID and event name 
#' may be appended as a prefix
#' 
#' @section REDCap API Documentation (6.5.0):
#' This method allows you to download a document that has been attached to an 
#' individual record for a File Upload field. Please note that this method may also 
#' be used for Signature fields (i.e. File Upload fields with "signature" validation type).
#' 
#' Note about export rights: Please be aware that Data Export user rights will be 
#' applied to this API request. For example, if you have "No Access" data export rights 
#' in the project, then the API file export will fail and return an error. And if you 
#' have "De-Identified" or "Remove all tagged Identifier fields" data export rights, 
#' then the API file export will fail and return an error *only if* the File Upload 
#' field has been tagged as an Identifier field. To make sure that your API request 
#' does not return an error, you should have "Full Data Set" export rights in the project.
#' 
#' @section REDCap Version:
#' 5.8.2+ 
#' 
#' @section Known REDCap Limitations:
#' None
#' 
#' @author Benjamin Nutter
#' 
#' @references
#' Please refer to your institution's API documentation.
#' 
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/vubiostat/redcapAPI/wiki/REDCap-API-Parameters}
#' 
#' @export

exportFiles <- function(rcon, 
                        record, 
                        field, 
                        event      = NULL, 
                        dir, 
                        filePrefix = TRUE, 
                        ...){
  UseMethod("exportFiles")
}

#' @rdname exportFiles
#' @export

exportFiles.redcapApiConnection <- function(rcon, 
                                            record, 
                                            field, 
                                            event           = NULL, 
                                            dir, 
                                            filePrefix      = TRUE, 
                                            repeat_instance = NULL,
                                            ...,
                                            error_handling  = getOption("redcap_error_handling"),
                                            config          = list(), 
                                            api_param       = list()){

  if (is.numeric(record)) record <- as.character(record)
  
   ###########################################################################
  # Check parameters passed to function
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon, 
                          classes = "redcapApiConnection", 
                          add = coll)
  
  checkmate::assert_character(x = record, 
                              len = 1, 
                              any.missing = FALSE,
                              add = coll)
  
  checkmate::assert_character(x = field, 
                              len = 1, 
                              any.missing = FALSE, 
                              add = coll)
  
  checkmate::assert_character(x = event, 
                              len = 1, 
                              any.missing = FALSE, 
                              null.ok = TRUE,
                              add = coll)
  
  checkmate::assert_character(x = dir, 
                              len = 1, 
                              any.missing = FALSE, 
                              add = coll)
  
  checkmate::assert_logical(x = filePrefix,
                            len = 1,
                            add = coll)
  
  checkmate::assert_integerish(x = repeat_instance, 
                               len = 1, 
                               any.missing = FALSE,
                               null.ok = TRUE,
                               add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"), 
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
 
  checkmate::assert_directory_exists(x = dir, 
                                     add = coll)

  # Secure the MetaData
  MetaData <- rcon$metadata()
  
  checkmate::assert_subset(x = field, 
                           choices = MetaData$field_name, 
                           add = coll)
  
  # make sure 'field' exist in the project and are 'file' fields
  if (!isTRUE(MetaData$field_type[MetaData$field_name == field] == "file"))
  {
    coll$push(paste0("'", field, "' is not of field type 'file'"))
  }
  
  # Check that event exists in the project
  is_project_longitudinal <- as.logical(rcon$projectInformation()$is_longitudinal)
  
  if (is_project_longitudinal)
  {
    EventsList <- rcon$events()
    
    if (nrow(EventsList) == 0)
    {
      message("No events defined in this project. Ignoring the 'event' argument.")
      event <- NULL
    } else {
      checkmate::assert_subset(x = event, 
                               choices = EventsList$unique_event_name, 
                               add = coll)
    }
  } else {
    event <- NULL
  }
  
  checkmate::reportAssertions(coll)
  
   ###########################################################################
  # Build the Body List
  body <- list(content = 'file',
               action = 'export', 
               returnFormat = 'csv',
               record = record,
               field = field, 
               event = event)
  
  body <- body[lengths(body) > 0]
  
   ###########################################################################
  # Make the API Call
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)

  if (response$status_code != 200) redcap_error(response, error_handling)
  
  prefix <- 
    if (filePrefix) sprintf("%s%s%s", 
                            record,
                            if (is.null(event)) "" else "-",
                            if (is.null(event)) "" else event) else ""
  
  file_saved <- reconstituteFileFromExport(response = response, 
                                           dir = dir, 
                                           dir_create = FALSE, 
                                           file_prefix = prefix)
  
  message(sprintf("The file was saved to '%s'", 
                  file.path(file_saved$directory, 
                            file_saved$filename)))
}
