% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read.biom.r
\name{read.biom}
\alias{read.biom}
\title{Extracts counts, metadata, taxonomy, and phylogeny from a biom file.}
\usage{
read.biom(src, tree = "auto", prune = FALSE)
}
\arguments{
\item{src}{Input data as either a file path, URL, or JSON string.
\code{read.biom} can read BIOM files formatted according to both the
version 1.0 (JSON) and 2.1 (HDF5)
\href{http://biom-format.org/documentation/}{specifications} as well as
classical tabular format. URLs must begin with \kbd{http://},
\kbd{https://}, \kbd{ftp://}, or \kbd{ftps://}. JSON files must have
\code{\{} as their first non-whitespace character. Compressed (gzip or
bzip2) BIOM files are also supported. NOTE: to read HDF5 formatted BIOM
files, the BioConductor R package \code{rhdf5} must be installed.}

\item{tree}{The default value of \code{auto} will read the tree from the 
BIOM file specified in \code{src}, if present. The value \code{TRUE} 
will do the same, but will generate an error message if a tree is not 
present. Setting \code{tree=FALSE} will return a \code{BIOM} object 
without any tree data. You may also provide a file path, URL, or Newick
string to load that tree data into the final \code{BIOM} object.}

\item{prune}{Should samples and taxa with zero observations be discarded?}
}
\value{
A \code{BIOM} class object containing the parsed data. This object
    can be treated as a list with the following named elements:
    \describe{
        \item{counts}{A numeric \code{slam} sparse matrix of observation
        counts. Taxa (OTUs) as rows and samples as columns.}
        \item{metadata}{A data frame containing any embedded metadata.
        Row names are sample IDs.}
        \item{taxonomy}{Character matrix of taxonomic names, if given.
        Row names are taxa (OTU) IDs. Column rows are named Kingdom,
        Phylum, Class, Order, Family, Genus, Species, and Strain, or
        TaxLvl.1, TaxLvl.2, ... , TaxLvl.N when more than 8 levels of
        taxonomy are encoded in the biom file.}
        \item{phylogeny}{An object of class \code{phylo} defining the
        phylogenetic relationships between the taxa. Although the
        official specification for BIOM only includes phylogenetic trees
        in BIOM version 2.1, if a BIOM version 1.0 file includes a
        \code{phylogeny} entry with newick data, then it will be loaded
        here as well. The \pkg{ape} package has additional functions
        for working with \code{phylo} objects.}
        \item{sequences}{A named character vector, where the names are
        taxonomic identifiers and the values are the sequences they
        represent. These values are not part of the official BIOM
        specification, but will be read and written when defined.}
        \item{info}{A list of other attributes defined in the BIOM file,
        such as \code{id}, \code{type}, \code{format}, \code{format_url},
        \code{generated_by}, \code{date}, \code{matrix_type},
        \code{matrix_element_type}, \code{Comment}, and \code{shape}}
       }

    \code{metadata}, \code{taxonomy}, and \code{phylogeny} are optional
    components of the BIOM file specification and therefore will be empty
    in the returned object when they are not provided by the BIOM file.
}
\description{
Extracts counts, metadata, taxonomy, and phylogeny from a biom file.
}
\examples{
    library(rbiom)

    infile <- system.file("extdata", "hmp50.bz2", package = "rbiom")
    biom <- read.biom(infile)

    summary(biom)

    # Taxa Abundances
    as.matrix(biom$counts[1:4,1:4])

    top5 <- names(head(rev(sort(slam::row_sums(biom$counts))), 5))
    biom$taxonomy[top5,c('Family', 'Genus')]
    as.matrix(biom$counts[top5, 1:6])

    # Metadata
    table(biom$metadata$Sex, biom$metadata$`Body Site`)
    sprintf("Mean age: \%.1f", mean(biom$metadata$Age))

    # Phylogenetic tree
    tree <- biom$phylogeny
    top5.tree <- rbiom::subtree(tree, top5)
    ape::plot.phylo(top5.tree)

}
