% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/locations.r
\name{locations}
\alias{locations}
\title{Select Representative Sampling Locations for Stratification Units}
\usage{
locations(
  ls.rast,
  su.rast,
  method = "buffer",
  constrained = TRUE,
  buf.quant = 0.9,
  buf.n = 1,
  abs.n = 1,
  tol = 2,
  parallel = FALSE,
  to.disk = FALSE,
  outdir = ".",
  verbose = FALSE,
  ...
)
}
\arguments{
\item{ls.rast}{SpatRaster, as in \code{\link[terra]{rast}}. Multi-layer
SpatRaster representing landscape similarities to stratification units.}

\item{su.rast}{SpatRaster. Single-layer SpatRaster representing the
stratification units occurring across geographic space. Integer values are
expected as cell values (i.e., numeric codes) of stratification units.}

\item{method}{Character. String denoting the sampling method. Current options
are "buffer" for the maximum similarity within buffer zones method, and
"absolute" for the absolute maximum similarity method. Default: "buffer"}

\item{constrained}{Boolean. Should the sampling process be constrained to the
boundaries of each stratification unit? See \strong{Details}. Default: TRUE}

\item{buf.quant}{Numeric. Number expressed in quantile notation (0-1)
indicating the similarity threshold for the creation of buffer zones. Only
zones enclosing raster cells with \emph{landscape similarity >= buf.quant}
will be created and thus, considered for sampling. See \strong{Details}.
Default: 0.9}

\item{buf.n}{Integer. Positive integer indicating the \emph{n} largest buffer
zones for which sampling locations will be selected (\emph{n} buffer zones
per stratification unit, one sampling location per buffer zone). Default: 1}

\item{abs.n}{Integer. When \emph{method = "absolute"}, Positive integer
indicating the number of sampling locations to select for each
stratification unit. See \strong{Details}. Default: 1}

\item{tol}{Numeric. This number will be subtracted from the sampled maximum
value of a landscape similarity layer to ensure two things: (1) that the
'true' maximum similarity value is found regardless of the size of the
layer, and (2) that the requested number of sampling locations will be
found (see \strong{Details}). The default assumes that landscape similarity
values are on a scale of 1 to 100. If these values are on a different scale
(e.g., decimal), then, \emph{tol} needs to be adjusted accordingly.
Default: 2}

\item{parallel}{Boolean. Perform parallel processing? A parallel backend must
be registered beforehand with \code{\link[doParallel]{registerDoParallel}}.
Keep in mind that the amount of RAM to allocate when performing parallel
processing can result prohibitive for large data sets. Default: FALSE}

\item{to.disk}{Boolean. Should output SpatVector(s) (as in
\code{\link[terra]{vect}}) be written to disk? Default: FALSE}

\item{outdir}{Character. If \emph{to.disk = TRUE}, string specifying the path
for the output SpatVector(s). Default: "."}

\item{verbose}{Boolean. Show warning messages in the console? Default: FALSE}

\item{...}{Additional arguments, as for \code{\link[terra]{writeVector}}.}
}
\value{
If \emph{method = "buffer"} and \emph{constrained = TRUE}, a list with the
following components:

\strong{locations}: SpatVector of point geometry. Each point in this vector
represents the sampling location placed at the maximum landscape similarity
value within a stratification unit's buffer zone. Tabular attributes in this
SpatVector are (1) \emph{SU} = stratification unit's numeric code, (2)
\emph{land_sim} = landscape similarity value, (3) \emph{x} = X coordinate,
and (4) \emph{y} = Y coordinate.

\strong{buffer}: SpatVector of polygon geometry. Each polygon in this vector
represents the buffer zone of an stratification unit.

If \emph{method = "buffer"} and \emph{constrained = FALSE}:

\strong{locations}: Same as \strong{locations} from \emph{method = "buffer"}
and \emph{constrained = TRUE}.

If \emph{method = "absolute"}:

\strong{locations}: SpatVector of point geometry. Each point in this vector
represents the sampling location placed at the maximum landscape similarity
value for an stratification unit. Tabular attributes in this SpatVector are
(1) \emph{SU} = stratification unit's numeric code, (2) \emph{land_sim} =
landscape similarity value, (3) \emph{x} = X coordinate, and (4) \emph{y} = Y
coordinate.
}
\description{
Selection of the \emph{representative sampling locations} based on landscape
similarity values for each stratification unit occurring across geographic
space. The \emph{XY} spatial location where the landscape similarity is
maximized for a given stratification unit is considered to contain the
optimal representation of the influence that landscape factors/factor scales
have on a response phenomenon. Currently, two methods are supported: (1)
\emph{maximum similarity within buffer zones} ("buffer"), and (2)
\emph{absolute maximum similarity} ("absolute"). For the buffer method, the
\emph{n} largest zones enclosing landscape similarity values above a certain
threshold are first identified. Then, for each zone, one sample is placed at
the \emph{XY} location where the landscape similarity value is maximized. For
the absolute method, a sample is placed at the \emph{XY} locations with the
\emph{n} maximum landscape similarity values. In both methods, it is possible
to constrain the sampling process to the boundaries of the stratification
unit. Constraining the process ensures that the sampling locations determined
for a given unit are placed within the boundaries of that unit. See
\strong{Details} for some guidance in the use of this function for
classification units.
}
\details{
Except when \emph{buf.n = 1} or \emph{abs.n = 1}, the number of returned
sampling locations per stratification unit may be smaller than requested,
especially when \emph{constrained = TRUE}. For the constrained buffer method,
reducing the landscape similarity threshold value \emph{buf.quant} will not
always result in more buffer zones; i.e., more sampling locations. The reason
for this is that reducing the threshold value for the creation of buffer
zones may actually promote the spatial contiguity of zones. For instance, two
buffer zones created at \emph{buf.quant = 0.9}, may be merged into a single
buffer zone when \emph{buf.quant = 0.80}. This will occur if the raster cells
between the two buffer zones satisfy: \emph{landscape similarity >=
quantile(landscape similarity, 0.8)}. For the absolute method, increasing the
value of the \emph{tol} argument will ensure a safer search for \emph{n}
sampling locations and thus, greater chances of getting the total number of
requested sampling locations per stratification unit.

Note that this sampling scheme can be applied for classification units. In
order to do this, one should replace the multi-layer SpatRaster of landscape
similarities with a multi-layer SpatRaster of spatial signatures. One should
also replace the raster layer of stratification units with that of
classification units.
}
\examples{
require(terra)
p <- system.file("exdat", package = "rassta")
# Multi-layer SpatRaster of landscape similarities
fls <- list.files(path = p, pattern = "su_", full.names = TRUE)
ls <- terra::rast(fls)
# Single-layer SpatRaster of stratification units
fsu <- list.files(path = p, pattern = "strata.tif", full.names = TRUE)
su <- terra::rast(fsu)
# Get 1 representative sampling location per stratification unit
rl <- locations(ls.rast = ls, su.rast = su)
# Plot representative locations (including buffer areas)
if(interactive()){
  plot(su, type = "classes", fun = function() c(points(rl$locations),
                                                polys(rl$buffers))
      )}

}
\seealso{
\code{\link{similarity}}, \code{\link{strata}}

Other Functions for Stratified Sampling: 
\code{\link{observation}()}
}
\concept{Functions for Stratified Sampling}
